<?php

namespace App\Http\Controllers;

use App\Models\AreaMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class AreaMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {

        $areas = AreaMaster::with(['ward' => function($query) {
            $query->select('Ward_Id', 'Ward_Code', 'Ward_Name', 'Zone_Id', 'is_active');
        }])->get();

        return response()->json([
            'success' => true,
            'data' => $areas
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Area_Code' => 'nullable|string|max:5|unique:area_master,Area_Code',
            'Area_Name' => 'required|string|max:255',
            'Ward_ID' => 'nullable|integer|exists:ward_master,Ward_Id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $area = AreaMaster::create($request->all());


        $area->load(['ward' => function($query) {
            $query->select('Ward_Id', 'Ward_Code', 'Ward_Name', 'Zone_Id', 'is_active');
        }]);

        return response()->json([
            'success' => true,
            'data' => $area,
            'message' => 'Area created successfully'
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {

        $area = AreaMaster::with(['ward' => function($query) {
            $query->select('Ward_Id', 'Ward_Code', 'Ward_Name', 'Zone_Id', 'is_active');
        }])->find($id);

        if (!$area) {
            return response()->json([
                'success' => false,
                'message' => 'Area not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $area
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $area = AreaMaster::find($id);

        if (!$area) {
            return response()->json([
                'success' => false,
                'message' => 'Area not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'Area_Code' => 'string|max:5|unique:area_master,Area_Code,' . $id . ',Area_Id',
            'Area_Name' => 'string|max:255',
            'Ward_ID' => 'nullable|integer|exists:ward_master,Ward_Id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $area->update($request->all());


        $area->load(['ward' => function($query) {
            $query->select('Ward_Id', 'Ward_Code', 'Ward_Name', 'Zone_Id', 'is_active');
        }]);

        return response()->json([
            'success' => true,
            'data' => $area,
            'message' => 'Area updated successfully'
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $area = AreaMaster::find($id);

        if (!$area) {
            return response()->json([
                'success' => false,
                'message' => 'Area not found'
            ], 404);
        }

        $area->delete();

        return response()->json([
            'success' => true,
            'message' => 'Area deleted successfully'
        ]);
    }

    /**
     * Search areas by name or code
     */
    public function search(Request $request)
    {
        $query = AreaMaster::with(['ward' => function($query) {
            $query->select('Ward_Id', 'Ward_Code', 'Ward_Name', 'Zone_Id', 'is_active');
        }]);

        if ($request->has('search')) {
            $search = $request->search;
            $query->where('Area_Name', 'like', '%' . $search . '%')
                  ->orWhere('Area_Code', 'like', '%' . $search . '%');
        }

        $areas = $query->get();

        return response()->json([
            'success' => true,
            'data' => $areas
        ]);
    }
}
