<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\GmcUser;
use App\Models\GmcUserAuditLog;
use App\Models\GmcUserSession;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class AuthController extends Controller
{
    /**
     * User Login
     */
    public function login(Request $request)
    {
        try {
            // ================= VALIDATION =================
            $validator = Validator::make($request->all(), [
                'username' => 'required|string',
                'password' => 'required|string',
                'remember_me' => 'boolean'
            ]);



            if ($validator->fails()) {

                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // ================= USER FETCH =================
            $user = GmcUser::where('username', $request->username)
                ->orWhere('email', $request->username)
                ->orWhere('mobile', $request->username)
                ->first();

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid credentials'
                ], 401);
            }

            if (!$user->is_active) {
                return response()->json([
                    'success' => false,
                    'message' => 'Your account is inactive. Please contact administrator.'
                ], 401);
            }

            if ($user->isAccountLocked()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Your account is locked.'
                ], 401);
            }

            // ================= PASSWORD CHECK =================
            if (!Hash::check($request->password, $user->password_hash)) {

                $user->incrementFailedAttempts();

                GmcUserAuditLog::logAction(
                    $user->user_id,
                    'LOGIN_FAILED',
                    'Failed login attempt',
                    $request->ip(),
                    $request->header('User-Agent')
                );

                return response()->json([
                    'success' => false,
                    'message' => 'Invalid credentials'
                ], 401);
            }

            if ($user->isPasswordExpired()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Password expired',
                    'requires_password_reset' => true
                ], 401);
            }

            // ================= DEVICE ID VERIFICATION =================
            // Hidden backend device id
           // ================= DEVICE ID VERIFICATION =================

$deviceId = sha1($request->header('User-Agent')); // IP hata diya

$deviceLog = DB::table('gmc_device_approval_logs')
    ->where('user_id', $user->user_id)
    ->whereRaw("JSON_EXTRACT(device_data, '$.device_id') = ?", [$deviceId])
    ->orderBy('performed_at', 'desc')
    ->first();

// First time device
if (!$deviceLog) {

    DB::table('gmc_device_approval_logs')->insert([
        'user_id' => $user->user_id,
        'action_type' => 'APPROVAL_REQUESTED',
        'action_details' => 'Device approval requested during login',
        'device_data' => json_encode([
            'device_id' => $deviceId
        ]),
        'performed_at' => now()
    ]);

    return response()->json([
        'success' => false,
        'message' => 'Device approval pending. Please wait for admin approval.'
    ], 403);
}

// Still pending
if ($deviceLog->action_type === 'APPROVAL_REQUESTED') {
    return response()->json([
        'success' => false,
        'message' => 'Device approval is pending.'
    ], 403);
}

// Not approved
if ($deviceLog->action_type !== 'REGISTRATION_SUBMITTED') {
    return response()->json([
        'success' => false,
        'message' => 'Device not authorized.'
    ], 403);
}


            // Pending approval
            if ($deviceLog->action_type === 'APPROVAL_REQUESTED') {
                return response()->json([
                    'success' => false,
                    'message' => 'Device approval is pending.'
                ], 403);
            }

            // Not approved
            if ($deviceLog->action_type !== 'REGISTRATION_SUBMITTED') {
                return response()->json([
                    'success' => false,
                    'message' => 'Device not authorized.'
                ], 403);
            }

            // ================= LOGIN SUCCESS =================
            $user->resetFailedAttempts();
            $user->update([
                'last_login' => now(),
                'failed_login_attempts' => 0
            ]);

            $sessionToken = Str::random(60);
            $sessionTimeout = $request->remember_me
                ? 60 * 24 * 30
                : $user->session_timeout;

            $session = GmcUserSession::create([
                'user_id' => $user->user_id,
                'session_token' => $sessionToken,
                'ip_address' => $request->ip(),
                'user_agent' => $request->header('User-Agent'),
                'device_info' => $deviceId,
                'login_time' => now(),
                'last_activity' => now(),
                'expires_at' => now()->addMinutes($sessionTimeout),
                'is_active' => true
            ]);

            GmcUserAuditLog::logAction(
                $user->user_id,
                'LOGIN_SUCCESS',
                'User logged in successfully',
                $request->ip(),
                $request->header('User-Agent')
            );

            return response()->json([
                'success' => true,
                'message' => 'Login successful',
                'data' => [
                    'user' => [
                        'user_id' => $user->user_id,
                        'username' => $user->username,
                        'email' => $user->email,
                        'mobile' => $user->mobile,
                        'role_id' => $user->Role_Id,
                        'mfa_enabled' => $user->mfa_enabled
                    ],
                    'session' => [
                        'token' => $sessionToken,
                        'expires_at' => $session->expires_at->format('Y-m-d H:i:s'),
                        'timeout_minutes' => $sessionTimeout
                    ]
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Login Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Login failed. Please try again later.'
            ], 500);
        }
    }
}
