<?php

namespace App\Http\Controllers;

use App\Models\AwardMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class AwardMasterController extends Controller
{
    // 🔹 GET ALL
    public function index()
    {
        try {
            $data = AwardMaster::where('is_active', true)->get();

            return response()->json([
                'success' => true,
                'message' => 'Awards fetched successfully',
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'award_code'          => 'nullable|string|max:30|unique:award_master,award_code',
                'award_name'          => 'required|string|max:200',
                'award_year'          => 'nullable|integer|min:1900|max:' . date('Y'),
                'awarding_authority'  => 'nullable|string|max:100',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $award = AwardMaster::create($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Award created successfully',
                'data' => $award
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create award',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $award = AwardMaster::find($id);

            if (!$award) {
                return response()->json([
                    'success' => false,
                    'message' => 'Award not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $award
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching award',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE (sometimes)
    public function update(Request $request, $id)
    {
        try {
            $award = AwardMaster::find($id);

            if (!$award) {
                return response()->json([
                    'success' => false,
                    'message' => 'Award not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'award_code'         => 'sometimes|required|string|max:30|unique:award_master,award_code,' . $id . ',award_id',
                'award_name'         => 'sometimes|required|string|max:200',
                'award_year'         => 'sometimes|nullable|integer|min:1900|max:' . date('Y'),
                'awarding_authority' => 'sometimes|nullable|string|max:100',
                'is_active'          => 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $award->update($request->only([
                'award_code',
                'award_name',
                'award_year',
                'awarding_authority',
                'is_active'
            ]));

            return response()->json([
                'success' => true,
                'message' => 'Award updated successfully',
                'data' => $award
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update award',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE (Soft)
    public function destroy($id)
    {
        try {
            $award = AwardMaster::find($id);

            if (!$award) {
                return response()->json([
                    'success' => false,
                    'message' => 'Award not found'
                ], 404);
            }

            $award->update(['is_active' => false]);

            return response()->json([
                'success' => true,
                'message' => 'Award deactivated successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete award',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
