<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class CategoryController extends Controller
{
    /**
     * Display a listing of categories (without pagination)
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        try {
            $categories = Category::with(['subCategories', 'subSubCategories'])
                ->orderBy('category_name')
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Categories fetched successfully',
                'data' => $categories
            ], 200);

        } catch (\Exception $e) {
            Log::error('Category Fetch Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch categories. Please try again later.'
            ], 500);
        }
    }

    /**
     * Display paginated categories (separate pagination API)
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function paginate(Request $request)
    {
        try {
            // Validate pagination parameters
            $validator = Validator::make($request->all(), [
                'per_page' => 'nullable|integer|min:1|max:100',
                'page' => 'nullable|integer|min:1',
                'search' => 'nullable|string|max:100',
                'is_active' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid pagination parameters',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Get pagination parameters with defaults
            $perPage = $request->get('per_page', 10);
            $page = $request->get('page', 1);
            $search = $request->get('search', '');
            $isActive = $request->get('is_active', null);

            // Build query
            $query = Category::with(['subCategories', 'subSubCategories'])
                ->when($search, function ($query, $search) {
                    return $query->where(function ($q) use ($search) {
                        $q->where('category_name', 'like', "%{$search}%")
                          ->orWhere('category_code', 'like', "%{$search}%")
                          ->orWhere('description', 'like', "%{$search}%");
                    });
                })
                ->when(!is_null($isActive), function ($query) use ($isActive) {
                    return $query->where('is_active', $isActive);
                })
                ->orderBy('category_name');

            // Get paginated results
            $categories = $query->paginate($perPage, ['*'], 'page', $page);

            // Format pagination response
            $response = [
                'success' => true,
                'message' => 'GMC Gadhinagar - Categories paginated successfully',
                'data' => $categories->items(),
                'pagination' => [
                    'current_page' => $categories->currentPage(),
                    'first_page_url' => $categories->url(1),
                    'from' => $categories->firstItem(),
                    'last_page' => $categories->lastPage(),
                    'last_page_url' => $categories->url($categories->lastPage()),
                    'links' => $categories->linkCollection()->toArray(),
                    'next_page_url' => $categories->nextPageUrl(),
                    'path' => $categories->path(),
                    'per_page' => $categories->perPage(),
                    'prev_page_url' => $categories->previousPageUrl(),
                    'to' => $categories->lastItem(),
                    'total' => $categories->total(),
                ],
                'meta' => [
                    'total_records' => $categories->total(),
                    'current_page_records' => count($categories->items()),
                    'total_pages' => $categories->lastPage(),
                    'has_more_pages' => $categories->hasMorePages()
                ]
            ];

            return response()->json($response, 200);

        } catch (\Exception $e) {
            Log::error('Category Paginate Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch paginated categories. Please try again later.'
            ], 500);
        }
    }

    /**
     * Store a newly created category.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    /**
 * Store a newly created category.
 *
 * @param  \Illuminate\Http\Request  $request
 * @return \Illuminate\Http\JsonResponse
 */
public function store(Request $request)
{
    try {
        $validator = Validator::make($request->all(), [
            'category_name' => 'required|string|max:100|unique:categories,category_name',
            'category_code' => 'nullable|string|max:50|unique:categories,category_code|regex:/^[A-Z0-9]+$/',
            'description' => 'nullable|string|max:500',
            'is_active' => 'boolean'
        ], [
            'category_name.required' => 'Category name is required',
            'category_name.unique' => 'This category name already exists',
            'category_code.unique' => 'This category code is already in use',
            'category_code.regex' => 'Category code should contain only uppercase letters and numbers'
        ]);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        $categoryData = [
            'category_name' => $request->category_name,
            'description' => $request->description,
            'is_active' => $request->boolean('is_active', true)
        ];

        // Only set category_code if provided and not empty
        if ($request->filled('category_code')) {
            $categoryData['category_code'] = strtoupper($request->category_code);
        }

        $category = Category::create($categoryData);

        return response()->json([
            'success' => true,
            'message' => 'GMC Gadhinagar - Category created successfully',
            'data' => $category
        ], 201);

    } catch (ValidationException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Validation failed',
            'errors' => $e->errors()
        ], 422);

    } catch (\Exception $e) {
        Log::error('Category Create Error: ' . $e->getMessage());

        return response()->json([
            'success' => false,
            'message' => 'Failed to create category. Please check the data and try again.',
            'error' => $e->getMessage() // Add this for debugging
        ], 500);
    }
}

    /**
     * Display the specified category.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $category = Category::with(['subCategories', 'subSubCategories'])->find($id);

            if (!$category) {
                return response()->json([
                    'success' => false,
                    'message' => 'Category not found. Please check the category ID.'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Category details retrieved successfully',
                'data' => $category
            ], 200);

        } catch (\Exception $e) {
            Log::error('Category Show Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to retrieve category details.'
            ], 500);
        }
    }

    /**
     * Update the specified category.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    /**
 * Update the specified category.
 *
 * @param  \Illuminate\Http\Request  $request
 * @param  int  $id
 * @return \Illuminate\Http\JsonResponse
 */
public function update(Request $request, $id)
{
    try {
        $category = Category::find($id);

        if (!$category) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found. Please check the category ID.'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'category_name' => 'required|string|max:100|unique:categories,category_name,' . $id . ',category_id',
            'category_code' => 'nullable|string|max:50|unique:categories,category_code,' . $id . ',category_id|regex:/^[A-Z0-9]+$/',
            'description' => 'nullable|string|max:500',
            'is_active' => 'boolean'
        ], [
            'category_name.required' => 'Category name is required',
            'category_name.unique' => 'This category name already exists',
            'category_code.unique' => 'This category code is already in use',
            'category_code.regex' => 'Category code should contain only uppercase letters and numbers'
        ]);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        $updateData = [
            'category_name' => $request->category_name,
            'description' => $request->description,
            'is_active' => $request->boolean('is_active', $category->is_active)
        ];

        // Only update category_code if provided
        if ($request->has('category_code')) {
            if ($request->filled('category_code')) {
                $updateData['category_code'] = strtoupper($request->category_code);
            } else {
                // If empty string is sent, set to null
                $updateData['category_code'] = null;
            }
        }

        $category->update($updateData);

        return response()->json([
            'success' => true,
            'message' => 'GMC Gadhinagar - Category updated successfully',
            'data' => $category->refresh()
        ], 200);

    } catch (ValidationException $e) {
        return response()->json([
            'success' => false,
            'message' => 'Validation failed',
            'errors' => $e->errors()
        ], 422);

    } catch (\Exception $e) {
        Log::error('Category Update Error: ' . $e->getMessage());

        return response()->json([
            'success' => false,
            'message' => 'Failed to update category. Please check the data and try again.',
            'error' => $e->getMessage() // Add this for debugging
        ], 500);
    }
}

    /**
     * Remove the specified category.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $category = Category::withCount(['subCategories'])->find($id);

            if (!$category) {
                return response()->json([
                    'success' => false,
                    'message' => 'Category not found. Please check the category ID.'
                ], 404);
            }

            if ($category->sub_categories_count > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete category because it has associated sub-categories. Please delete sub-categories first.'
                ], 400);
            }

            $category->delete();

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Category deleted successfully',
                'data' => null
            ], 200);

        } catch (\Illuminate\Database\QueryException $e) {
            Log::error('Category Delete Query Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Cannot delete category due to database constraints. Please remove all related records first.'
            ], 409);

        } catch (\Exception $e) {
            Log::error('Category Delete Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete category. Please try again later.'
            ], 500);
        }
    }
}
