<?php

namespace App\Http\Controllers;

use App\Models\CccMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CccMasterController extends Controller
{
    public function index(Request $request)
    {
        try {
            $query = CccMaster::with(['ward', 'ward.zone', 'ward.zone.organisation'])
                        ->active();

            // Filter by ward if provided
            if ($request->has('ward_id')) {
                $query->byWard($request->ward_id);
            }

            // Filter by zone if provided
            if ($request->has('zone_id')) {
                $query->whereHas('ward', function ($q) use ($request) {
                    $q->where('Zone_Id', $request->zone_id);
                });
            }

            // Search functionality
            if ($request->has('search')) {
                $query->search($request->search);
            }

            // Pagination
            $perPage = $request->get('per_page', 15);
            $cccs = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $cccs,
                'message' => 'CCCs retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve CCCs',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'CCC_Code' => 'nullable|string|max:5',
                'CCC_Name' => 'required|string|max:25',
                'Ward_Id' => 'required|exists:ward_master,Ward_Id',
            ], CccMaster::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $ccc = CccMaster::create([
                'CCC_Code' => $request->CCC_Code,
                'CCC_Name' => $request->CCC_Name,
                'Ward_Id' => $request->Ward_Id,
                'is_active' => true
            ]);

            return response()->json([
                'success' => true,
                'message' => 'CCC created successfully',
                'data' => $ccc->load(['ward', 'ward.zone', 'ward.zone.organisation'])
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create CCC',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            $ccc = CccMaster::with(['ward', 'ward.zone', 'ward.zone.organisation'])
                    ->find($id);

            if (!$ccc) {
                return response()->json([
                    'success' => false,
                    'message' => 'CCC not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $ccc,
                'message' => 'CCC retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve CCC',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $ccc = CccMaster::find($id);

            if (!$ccc) {
                return response()->json([
                    'success' => false,
                    'message' => 'CCC not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'CCC_Code' => 'sometimes|required|string|max:5',
                'CCC_Name' => 'sometimes|required|string|max:25',
                'Ward_Id' => 'sometimes|required|exists:ward_master,Ward_Id',
                'is_active' => 'sometimes|boolean'
            ], CccMaster::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $ccc->update($request->all());

            return response()->json([
                'success' => true,
                'message' => 'CCC updated successfully',
                'data' => $ccc->load(['ward', 'ward.zone', 'ward.zone.organisation'])
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update CCC',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $ccc = CccMaster::find($id);

            if (!$ccc) {
                return response()->json([
                    'success' => false,
                    'message' => 'CCC not found'
                ], 404);
            }

            if (!$ccc->canBeDeleted()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete CCC as it has dependent records'
                ], 400);
            }

            $ccc->deactivate();

            return response()->json([
                'success' => true,
                'message' => 'CCC deactivated successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to deactivate CCC',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function restore($id)
    {
        try {
            $ccc = CccMaster::withTrashed()->find($id);

            if (!$ccc) {
                return response()->json([
                    'success' => false,
                    'message' => 'CCC not found'
                ], 404);
            }

            $ccc->activate();

            return response()->json([
                'success' => true,
                'message' => 'CCC activated successfully',
                'data' => $ccc->load(['ward', 'ward.zone', 'ward.zone.organisation'])
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to activate CCC',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // Get CCCs by ward
    public function getByWard($wardId)
    {
        try {
            $cccs = CccMaster::with(['ward', 'ward.zone'])
                    ->byWard($wardId)
                    ->active()
                    ->get();

            return response()->json([
                'success' => true,
                'data' => $cccs,
                'message' => 'CCCs retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve CCCs',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // Get CCCs by zone
    public function getByZone($zoneId)
    {
        try {
            $cccs = CccMaster::with(['ward', 'ward.zone'])
                    ->whereHas('ward', function ($q) use ($zoneId) {
                        $q->where('Zone_Id', $zoneId);
                    })
                    ->active()
                    ->get();

            return response()->json([
                'success' => true,
                'data' => $cccs,
                'message' => 'CCCs retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve CCCs',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
