<?php

namespace App\Http\Controllers;

use App\Models\CoaDepMapping;
use App\Models\MainDepartment;
use App\Models\LedgGrp;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class CoaDepMappingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        try {
            $mappings = CoaDepMapping::with(['mainDepartment'])
                ->get()
                ->map(function ($mapping) {
                    $mapping->all_groups = $mapping->ledgerGroups();
                    return $mapping;
                });

            return response()->json([
                'success' => true,
                'data' => $mappings,
                'message' => 'COA department mappings retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve mappings.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     * यदि department ID के लिए पहले से record है तो उसमें groups add होगा
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Dep_Id' => 'nullable|exists:main_departments,id',
            'Gr_IDs' => 'required|array|min:1',
            'Gr_IDs.*' => 'exists:ledg_grps,Gr_Id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
                'message' => 'Validation failed.'
            ], 422);
        }

        try {
            DB::beginTransaction();

            $groupIds = array_unique($request->Gr_IDs);
            $depId = $request->Dep_Id;

            // Check if mapping already exists for this department
            $existingMapping = null;
            if ($depId) {
                $existingMapping = CoaDepMapping::where('Dep_Id', $depId)->first();
            }

            if ($existingMapping) {
                // Update existing mapping - merge groups
                $currentGroupIds = $existingMapping->Gr_IDs ?? [];
                $mergedGroupIds = array_unique(array_merge($currentGroupIds, $groupIds));

                $existingMapping->update(['Gr_IDs' => $mergedGroupIds]);
                $mapping = $existingMapping;

                $message = 'Groups added to existing department mapping successfully.';
            } else {
                // Create new mapping
                $mapping = CoaDepMapping::create([
                    'Dep_Id' => $depId,
                    'Gr_IDs' => $groupIds,
                ]);

                $message = 'COA department mapping created successfully.';
            }

            $mapping->load(['mainDepartment']);
            $mapping->all_groups = $mapping->ledgerGroups();

            DB::commit();

            return response()->json([
                'success' => true,
                'data' => $mapping,
                'message' => $message,
                'action' => $existingMapping ? 'updated' : 'created'
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to process mapping.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try {
            $mapping = CoaDepMapping::with(['mainDepartment'])->find($id);

            if (!$mapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Mapping not found.'
                ], 404);
            }

            $mapping->all_groups = $mapping->ledgerGroups();

            return response()->json([
                'success' => true,
                'data' => $mapping,
                'message' => 'Mapping retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve mapping.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $mapping = CoaDepMapping::find($id);

            if (!$mapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Mapping not found.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Dep_Id' => 'sometimes|exists:main_departments,id',
                'Gr_IDs' => 'sometimes|array|min:1',
                'Gr_IDs.*' => 'exists:ledg_grps,Gr_Id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed.'
                ], 422);
            }

            // Check if updating Dep_Id would cause duplicate
            if ($request->has('Dep_Id') && $request->Dep_Id != $mapping->Dep_Id) {
                $existingWithNewDep = CoaDepMapping::where('Dep_Id', $request->Dep_Id)
                    ->where('id', '!=', $id)
                    ->first();

                if ($existingWithNewDep) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Another mapping already exists for this department ID.',
                        'existing_mapping_id' => $existingWithNewDep->id
                    ], 409);
                }
            }

            $updateData = [];

            if ($request->has('Dep_Id')) {
                $updateData['Dep_Id'] = $request->Dep_Id;
            }

            if ($request->has('Gr_IDs')) {
                $updateData['Gr_IDs'] = array_unique($request->Gr_IDs);
            }

            if (!empty($updateData)) {
                $mapping->update($updateData);
            }

            $mapping->load(['mainDepartment']);
            $mapping->all_groups = $mapping->ledgerGroups();

            return response()->json([
                'success' => true,
                'data' => $mapping,
                'message' => 'Mapping updated successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update mapping.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $mapping = CoaDepMapping::find($id);

            if (!$mapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Mapping not found.'
                ], 404);
            }

            $mapping->delete();

            return response()->json([
                'success' => true,
                'message' => 'Mapping deleted successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete mapping.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get mappings by department ID
     */
    public function byDepartment($depId)
    {
        try {
            $mappings = CoaDepMapping::with(['mainDepartment'])
                ->where('Dep_Id', $depId)
                ->get()
                ->map(function ($mapping) {
                    $mapping->all_groups = $mapping->ledgerGroups();
                    return $mapping;
                });

            return response()->json([
                'success' => true,
                'data' => $mappings,
                'message' => 'Mappings retrieved by department successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve mappings.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get mappings by group ID
     */
    public function byGroup($groupId)
    {
        try {
            $mappings = CoaDepMapping::with(['mainDepartment'])
                ->whereJsonContains('Gr_IDs', $groupId)
                ->get()
                ->map(function ($mapping) {
                    $mapping->all_groups = $mapping->ledgerGroups();
                    return $mapping;
                });

            return response()->json([
                'success' => true,
                'data' => $mappings,
                'message' => 'Mappings retrieved by group successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve mappings.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Add group IDs to an existing mapping
     */
    public function addGroups(Request $request, $id)
    {
        try {
            $mapping = CoaDepMapping::find($id);

            if (!$mapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Mapping not found.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Gr_IDs' => 'required|array|min:1',
                'Gr_IDs.*' => 'exists:ledg_grps,Gr_Id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed.'
                ], 422);
            }

            $currentGroupIds = $mapping->Gr_IDs ?? [];
            $newGroupIds = array_unique($request->Gr_IDs);
            $mergedGroupIds = array_unique(array_merge($currentGroupIds, $newGroupIds));

            $mapping->Gr_IDs = $mergedGroupIds;
            $mapping->save();

            $mapping->load(['mainDepartment']);
            $mapping->all_groups = $mapping->ledgerGroups();

            return response()->json([
                'success' => true,
                'data' => $mapping,
                'added_count' => count(array_diff($newGroupIds, $currentGroupIds)),
                'total_groups' => count($mergedGroupIds),
                'message' => 'Groups added successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to add groups.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove group IDs from an existing mapping
     */
    public function removeGroups(Request $request, $id)
    {
        try {
            $mapping = CoaDepMapping::find($id);

            if (!$mapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Mapping not found.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Gr_IDs' => 'required|array|min:1',
                'Gr_IDs.*' => 'exists:ledg_grps,Gr_Id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed.'
                ], 422);
            }

            $currentGroupIds = $mapping->Gr_IDs ?? [];
            $removeGroupIds = $request->Gr_IDs;
            $updatedGroupIds = array_diff($currentGroupIds, $removeGroupIds);

            if (empty($updatedGroupIds)) {
                $mapping->delete();
                return response()->json([
                    'success' => true,
                    'message' => 'All groups removed. Mapping deleted.',
                ]);
            }

            $mapping->Gr_IDs = array_values($updatedGroupIds);
            $mapping->save();

            $mapping->load(['mainDepartment']);
            $mapping->all_groups = $mapping->ledgerGroups();

            return response()->json([
                'success' => true,
                'data' => $mapping,
                'removed_count' => count(array_intersect($currentGroupIds, $removeGroupIds)),
                'remaining_groups' => count($updatedGroupIds),
                'message' => 'Groups removed successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to remove groups.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * NEW API: Get all groups for a specific department with details
     */
    public function getDepartmentLedgerGroups($depId)
    {
        try {
            // First check if department exists
            $department = MainDepartment::find($depId);

            if (!$department) {
                return response()->json([
                    'success' => false,
                    'message' => 'Department not found.'
                ], 404);
            }

            // Get all mappings for this department
            $mappings = CoaDepMapping::where('Dep_Id', $depId)->get();

            if ($mappings->isEmpty()) {
                return response()->json([
                    'success' => true,
                    'data' => [
                        'department' => $department,
                        'groups' => [],
                        'mappings' => []
                    ],
                    'message' => 'No groups found for this department.',
                    'total_groups' => 0
                ]);
            }

            // Collect all unique group IDs
            $allGroupIds = [];
            foreach ($mappings as $mapping) {
                if (!empty($mapping->Gr_IDs) && is_array($mapping->Gr_IDs)) {
                    $allGroupIds = array_merge($allGroupIds, $mapping->Gr_IDs);
                }
            }

            $allGroupIds = array_unique($allGroupIds);

            // Get all groups with their details
            $groups = LedgGrp::whereIn('Gr_Id', $allGroupIds)->get();

            // Format response with mapping details
            $formattedMappings = $mappings->map(function ($mapping) {
                return [
                    'mapping_id' => $mapping->id,
                    'group_ids' => $mapping->Gr_IDs ?? [],
                    'created_at' => $mapping->created_at,
                    'updated_at' => $mapping->updated_at,
                    'groups_details' => $mapping->ledgerGroups()
                ];
            });

            return response()->json([
                'success' => true,
                'data' => [
                    'department' => $department,
                    'groups' => $groups,
                    'mappings' => $formattedMappings,
                    'all_group_ids' => $allGroupIds
                ],
                'total_unique_groups' => count($allGroupIds),
                'total_mappings' => $mappings->count(),
                'message' => 'Department ledger groups retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve department groups.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all departments for a specific group
     */
    public function departmentsForGroup($groupId)
    {
        try {
            $mappings = CoaDepMapping::with('mainDepartment')
                ->whereJsonContains('Gr_IDs', $groupId)
                ->get();

            $departments = $mappings->pluck('mainDepartment')->filter();

            return response()->json([
                'success' => true,
                'data' => $departments,
                'total_mappings' => $mappings->count(),
                'message' => 'Departments for group retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve departments.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
