<?php

namespace App\Http\Controllers;

use App\Models\Department;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class DepartmentController extends Controller
{
    /**
     * Display a listing of departments with optional filters
     */
    public function index(Request $request)
    {
        try {
            $query = Department::query();

            // Filter by Dept_Type if provided (multiple types allowed)
            if ($request->has('dept_type')) {
                $types = explode(',', $request->dept_type);
                $query->whereIn('Dept_Type', $types);
            }

            // Search by name or code if provided
            if ($request->has('search')) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('Dept_Name', 'like', '%' . $search . '%')
                      ->orWhere('Dept_Code', 'like', '%' . $search . '%')
                      ->orWhere('Dept_Type', 'like', '%' . $search . '%');
                });
            }

            // Order by
            $orderBy = $request->get('order_by', 'Dept_Id');
            $orderDirection = $request->get('order_dir', 'asc');
            $query->orderBy($orderBy, $orderDirection);

            // Pagination (default 10 per page)
            $perPage = $request->get('per_page', 10);
            $departments = $query->paginate($perPage);

            // Get unique department types for reference
            $uniqueTypes = Department::select('Dept_Type')->distinct()->pluck('Dept_Type');

            return response()->json([
                'success' => true,
                'message' => 'Departments retrieved successfully',
                'data' => $departments->items(),
                'available_types' => $uniqueTypes,
                'pagination' => [
                    'total' => $departments->total(),
                    'per_page' => $departments->perPage(),
                    'current_page' => $departments->currentPage(),
                    'last_page' => $departments->lastPage(),
                    'from' => $departments->firstItem(),
                    'to' => $departments->lastItem()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve departments',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
 * Get all departments without pagination (for dropdowns, exports, etc.)
 */
public function getAllDepartments(Request $request)
{
    try {
        $query = Department::query();

        // Optional filter by type
        if ($request->has('dept_type')) {
            $types = explode(',', $request->dept_type);
            $query->whereIn('Dept_Type', $types);
        }

        // Optional search
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('Dept_Name', 'like', '%' . $search . '%')
                  ->orWhere('Dept_Code', 'like', '%' . $search . '%');
            });
        }

        // Order by
        $orderBy = $request->get('order_by', 'Dept_Id');
        $orderDirection = $request->get('order_dir', 'asc');
        $query->orderBy($orderBy, $orderDirection);

        // Get all data (no pagination)
        $departments = $query->get();

        return response()->json([
            'success' => true,
            'message' => 'All departments retrieved successfully',
            'data' => $departments,
            'count' => $departments->count()
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to retrieve all departments',
            'error' => $e->getMessage()
        ], 500);
    }
}

    /**
     * Store a newly created department
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'Dept_Code' => 'nullable|string|max:50|unique:department_master,Dept_Code',
                'Dept_Name' => 'required|string|max:25',
                'Dept_Type' => 'required|string|max:100' // Increased to 5 chars, no restrictions
            ], [
                'Dept_Code.unique' => 'This department code already exists',
                'Dept_Code.max' => 'Department code should not exceed 5 characters',
                'Dept_Name.max' => 'Department name should not exceed 25 characters',
                'Dept_Type.max' => 'Department type should not exceed 5 characters'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $department = Department::create([
                'Dept_Code' => $request->Dept_Code,
                'Dept_Name' => $request->Dept_Name,
                'Dept_Type' => strtoupper($request->Dept_Type) // Convert to uppercase for consistency
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Department created successfully',
                'data' => $department
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create department',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified department
     */
    public function show($id)
    {
        try {
            $department = Department::find($id);

            if (!$department) {
                return response()->json([
                    'success' => false,
                    'message' => 'Department not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Department retrieved successfully',
                'data' => $department
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve department',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified department
     */
    public function update(Request $request, $id)
    {
        try {
            $department = Department::find($id);

            if (!$department) {
                return response()->json([
                    'success' => false,
                    'message' => 'Department not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Dept_Code' => 'sometimes|required|string|max:50|unique:department_master,Dept_Code,' . $id . ',Dept_Id',
                'Dept_Name' => 'sometimes|required|string|max:25',
                'Dept_Type' => 'sometimes|required|string|max:100'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Prepare update data
            $updateData = [];
            if ($request->has('Dept_Code')) {
                $updateData['Dept_Code'] = $request->Dept_Code;
            }
            if ($request->has('Dept_Name')) {
                $updateData['Dept_Name'] = $request->Dept_Name;
            }
            if ($request->has('Dept_Type')) {
                $updateData['Dept_Type'] = strtoupper($request->Dept_Type);
            }

            $department->update($updateData);

            return response()->json([
                'success' => true,
                'message' => 'Department updated successfully',
                'data' => $department
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update department',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified department
     */
    public function destroy($id)
    {
        try {
            $department = Department::find($id);

            if (!$department) {
                return response()->json([
                    'success' => false,
                    'message' => 'Department not found'
                ], 404);
            }

            $department->delete();

            return response()->json([
                'success' => true,
                'message' => 'Department deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete department',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk update department types
     */
    public function bulkUpdateType(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'department_ids' => 'required|array|min:1',
                'department_ids.*' => 'exists:department_master,Dept_Id',
                'new_type' => 'required|string|max:5'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $updatedCount = Department::whereIn('Dept_Id', $request->department_ids)
                ->update(['Dept_Type' => strtoupper($request->new_type)]);

            return response()->json([
                'success' => true,
                'message' => $updatedCount . ' department(s) updated successfully',
                'updated_count' => $updatedCount
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update departments',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get department statistics by type
     */
    public function getTypeStatistics()
    {
        try {
            $statistics = Department::select('Dept_Type', DB::raw('count(*) as total'))
                ->groupBy('Dept_Type')
                ->orderBy('total', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'Department statistics retrieved successfully',
                'data' => $statistics
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve statistics',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all unique department types
     */
    public function getDepartmentTypes()
    {
        try {
            $types = Department::select('Dept_Type')
                ->distinct()
                ->orderBy('Dept_Type')
                ->pluck('Dept_Type');

            return response()->json([
                'success' => true,
                'message' => 'Department types retrieved successfully',
                'data' => $types
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve department types',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
