<?php

namespace App\Http\Controllers;

use App\Models\DepartmentManagement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DepartmentManagementController extends Controller
{
    public function index(Request $request)
    {
        try {
            $query = DepartmentManagement::query();

            // Apply filters
            if ($request->has('dept_id')) {
                $query->where('Dept_Id', $request->dept_id);
            }

            if ($request->has('user_id')) {
                $query->where('Dept_User_Id', $request->user_id);
            }

            if ($request->has('designation_id')) {
                $query->where('designation_id', $request->designation_id);
            }
            if ($request->has('reporting_designation_id')) {
                $query->where('reporting_designation_id', $request->reporting_designation_id);
            }


            if ($request->has('wef_date')) {
                $query->whereDate('Wef_Date', Carbon::parse($request->wef_date));
            }

            // Load relationships
            $with = [];
            if ($request->has('with')) {
                $relations = explode(',', $request->with);
                $allowedRelations = [
                    'department',
                    'user',
                    'designation',
                    'reportingManager',
                    'subordinates'
                ];
                $with = array_intersect($relations, $allowedRelations);
            } else {
                // Load default relations - ADD reportingManager HERE
                $with = ['department', 'user', 'designation', 'reportingDesignation'];
            }

            $query->with($with);

            // Sorting
            $sortField = $request->get('sort_by', 'Dept_Mng_Id');
            $sortOrder = $request->get('sort_order', 'desc');
            $query->orderBy($sortField, $sortOrder);

            // Pagination
            $perPage = $request->get('per_page', 15);
            $departmentManagements = $query->paginate($perPage);

            // Format response data
            $formattedData = $departmentManagements->map(function ($item) {
                return $this->formatDepartmentManagementData($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'Department management records retrieved successfully',
                'data' => $formattedData,
                'pagination' => [
                    'current_page' => $departmentManagements->currentPage(),
                    'per_page' => $departmentManagements->perPage(),
                    'total' => $departmentManagements->total(),
                    'last_page' => $departmentManagements->lastPage(),
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve department management records',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created department management record.
     */
    public function store(Request $request)
    {
        DB::beginTransaction();

        try {
            $validator = Validator::make($request->all(), [
                'Dept_Id' => 'required|integer|exists:main_departments,id',
                'Wef_Date' => 'required|date_format:Y-m-d',
                'Dept_User_Id' => 'required|integer|exists:gmc_users,user_id',
                'designation_id' => 'nullable|integer|exists:designation_menus,id',
                'reporting_designation_id' => 'nullable|integer|exists:designation_menus,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Check if user already has assignment in same department
            $existingAssignment = DepartmentManagement::where('Dept_User_Id', $request->Dept_User_Id)
                ->where('Dept_Id', $request->Dept_Id)
                ->where('Wef_Date', $request->Wef_Date)
                ->first();

            if ($existingAssignment) {
                return response()->json([
                    'success' => false,
                    'message' => 'User already has an assignment in this department with the same effective date'
                ], 409);
            }

            $departmentManagement = DepartmentManagement::create([
                'Dept_Id' => $request->Dept_Id,
                'Wef_Date' => $request->Wef_Date,
                'Dept_User_Id' => $request->Dept_User_Id,
                'designation_id' => $request->designation_id,   // ✅
                'reporting_designation_id' => $request->reporting_designation_id,
            ]);

            DB::commit();

            // Load relationships for response
            $departmentManagement->load(['department', 'user', 'designation', 'reportingDesignation']);

            return response()->json([
                'success' => true,
                'message' => 'Department management record created successfully',
                'data' => $this->formatDepartmentManagementData($departmentManagement)
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to create department management record',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            $departmentManagement = DepartmentManagement::with([
                'department',
                'user',
                'designation',
                'reportingManager',
                'subordinates.user',
                'reportingDesignation', // 🔹 ADD THIS

                'subordinates.designation'
            ])->find($id);

            if (!$departmentManagement) {
                return response()->json([
                    'success' => false,
                    'message' => 'Department management record not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Department management record retrieved successfully',
                'data' => $this->formatDepartmentManagementData($departmentManagement)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve department management record',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified department management record.
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            $departmentManagement = DepartmentManagement::find($id);

            if (!$departmentManagement) {
                return response()->json([
                    'success' => false,
                    'message' => 'Department management record not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Dept_Id' => 'sometimes|integer|exists:main_departments,id',
                'Wef_Date' => 'sometimes|date_format:Y-m-d',
                'Dept_User_Id' => 'sometimes|integer|exists:gmc_users,user_id',
                'designation_id' => 'nullable|integer|exists:designation_menus,id',
                'reporting_designation_id' => 'nullable|integer|exists:designation_menus,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Check for duplicate assignment if updating user or department
            if ($request->has('Dept_User_Id') || $request->has('Dept_Id') || $request->has('Wef_Date')) {
                $existingAssignment = DepartmentManagement::where('Dept_User_Id', $request->Dept_User_Id ?? $departmentManagement->Dept_User_Id)
                    ->where('Dept_Id', $request->Dept_Id ?? $departmentManagement->Dept_Id)
                    ->where('Wef_Date', $request->Wef_Date ?? $departmentManagement->Wef_Date)
                    ->where('Dept_Mng_Id', '!=', $id)
                    ->first();

                if ($existingAssignment) {
                    return response()->json([
                        'success' => false,
                        'message' => 'User already has an assignment in this department with the same effective date'
                    ], 409);
                }
            }

            $departmentManagement->update($request->only([
                'Dept_Id',
                'Wef_Date',
                'Dept_User_Id',
                'Dept_Post_ID',
                'reporting_designation_id',
            ]));

            DB::commit();

            // Refresh with relationships
            $departmentManagement->load(['department', 'user', 'designation', 'reportingDesignation']);

            return response()->json([
                'success' => true,
                'message' => 'Department management record updated successfully',
                'data' => $this->formatDepartmentManagementData($departmentManagement)
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to update department management record',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified department management record.
     */
    public function destroy($id)
    {
        DB::beginTransaction();

        try {
            $departmentManagement = DepartmentManagement::find($id);

            if (!$departmentManagement) {
                return response()->json([
                    'success' => false,
                    'message' => 'Department management record not found'
                ], 404);
            }

            $departmentManagement->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Department management record deleted successfully'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete department management record',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get users by department ID
     */
    public function getUsersByDepartment($deptId)
    {
        try {
            $users = DepartmentManagement::with(['user', 'designation', 'reportingManager'])
                ->where('Dept_Id', $deptId)
                ->current()
                ->get()
                ->unique('Dept_User_Id');

            if ($users->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No users found in this department'
                ], 404);
            }

            $formattedData = $users->map(function ($item) {
                return $this->formatDepartmentManagementData($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'Users retrieved successfully',
                'data' => $formattedData
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve users by department',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get reporting hierarchy for a user
     */
    public function getReportingHierarchy($userId)
    {
        try {
            $userAssignment = DepartmentManagement::with(['user', 'reportingManager'])
                ->where('Dept_User_Id', $userId)
                ->current()
                ->first();

            if (!$userAssignment) {
                return response()->json([
                    'success' => false,
                    'message' => 'User assignment not found'
                ], 404);
            }

            $hierarchy = [];
            $current = $userAssignment;
            $level = 0;

            // Build hierarchy upward
            while ($current && $current->reporting_user_id) {
                $level++;
                $managerAssignment = DepartmentManagement::with(['user', 'designation'])
                    ->where('Dept_User_Id', $current->reporting_user_id)
                    ->current()
                    ->first();

                if ($managerAssignment) {
                    $hierarchy[] = [
                        'level' => $level,
                        'data' => $this->formatDepartmentManagementData($managerAssignment)
                    ];
                    $current = $managerAssignment;
                } else {
                    break;
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Reporting hierarchy retrieved successfully',
                'user' => $this->formatDepartmentManagementData($userAssignment),
                'hierarchy' => $hierarchy,
                'has_subordinates' => $userAssignment->has_subordinates
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve reporting hierarchy',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get subordinates for a user
     */
    public function getSubordinates($userId)
    {
        try {
            $subordinates = DepartmentManagement::with(['user', 'designation', 'department'])
                ->where('reporting_user_id', $userId)
                ->current()
                ->get();

            if ($subordinates->isEmpty()) {
                return response()->json([
                    'success' => true,
                    'message' => 'No subordinates found for this user',
                    'data' => []
                ]);
            }

            $formattedData = $subordinates->map(function ($item) {
                return $this->formatDepartmentManagementData($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'Subordinates retrieved successfully',
                'count' => $subordinates->count(),
                'data' => $formattedData
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve subordinates',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get current department assignments for a user
     */
    public function getUserCurrentAssignments($userId)
    {
        try {
            $assignments = DepartmentManagement::with(['department', 'designation', 'reportingManager'])
                ->where('Dept_User_Id', $userId)
                ->current()
                ->get();

            if ($assignments->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No current assignments found for this user'
                ], 404);
            }

            $formattedData = $assignments->map(function ($item) {
                return $this->formatDepartmentManagementData($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'User assignments retrieved successfully',
                'data' => $formattedData
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve user assignments',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get users by reporting manager
     */
    public function getUsersByReportingManager($managerId)
    {
        try {
            $users = DepartmentManagement::with(['user', 'designation', 'department'])
                ->where('reporting_designation_id', $designationId)
                ->current()
                ->get();

            if ($users->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No users found under this reporting manager'
                ], 404);
            }

            $formattedData = $users->map(function ($item) {
                return $this->formatDepartmentManagementData($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'Users retrieved successfully',
                'data' => $formattedData
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve users by reporting manager',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk assign users to department
     */
    public function bulkAssign(Request $request)
    {
        DB::beginTransaction();

        try {
            $validator = Validator::make($request->all(), [
                'Dept_Id' => 'required|integer|exists:main_departments,id',
                'Wef_Date' => 'required|date_format:Y-m-d',
                'users' => 'required|array|min:1',
                'users.*.Dept_User_Id' => 'required|integer|exists:gmc_users,user_id',
                'users.*.Dept_Post_ID' => 'required|integer|exists:employee_designation_master,designation_id',
                'users.*.reporting_user_id' => 'nullable|integer|exists:gmc_users,user_id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $createdRecords = [];
            $failedRecords = [];

            foreach ($request->users as $userData) {
                try {
                    // Check for existing assignment
                    $existing = DepartmentManagement::where('Dept_User_Id', $userData['Dept_User_Id'])
                        ->where('Dept_Id', $request->Dept_Id)
                        ->where('Wef_Date', $request->Wef_Date)
                        ->first();

                    if ($existing) {
                        $failedRecords[] = [
                            'user_id' => $userData['Dept_User_Id'],
                            'error' => 'Already assigned'
                        ];
                        continue;
                    }

                    $record = DepartmentManagement::create([
                        'Dept_Id' => $request->Dept_Id,
                        'Wef_Date' => $request->Wef_Date,
                        'Dept_User_Id' => $userData['Dept_User_Id'],
                        'Dept_Post_ID' => $userData['Dept_Post_ID'],
                        'reporting_user_id' => $userData['reporting_user_id'] ?? null,
                    ]);

                    $createdRecords[] = $record;
                } catch (\Exception $e) {
                    $failedRecords[] = [
                        'user_id' => $userData['Dept_User_Id'],
                        'error' => $e->getMessage()
                    ];
                }
            }

            DB::commit();

            // Load relationships for created records
            $createdRecords = DepartmentManagement::with(['user', 'designation'])
                ->whereIn('Dept_Mng_Id', collect($createdRecords)->pluck('Dept_Mng_Id')->toArray())
                ->get();

            $formattedData = $createdRecords->map(function ($item) {
                return $this->formatDepartmentManagementData($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'Bulk assignment completed',
                'created' => count($createdRecords),
                'failed' => count($failedRecords),
                'data' => $formattedData,
                'failures' => $failedRecords
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to process bulk assignment',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get department statistics
     */
    public function getStatistics()
    {
        try {
            $stats = [
                'total_assignments' => DepartmentManagement::count(),
                'total_departments' => DB::table('department_management')->distinct('Dept_Id')->count('Dept_Id'),
                'total_users' => DB::table('department_management')->distinct('Dept_User_Id')->count('Dept_User_Id'),
                'users_without_manager' => DepartmentManagement::whereNull('reporting_user_id')->count(),
                'assignments_by_department' => DepartmentManagement::select('Dept_Id', DB::raw('count(*) as count'))
                    ->groupBy('Dept_Id')
                    ->with('department')
                    ->get()
                    ->map(function ($item) {
                        return [
                            'dept_id' => $item->Dept_Id,
                            'count' => $item->count,
                            'department' => $item->department ? [
                                'id' => $item->department->id,
                                'desc' => $item->department->desc,
                            ] : null
                        ];
                    }),
                'recent_assignments' => DepartmentManagement::with(['department', 'user'])
                    ->orderBy('created_at', 'desc')
                    ->limit(10)
                    ->get()
                    ->map(function ($item) {
                        return $this->formatDepartmentManagementData($item);
                    })
            ];

            return response()->json([
                'success' => true,
                'message' => 'Statistics retrieved successfully',
                'data' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve statistics',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    /**
     * Helper method to format department management data
     */
    private function formatDepartmentManagementData($item)
    {
        // Format Wef_Date
        $wefDate = $item->Wef_Date;
        if ($wefDate instanceof \Carbon\Carbon) {
            $formattedWefDate = $wefDate->format('d-m-Y');
        } else {
            try {
                $formattedWefDate = Carbon::parse($wefDate)->format('d-m-Y');
            } catch (\Exception $e) {
                $formattedWefDate = $wefDate;
            }
        }

        $data = [
            'dept_mng_id' => $item->Dept_Mng_Id,
            'dept_id' => $item->Dept_Id,
            'wef_date' => $item->Wef_Date,
            'formatted_wef_date' => $formattedWefDate,
            'dept_user_id' => $item->Dept_User_Id,
            'designation_id' => $item->designation_id,
            'reporting_designation_id' => $item->reporting_designation_id, // ✅ ADD THIS
            'reporting_user_id' => $item->reporting_user_id,
            'created_at' => $item->created_at ? $item->created_at->format('Y-m-d H:i:s') : null,
            'updated_at' => $item->updated_at ? $item->updated_at->format('Y-m-d H:i:s') : null,
        ];

        // Add relationships if loaded
        // if ($item->relationLoaded('designationMenu') && $item->designationMenu) {
        //     $data['designation'] = [
        //         'id' => $item->designationMenu->id,
        //         'name' => $item->designationMenu->name ?? null,
        //     ];
        // }

        // designation – FULL DATA
        if ($item->relationLoaded('designation') && $item->designation) {
            $data['designation'] = $item->designation->toArray();
        }
        if ($item->relationLoaded('department') && $item->department) {
            $data['department'] = $item->department->toArray();
        }


        if ($item->relationLoaded('user') && $item->user) {
            $data['user'] = [
                'user_id' => $item->user->user_id,
                'username' => $item->user->username,
                'email' => $item->user->email ?? null,
                'mobile' => $item->user->mobile ?? null,
                'Role_Id' => $item->user->Role_Id,
            ];
        }

        // designation – FULL DATA
        if ($item->relationLoaded('designation') && $item->designation) {
            $data['designation'] = $item->designation->toArray();
        }


        // if ($item->relationLoaded('reportingManager') && $item->reportingManager) {
        //     $data['reporting_manager'] = [
        //         'user_id' => $item->reportingManager->user_id,
        //         'username' => $item->reportingManager->username,
        //         'email' => $item->reportingManager->email ?? null,
        //         'mobile' => $item->reportingManager->mobile ?? null,
        //         'Role_Id' => $item->reportingManager->Role_Id,
        //     ];
        // }
        if ($item->relationLoaded('reportingDesignation') && $item->reportingDesignation) {
            $data['reporting_designation'] = [
                'id'         => $item->reportingDesignation->id,
                'title'      => $item->reportingDesignation->title ?? null,
                'desc'       => $item->reportingDesignation->desc ?? null,
                'func_id'    => $item->reportingDesignation->func_id ?? null,
                'Upgrp'      => $item->reportingDesignation->Upgrp ?? null,
                'Grcod'      => $item->reportingDesignation->Grcod ?? null,
                'Grsrn'      => $item->reportingDesignation->Grsrn ?? null,
                'Grsh'       => $item->reportingDesignation->Grsh ?? null,
                'isvisible'  => $item->reportingDesignation->isvisible ?? null,
                'Menu_Links' => $item->reportingDesignation->Menu_Links ?? null,
                'created_at' => optional($item->reportingDesignation->created_at)->toDateTimeString(),
                'updated_at' => optional($item->reportingDesignation->updated_at)->toDateTimeString(),
            ];
        }



        if ($item->relationLoaded('subordinates')) {
            $data['subordinates'] = $item->subordinates->map(function ($subordinate) {
                return [
                    'dept_mng_id' => $subordinate->Dept_Mng_Id,
                    'dept_user_id' => $subordinate->Dept_User_Id,
                    'user_name' => $subordinate->user ? $subordinate->user->username : null,
                    'designation_name' => $subordinate->designation ? $subordinate->designation->designation_name : null,
                ];
            })->toArray();

            $data['subordinates_count'] = $item->subordinates->count();
        }

        // Add computed attributes
        if (isset($item->has_subordinates)) {
            $data['has_subordinates'] = $item->has_subordinates;
        }

        if (isset($item->reporting_manager_name)) {
            $data['reporting_manager_name'] = $item->reporting_manager_name;
        }

        return $data;
    }
}
