<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\GmcDeviceRegistrationQueue;
use App\Models\GmcDeviceApproval;
use App\Models\GmcDeviceApprovalLog;
use App\Models\GmcUser;
use App\Models\Department;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class DeviceApprovalController extends Controller
{
    /**
     * Get pending device approval queue
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getPendingQueue(Request $request): JsonResponse
    {
        try {
            $perPage = $request->get('per_page', 20);
            $page = $request->get('page', 1);
            $departmentId = $request->get('department_id');
            $deviceType = $request->get('device_type');
            $dateFrom = $request->get('date_from');
            $dateTo = $request->get('date_to');

            $query = GmcDeviceRegistrationQueue::with(['user', 'department'])
                ->where('status', 'PENDING')
                ->where('is_citizen', false) // Only show employees
                ->orderBy('submitted_at', 'asc');

            // Apply filters
            if ($departmentId) {
                $query->where('department_id', $departmentId);
            }

            if ($deviceType) {
                $query->where('device_type', $deviceType);
            }

            if ($dateFrom) {
                $query->whereDate('submitted_at', '>=', $dateFrom);
            }

            if ($dateTo) {
                $query->whereDate('submitted_at', '<=', $dateTo);
            }

            // Check if user has permission to view specific departments
            $currentUser = auth()->user();
            if ($currentUser && $currentUser->Role_Id != 2) { // Not superadmin
                // Get departments user can manage
                $allowedDepartments = $this->getManageableDepartments($currentUser->user_id);
                if (!empty($allowedDepartments)) {
                    $query->whereIn('department_id', $allowedDepartments);
                }
            }

            $queue = $query->paginate($perPage, ['*'], 'page', $page);

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Pending device approvals retrieved',
                'data' => $this->formatQueueData($queue->items()),
                'meta' => [
                    'current_page' => $queue->currentPage(),
                    'total' => $queue->total(),
                    'per_page' => $queue->perPage(),
                    'last_page' => $queue->lastPage(),
                    'total_pending' => $queue->total()
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Get Pending Queue Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to retrieve pending approvals'
            ], 500);
        }
    }

    /**
     * Format queue data for response
     */
    private function formatQueueData(array $items): array
    {
        return array_map(function ($item) {
            $registrationData = $item->registration_data ?? [];

            return [
                'queue_id' => $item->queue_id,
                'user' => [
                    'user_id' => $item->user_id,
                    'username' => $registrationData['username'] ?? $item->user->username ?? 'N/A',
                    'email' => $registrationData['email'] ?? $item->user->email ?? 'N/A',
                    'mobile' => $registrationData['mobile'] ?? $item->user->mobile ?? 'N/A',
                    'Role_Id' => $registrationData['Role_Id'] ?? $item->user->Role_Id ?? null,
                    'role_name' => $registrationData['role_name'] ?? 'Unknown'
                ],
                'device_info' => [
                    'ip_address' => $item->ip_address,
                    'mac_address' => $item->mac_address,
                    'hdd_serial' => $item->hdd_serial,
                    'device_name' => $item->device_name,
                    'device_type' => $item->device_type,
                    'operating_system' => $item->operating_system,
                    'browser' => $item->browser,
                    'browser_fingerprint' => substr($item->browser_fingerprint, 0, 20) . '...',
                    'submitted_at' => $item->submitted_at->format('Y-m-d H:i:s'),
                    'hours_ago' => $item->submitted_at->diffInHours(now())
                ],
                'department' => $item->department ? [
                    'department_id' => $item->department->department_id,
                    'department_name' => $item->department->department_name
                ] : null,
                'location_id' => $item->location_id,
                'status' => $item->status,
                'is_citizen' => $item->is_citizen
            ];
        }, $items);
    }

    /**
     * Approve a device registration
     *
     * @param Request $request
     * @param int $queueId
     * @return JsonResponse
     */
    public function approveDevice(Request $request, $queueId): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'remarks' => 'nullable|string|max:500',
                'validity_days' => 'integer|min:30|max:3650',
                'device_category' => 'in:PRIMARY,SECONDARY,MOBILE,EMERGENCY',
                'make_primary' => 'boolean',
                'notify_user' => 'boolean'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $queue = GmcDeviceRegistrationQueue::with('user')->find($queueId);

            if (!$queue) {
                return response()->json([
                    'success' => false,
                    'message' => 'Device registration request not found'
                ], 404);
            }

            if ($queue->status !== 'PENDING') {
                return response()->json([
                    'success' => false,
                    'message' => 'This request has already been processed'
                ], 400);
            }

            if ($queue->is_citizen) {
                return response()->json([
                    'success' => false,
                    'message' => 'Citizen devices are auto-approved'
                ], 400);
            }

            $currentUser = auth()->user();
            if (!$this->canApproveDevice($currentUser, $queue)) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to approve this device'
                ], 403);
            }

            // Check device quota
            if (!$this->checkDeviceQuota($queue->user_id)) {
                return response()->json([
                    'success' => false,
                    'message' => 'User has reached maximum device limit (3 devices)'
                ], 400);
            }

            // Set validity (default 1 year, configurable)
            $validityDays = $request->validity_days ?? 365;
            $deviceCategory = $request->device_category ?? 'PRIMARY';
            $makePrimary = $request->boolean('make_primary', true);

            // Approve the queue entry
            $queue->approve($currentUser->user_id, $request->remarks);

            // Create device approval record
            $approval = GmcDeviceApproval::createFromQueue($queue, $currentUser->user_id);

            // Update approval with additional data
            $approval->update([
                'device_category' => $deviceCategory,
                'is_primary_device' => $makePrimary,
                'valid_until' => now()->addDays($validityDays)
            ]);

            // If making primary, ensure only one primary device
            if ($makePrimary) {
                GmcDeviceApproval::where('user_id', $queue->user_id)
                    ->where('approval_id', '!=', $approval->approval_id)
                    ->update(['is_primary_device' => false]);
            }

            // Log the approval
            GmcDeviceApprovalLog::logApproval(
                $queue->queue_id,
                $approval->approval_id,
                $queue->user_id,
                $currentUser->user_id,
                'APPROVED',
                'Device approved by ' . $currentUser->username .
                ($request->remarks ? ': ' . $request->remarks : '')
            );

            // Update user's last login if this is their first approved device
            $this->updateUserLoginInfo($queue->user_id);

            // Send notification if requested
            if ($request->boolean('notify_user', true)) {
                $this->sendApprovalNotification($queue->user, $approval);
            }

            return response()->json([
                'success' => true,
                'message' => 'Device approved successfully',
                'data' => [
                    'approval_id' => $approval->approval_id,
                    'user_id' => $queue->user_id,
                    'username' => $queue->user->username ?? 'N/A',
                    'device_unique_id' => $approval->device_unique_id,
                    'device_name' => $approval->device_name,
                    'valid_from' => $approval->valid_from->format('Y-m-d H:i:s'),
                    'valid_until' => $approval->valid_until->format('Y-m-d H:i:s'),
                    'validity_days' => $validityDays,
                    'status' => $approval->status,
                    'approved_by' => $currentUser->username,
                    'approved_at' => $approval->approved_at->format('Y-m-d H:i:s'),
                    'device_category' => $deviceCategory,
                    'is_primary' => $makePrimary
                ]
            ], 200);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            Log::error('Approve Device Error: ' . $e->getMessage());
            Log::error($e->getTraceAsString());

            return response()->json([
                'success' => false,
                'message' => 'Failed to approve device'
            ], 500);
        }
    }

    /**
     * Reject a device registration
     *
     * @param Request $request
     * @param int $queueId
     * @return JsonResponse
     */
    public function rejectDevice(Request $request, $queueId): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'reason' => 'required|string|max:1000',
                'instructions' => 'nullable|string|max:500',
                'notify_user' => 'boolean'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $queue = GmcDeviceRegistrationQueue::with('user')->find($queueId);

            if (!$queue) {
                return response()->json([
                    'success' => false,
                    'message' => 'Device registration request not found'
                ], 404);
            }

            if ($queue->status !== 'PENDING') {
                return response()->json([
                    'success' => false,
                    'message' => 'This request has already been processed'
                ], 400);
            }

            if ($queue->is_citizen) {
                return response()->json([
                    'success' => false,
                    'message' => 'Citizen devices are auto-approved'
                ], 400);
            }

            $currentUser = auth()->user();
            if (!$this->canApproveDevice($currentUser, $queue)) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to reject this device'
                ], 403);
            }

            // Reject the queue entry
            $queue->reject($currentUser->user_id, $request->reason);

            // Log the rejection
            GmcDeviceApprovalLog::logApproval(
                $queue->queue_id,
                null,
                $queue->user_id,
                $currentUser->user_id,
                'REJECTED',
                'Device rejected by ' . $currentUser->username .
                ': ' . $request->reason .
                ($request->instructions ? ' | Instructions: ' . $request->instructions : '')
            );

            // Send notification if requested
            if ($request->boolean('notify_user', true)) {
                $this->sendRejectionNotification($queue->user, $request->reason, $request->instructions);
            }

            return response()->json([
                'success' => true,
                'message' => 'Device registration rejected',
                'data' => [
                    'queue_id' => $queue->queue_id,
                    'user_id' => $queue->user_id,
                    'username' => $queue->user->username ?? 'N/A',
                    'status' => $queue->status,
                    'rejection_reason' => $request->reason,
                    'rejected_by' => $currentUser->username,
                    'rejected_at' => $queue->processed_at->format('Y-m-d H:i:s'),
                    'instructions' => $request->instructions
                ]
            ], 200);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            Log::error('Reject Device Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to reject device'
            ], 500);
        }
    }

    /**
     * Get user's approved devices
     *
     * @param Request $request
     * @param int $userId
     * @return JsonResponse
     */
    public function getUserDevices(Request $request, $userId): JsonResponse
    {
        try {
            $user = GmcUser::find($userId);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            $currentUser = auth()->user();
            if (!$this->canViewUserDevices($currentUser, $user)) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to view these devices'
                ], 403);
            }

            $includeExpired = $request->boolean('include_expired', false);
            $includeInactive = $request->boolean('include_inactive', false);

            $query = GmcDeviceApproval::with(['department'])
                ->where('user_id', $userId)
                ->orderBy('is_primary_device', 'desc')
                ->orderBy('valid_until', 'desc');

            if (!$includeExpired) {
                $query->where('valid_until', '>=', now());
            }

            if (!$includeInactive) {
                $query->where('is_active', true)
                      ->where('status', 'ACTIVE');
            }

            $devices = $query->get();

            // Get pending requests
            $pendingQueue = GmcDeviceRegistrationQueue::where('user_id', $userId)
                ->where('status', 'PENDING')
                ->where('is_citizen', false)
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'User devices retrieved successfully',
                'data' => [
                    'user' => [
                        'user_id' => $user->user_id,
                        'username' => $user->username,
                        'email' => $user->email,
                        'Role_Id' => $user->Role_Id,
                        'department' => $user->department_id ? Department::find($user->department_id) : null
                    ],
                    'devices' => $this->formatDeviceData($devices),
                    'pending_requests' => $this->formatPendingData($pendingQueue),
                    'device_policy' => $this->getDevicePolicy($user->Role_Id),
                    'stats' => [
                        'total_devices' => $devices->count(),
                        'active_devices' => $devices->where('is_active', true)->where('status', 'ACTIVE')->count(),
                        'primary_device' => $devices->where('is_primary_device', true)->first(),
                        'expired_devices' => $devices->where('valid_until', '<', now())->count(),
                        'pending_requests' => $pendingQueue->count()
                    ]
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Get User Devices Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to retrieve user devices'
            ], 500);
        }
    }

    /**
     * Format device data for response
     */
    private function formatDeviceData($devices): array
    {
        return $devices->map(function ($device) {
            $isExpired = $device->valid_until->isPast();
            $daysRemaining = $isExpired ? 0 : $device->valid_until->diffInDays(now());

            return [
                'approval_id' => $device->approval_id,
                'device_unique_id' => $device->device_unique_id,
                'device_name' => $device->device_name,
                'device_type' => $device->device_type,
                'device_category' => $device->device_category,
                'ip_address' => $device->ip_address,
                'mac_address' => $device->mac_address,
                'hdd_serial' => $device->hdd_serial ? substr($device->hdd_serial, 0, 10) . '...' : null,
                'operating_system' => $device->operating_system,
                'browser' => $device->browser,
                'status' => $device->status,
                'is_active' => (bool) $device->is_active,
                'is_primary' => (bool) $device->is_primary_device,
                'valid_from' => $device->valid_from->format('Y-m-d H:i:s'),
                'valid_until' => $device->valid_until->format('Y-m-d H:i:s'),
                'days_remaining' => $daysRemaining,
                'is_expired' => $isExpired,
                'login_count' => $device->login_count,
                'last_login_at' => $device->last_login_at ? $device->last_login_at->format('Y-m-d H:i:s') : null,
                'approved_by' => $device->approver ? $device->approver->username : 'System',
                'approved_at' => $device->approved_at ? $device->approved_at->format('Y-m-d H:i:s') : null,
                'approval_type' => $device->approval_type,
                'department' => $device->department ? [
                    'department_id' => $device->department->department_id,
                    'department_name' => $device->department->department_name
                ] : null
            ];
        })->toArray();
    }

    /**
     * Format pending data for response
     */
    private function formatPendingData($pendingQueue): array
    {
        return $pendingQueue->map(function ($queue) {
            return [
                'queue_id' => $queue->queue_id,
                'device_name' => $queue->device_name,
                'device_type' => $queue->device_type,
                'ip_address' => $queue->ip_address,
                'mac_address' => $queue->mac_address,
                'submitted_at' => $queue->submitted_at->format('Y-m-d H:i:s'),
                'days_pending' => $queue->submitted_at->diffInDays(now())
            ];
        })->toArray();
    }

    /**
     * Renew device approval
     *
     * @param Request $request
     * @param int $approvalId
     * @return JsonResponse
     */
    public function renewDevice(Request $request, $approvalId): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'validity_days' => 'required|integer|min:30|max:3650',
                'remarks' => 'nullable|string|max:500',
                'notify_user' => 'boolean'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $approval = GmcDeviceApproval::with('user')->find($approvalId);

            if (!$approval) {
                return response()->json([
                    'success' => false,
                    'message' => 'Device approval not found'
                ], 404);
            }

            $currentUser = auth()->user();
            if (!$this->canRenewDevice($currentUser, $approval)) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to renew this device'
                ], 403);
            }

            // Store old validity for logging
            $oldValidUntil = $approval->valid_until;
            $validityDays = $request->validity_days;

            // Renew the device
            $approval->renew($validityDays, $currentUser->user_id);

            // Log the renewal
            GmcDeviceApprovalLog::logApproval(
                null,
                $approval->approval_id,
                $approval->user_id,
                $currentUser->user_id,
                'RENEWED',
                'Device renewed by ' . $currentUser->username .
                ' for ' . $validityDays . ' days' .
                ($request->remarks ? ': ' . $request->remarks : '')
            );

            // Send notification if requested
            if ($request->boolean('notify_user', true)) {
                $this->sendRenewalNotification($approval->user, $approval, $validityDays);
            }

            return response()->json([
                'success' => true,
                'message' => 'Device approval renewed successfully',
                'data' => [
                    'approval_id' => $approval->approval_id,
                    'user_id' => $approval->user_id,
                    'username' => $approval->user->username ?? 'N/A',
                    'device_name' => $approval->device_name,
                    'old_valid_until' => $oldValidUntil->format('Y-m-d H:i:s'),
                    'new_valid_until' => $approval->valid_until->format('Y-m-d H:i:s'),
                    'validity_days' => $validityDays,
                    'renewed_by' => $currentUser->username,
                    'renewed_at' => now()->format('Y-m-d H:i:s')
                ]
            ], 200);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            Log::error('Renew Device Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to renew device approval'
            ], 500);
        }
    }

    /**
     * Revoke device approval
     *
     * @param Request $request
     * @param int $approvalId
     * @return JsonResponse
     */
    public function revokeDevice(Request $request, $approvalId): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'reason' => 'required|string|max:1000',
                'notify_user' => 'boolean',
                'invalidate_sessions' => 'boolean'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            $approval = GmcDeviceApproval::with('user')->find($approvalId);

            if (!$approval) {
                return response()->json([
                    'success' => false,
                    'message' => 'Device approval not found'
                ], 404);
            }

            $currentUser = auth()->user();
            if (!$this->canRevokeDevice($currentUser, $approval)) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to revoke this device'
                ], 403);
            }

            // Check if this is the only active device
            $activeDeviceCount = GmcDeviceApproval::where('user_id', $approval->user_id)
                ->where('is_active', true)
                ->where('status', 'ACTIVE')
                ->where('approval_id', '!=', $approvalId)
                ->count();

            if ($activeDeviceCount === 0 && $request->boolean('force', false) === false) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot revoke the only active device. Use force parameter if needed.',
                    'data' => [
                        'has_other_devices' => false,
                        'warning' => 'User will lose all device access'
                    ]
                ], 400);
            }

            // Revoke the device
            $approval->revoke($request->reason);

            // Invalidate active sessions if requested
            if ($request->boolean('invalidate_sessions', true)) {
                $this->invalidateDeviceSessions($approval->user_id, $approval->device_unique_id);
            }

            // Log the revocation
            GmcDeviceApprovalLog::logApproval(
                null,
                $approval->approval_id,
                $approval->user_id,
                $currentUser->user_id,
                'REVOKED',
                'Device revoked by ' . $currentUser->username . ': ' . $request->reason
            );

            // Send notification if requested
            if ($request->boolean('notify_user', true)) {
                $this->sendRevocationNotification($approval->user, $approval, $request->reason);
            }

            return response()->json([
                'success' => true,
                'message' => 'Device approval revoked successfully',
                'data' => [
                    'approval_id' => $approval->approval_id,
                    'user_id' => $approval->user_id,
                    'username' => $approval->user->username ?? 'N/A',
                    'device_name' => $approval->device_name,
                    'status' => $approval->status,
                    'revoked_by' => $currentUser->username,
                    'revoked_at' => now()->format('Y-m-d H:i:s'),
                    'reason' => $request->reason,
                    'sessions_invalidated' => $request->boolean('invalidate_sessions', true),
                    'remaining_active_devices' => $activeDeviceCount
                ]
            ], 200);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            Log::error('Revoke Device Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to revoke device approval'
            ], 500);
        }
    }

    /**
     * Get device statistics
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getStatistics(Request $request): JsonResponse
    {
        try {
            $currentUser = auth()->user();
            if (!$this->canViewStatistics($currentUser)) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to view statistics'
                ], 403);
            }

            // Get time range
            $days = $request->get('days', 30);
            $startDate = now()->subDays($days);

            // Statistics
            $stats = [
                'total_approvals' => GmcDeviceApproval::count(),
                'active_approvals' => GmcDeviceApproval::where('is_active', true)
                    ->where('status', 'ACTIVE')
                    ->where('valid_until', '>=', now())
                    ->count(),
                'pending_requests' => GmcDeviceRegistrationQueue::where('status', 'PENDING')
                    ->where('is_citizen', false)
                    ->count(),
                'approved_today' => GmcDeviceApproval::whereDate('approved_at', today())->count(),
                'rejected_today' => GmcDeviceApprovalLog::where('action_type', 'REJECTED')
                    ->whereDate('performed_at', today())
                    ->count(),
                'expiring_soon' => GmcDeviceApproval::where('is_active', true)
                    ->where('status', 'ACTIVE')
                    ->whereBetween('valid_until', [now(), now()->addDays(30)])
                    ->count(),
                'citizen_devices' => GmcDeviceApproval::where('approval_type', 'AUTO_CITIZEN')->count(),
                'employee_devices' => GmcDeviceApproval::where('approval_type', 'MANUAL')->count(),
                'department_distribution' => $this->getDepartmentDistribution(),
                'device_type_distribution' => $this->getDeviceTypeDistribution(),
                'recent_activity' => $this->getRecentActivity($startDate)
            ];

            return response()->json([
                'success' => true,
                'message' => 'Device approval statistics retrieved',
                'data' => $stats
            ], 200);

        } catch (\Exception $e) {
            Log::error('Get Statistics Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to retrieve statistics'
            ], 500);
        }
    }

    /**
     * Check device hardware status
     *
     * @param Request $request
     * @param int $approvalId
     * @return JsonResponse
     */
    public function checkHardwareStatus(Request $request, $approvalId): JsonResponse
    {
        try {
            $approval = GmcDeviceApproval::with('user')->find($approvalId);

            if (!$approval) {
                return response()->json([
                    'success' => false,
                    'message' => 'Device approval not found'
                ], 404);
            }

            $currentUser = auth()->user();
            if (!$this->canCheckHardware($currentUser, $approval)) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to check hardware status'
                ], 403);
            }

            // This would normally call an external service or run diagnostics
            // For now, simulate hardware check
            $hardwareStatus = $this->simulateHardwareCheck($approval);

            return response()->json([
                'success' => true,
                'message' => 'Hardware status retrieved',
                'data' => [
                    'approval_id' => $approval->approval_id,
                    'device_name' => $approval->device_name,
                    'user' => [
                        'user_id' => $approval->user_id,
                        'username' => $approval->user->username ?? 'N/A'
                    ],
                    'hardware_status' => $hardwareStatus,
                    'last_hardware_check' => $approval->last_hardware_check?->format('Y-m-d H:i:s') ?? 'Never',
                    'hardware_change_count' => $approval->hardware_change_count,
                    'recommendation' => $hardwareStatus['overall'] === 'HEALTHY'
                        ? 'Device hardware is healthy'
                        : 'Hardware changes detected, re-approval may be required'
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Check Hardware Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to check hardware status'
            ], 500);
        }
    }

    // ==================== HELPER METHODS ====================

    /**
     * Check if user can approve device
     */
    private function canApproveDevice($user, $queue): bool
    {
        // Superadmin can approve everything
        if ($user->Role_Id == 2) { // Assuming 2 is superadmin
            return true;
        }

        // Admin can approve devices in their department
        if ($user->Role_Id == 1) { // Assuming 1 is admin
            if ($queue->department_id && $user->department_id == $queue->department_id) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check device quota
     */
    private function checkDeviceQuota($userId): bool
    {
        $maxDevices = 3; // Configurable
        $activeDevices = GmcDeviceApproval::where('user_id', $userId)
            ->where('is_active', true)
            ->where('status', 'ACTIVE')
            ->count();

        return $activeDevices < $maxDevices;
    }

    /**
     * Get manageable departments for user
     */
    private function getManageableDepartments($userId): array
    {
        $user = GmcUser::find($userId);

        if ($user->Role_Id == 2) { // Superadmin
            return Department::pluck('department_id')->toArray();
        }

        if ($user->Role_Id == 1 && $user->department_id) { // Admin
            return [$user->department_id];
        }

        return [];
    }

    /**
     * Check if user can view another user's devices
     */
    private function canViewUserDevices($currentUser, $targetUser): bool
    {
        // Users can view their own devices
        if ($currentUser->user_id == $targetUser->user_id) {
            return true;
        }

        // Superadmin can view all
        if ($currentUser->Role_Id == 2) {
            return true;
        }

        // Admin can view devices in their department
        if ($currentUser->Role_Id == 1 && $currentUser->department_id == $targetUser->department_id) {
            return true;
        }

        return false;
    }

    /**
     * Check if user can renew device
     */
    private function canRenewDevice($user, $approval): bool
    {
        return $this->canApproveDevice($user, (object) [
            'department_id' => $approval->department_id,
            'is_citizen' => false
        ]);
    }

    /**
     * Check if user can revoke device
     */
    private function canRevokeDevice($user, $approval): bool
    {
        return $this->canApproveDevice($user, (object) [
            'department_id' => $approval->department_id,
            'is_citizen' => false
        ]);
    }

    /**
     * Check if user can view statistics
     */
    private function canViewStatistics($user): bool
    {
        return in_array($user->Role_Id, [1, 2]); // Admin and Superadmin
    }

    /**
     * Check if user can check hardware
     */
    private function canCheckHardware($user, $approval): bool
    {
        return $this->canApproveDevice($user, (object) [
            'department_id' => $approval->department_id,
            'is_citizen' => false
        ]);
    }

    /**
     * Get device policy based on role
     */
    private function getDevicePolicy($roleId): array
    {
        if ($roleId == 4) { // Citizen
            return [
                'max_devices' => 3,
                'requires_approval' => false,
                'validity_days' => 3650, // 10 years
                'auto_approval' => true,
                'device_types_allowed' => ['DESKTOP', 'LAPTOP', 'MOBILE', 'TABLET'],
                'ip_restrictions' => false
            ];
        } else { // Employee
            return [
                'max_devices' => 3,
                'requires_approval' => true,
                'validity_days' => 365, // 1 year
                'auto_approval' => false,
                'device_types_allowed' => ['DESKTOP', 'LAPTOP', 'MOBILE'],
                'ip_restrictions' => true,
                'department_whitelist' => true
            ];
        }
    }

    /**
     * Update user login information
     */
    private function updateUserLoginInfo($userId): void
    {
        $user = GmcUser::find($userId);
        if ($user && !$user->last_login) {
            $user->update(['last_login' => now()]);
        }
    }

    /**
     * Invalidate device sessions
     */
    private function invalidateDeviceSessions($userId, $deviceUniqueId): void
    {
        // This would invalidate sessions for the specific device
        // Implementation depends on your session management
        Log::info("Invalidating sessions for user $userId, device $deviceUniqueId");
    }

    /**
     * Get department distribution
     */
    private function getDepartmentDistribution(): array
    {
        return GmcDeviceApproval::selectRaw('department_id, COUNT(*) as count')
            ->where('is_active', true)
            ->where('status', 'ACTIVE')
            ->groupBy('department_id')
            ->with('department')
            ->get()
            ->map(function ($item) {
                return [
                    'department_id' => $item->department_id,
                    'department_name' => $item->department->department_name ?? 'Unknown',
                    'count' => $item->count
                ];
            })
            ->toArray();
    }

    /**
     * Get device type distribution
     */
    private function getDeviceTypeDistribution(): array
    {
        return GmcDeviceApproval::selectRaw('device_type, COUNT(*) as count')
            ->where('is_active', true)
            ->where('status', 'ACTIVE')
            ->groupBy('device_type')
            ->get()
            ->mapWithKeys(function ($item) {
                return [$item->device_type => $item->count];
            })
            ->toArray();
    }

    /**
     * Get recent activity
     */
    private function getRecentActivity($startDate): array
    {
        return GmcDeviceApprovalLog::selectRaw('action_type, DATE(performed_at) as date, COUNT(*) as count')
            ->where('performed_at', '>=', $startDate)
            ->groupBy('action_type', 'date')
            ->orderBy('date', 'desc')
            ->orderBy('action_type')
            ->get()
            ->groupBy('date')
            ->map(function ($items) {
                return $items->mapWithKeys(function ($item) {
                    return [$item->action_type => $item->count];
                });
            })
            ->toArray();
    }

    /**
     * Simulate hardware check
     */
    private function simulateHardwareCheck($approval): array
    {
        // Simulate hardware check results
        return [
            'overall' => 'HEALTHY',
            'components' => [
                'mac_address' => [
                    'status' => 'VALID',
                    'value' => $approval->mac_address,
                    'last_change' => null
                ],
                'hdd_serial' => [
                    'status' => 'VALID',
                    'value' => substr($approval->hdd_serial ?? '', 0, 10) . '...',
                    'last_change' => $approval->last_hardware_change?->format('Y-m-d H:i:s')
                ],
                'browser_fingerprint' => [
                    'status' => 'VALID',
                    'value' => substr($approval->browser_fingerprint, 0, 10) . '...',
                    'last_change' => null
                ]
            ],
            'last_check' => now()->format('Y-m-d H:i:s'),
            'requires_reapproval' => false
        ];
    }

    /**
     * Send approval notification
     */
    private function sendApprovalNotification($user, $approval): void
    {
        // Implementation for sending email/SMS notification
        Log::info("Approval notification sent to user {$user->user_id} for device {$approval->device_name}");
    }

    /**
     * Send rejection notification
     */
    private function sendRejectionNotification($user, $reason, $instructions = null): void
    {
        // Implementation for sending email/SMS notification
        Log::info("Rejection notification sent to user {$user->user_id}: $reason");
    }

    /**
     * Send renewal notification
     */
    private function sendRenewalNotification($user, $approval, $validityDays): void
    {
        // Implementation for sending email/SMS notification
        Log::info("Renewal notification sent to user {$user->user_id} for {$validityDays} days");
    }

    /**
     * Send revocation notification
     */
    private function sendRevocationNotification($user, $approval, $reason): void
    {
        // Implementation for sending email/SMS notification
        Log::info("Revocation notification sent to user {$user->user_id}: $reason");
    }
}
