<?php

namespace App\Http\Controllers;

use App\Models\DeviceApprovalLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\RoleMaster;

class DeviceApprovalLogController extends Controller
{
    /**
     * List all logs
     */
public function index()
{
    $logs = DeviceApprovalLog::with(['user', 'user.department'])
        ->orderBy('log_id', 'desc')
        ->get();

    foreach ($logs as $log) {

        if ($log->user && $log->user->Role_Id) {

            $role = RoleMaster::where(
                'role_id',
                $log->user->Role_Id
            )->first();

            $log->user->setRelation('role', $role);

        } else {
            $log->user?->setRelation('role', null);
        }
    }

    return response()->json($logs, 200);
}

    /**
     * Store new log
     */
public function store(Request $request)
{
    $validator = Validator::make($request->all(), [
        'queue_id' => 'nullable|integer',
        'approval_id' => 'nullable|integer',
        'user_id' => 'nullable|exists:gmc_users,user_id',
        'action_type' => 'nullable|string',
        'action_details' => 'nullable|string',
        'device_data' => 'nullable|array',
        'performed_by' => 'nullable|string',
        'ip_address' => 'nullable|string',
        'user_agent' => 'nullable|string',
        'performed_at' => 'nullable|date',
    ]);

    if ($validator->fails()) {
        return response()->json([
            'message' => 'Validation error',
            'errors' => $validator->errors()
        ], 422);
    }

    $log = DeviceApprovalLog::create($request->only([
        'queue_id',
        'approval_id',
        'user_id',
        'action_type',
        'action_details',
        'device_data',
        'performed_by',
        'ip_address',
        'user_agent',
        'performed_at'
    ]));

    return response()->json([
        'message' => 'Device approval log created successfully',
        'data' => $log->load('user')
    ], 201);
}


    /**
     * Show single log
     */
public function show($id)
{
    $log = DeviceApprovalLog::with('user')->find($id);

    if (!$log) {
        return response()->json(['message' => 'Log not found'], 404);
    }

    if ($log->user && $log->user->Role_Id) {

        $role = RoleMaster::where(
            'role_id',
            $log->user->Role_Id
        )->first();

        // ✅ CORRECT
        $log->user->setRelation('role', $role);
    } else {
        $log->user?->setRelation('role', null);
    }

    return response()->json($log, 200);
}

    /**
     * Update log
     */
public function update(Request $request, $id)
{
    $log = DeviceApprovalLog::find($id);

    if (!$log) {
        return response()->json(['message' => 'Log not found'], 404);
    }

    $validator = Validator::make($request->all(), [
        'queue_id' => 'nullable|integer',
        'approval_id' => 'nullable|integer',
        'user_id' => 'nullable|exists:gmc_users,user_id',
        'action_type' => 'nullable|string',
        'action_details' => 'nullable|string',
        'device_data' => 'nullable|array',
        'performed_by' => 'nullable|string',
        'ip_address' => 'nullable|string',
        'user_agent' => 'nullable|string',
        'performed_at' => 'nullable|date',
    ]);

    if ($validator->fails()) {
        return response()->json([
            'message' => 'Validation error',
            'errors' => $validator->errors()
        ], 422);
    }

    $log->update($request->only([
        'queue_id',
        'approval_id',
        'user_id',
        'action_type',
        'action_details',
        'device_data',
        'performed_by',
        'ip_address',
        'user_agent',
        'performed_at'
    ]));

    return response()->json([
        'message' => 'Device approval log updated successfully',
        'data' => $log->load('user')
    ], 200);
}


    /**
     * Delete log
     */
    public function destroy($id)
    {
        $log = DeviceApprovalLog::find($id);

        if (!$log) {
            return response()->json(['message' => 'Log not found'], 404);
        }

        $log->delete();

        return response()->json([
            'message' => 'Device approval log deleted successfully'
        ], 200);
    }
}
