<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\DocumentLibrary;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class DocumentLibraryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = DocumentLibrary::with('functionType');

            // Search functionality
            if ($request->has('search')) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('Rec_Id', 'LIKE', "%{$search}%")
                      ->orWhere('Description', 'LIKE', "%{$search}%")
                      ->orWhere('Serial_No', 'LIKE', "%{$search}%");
                });
            }

            // Filter by Menu_Id
            if ($request->has('menu_id')) {
                $query->where('Menu_Id', $request->menu_id);
            }

            // Order by
            $query->orderBy('Serial_No', 'asc');

            $perPage = $request->get('per_page', 15);
            $documents = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $documents,
                'message' => 'Documents retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving documents.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */


public function store(Request $request)
{
    try {
        $validator = Validator::make($request->all(), [
            'Menu_Id' => 'required|exists:function_types,id',
            'Description' => 'required|string',
            'Doc_Upload' => 'nullable|array',
            'Doc_Upload.*' => 'file|max:204800', 
            'Serial_No' => 'nullable|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $request->only(['Menu_Id', 'Description', 'Serial_No']);

        // Handle file uploads
        if ($request->hasFile('Doc_Upload')) {
            $uploadedFiles = [];
            foreach ($request->file('Doc_Upload') as $index => $file) {
                // Create safe filename
                $originalName = $file->getClientOriginalName();
                $filename = time() . '_' . $index . '_' . Str::slug(pathinfo($originalName, PATHINFO_FILENAME))
                          . '.' . $file->getClientOriginalExtension();

                // Create folder structure: documents/2026/01/25/
                $year = date('Y');
                $month = date('m');
                $menuId = $request->Menu_Id;
                $folderPath = "documents/{$year}/{$month}/{$menuId}/";

                // Store in storage/app/public/documents/2026/01/25/
                $storagePath = $file->storeAs($folderPath, $filename, 'public');

                // Public accessible path (after storage:link)
                $publicPath = "storage/{$folderPath}{$filename}";

                // Store file info
                $uploadedFiles[] = [
                    'filename' => $filename,
                    'original_name' => $originalName,
                    'storage_path' => $storagePath, // storage/app/public/documents/2026/01/25/filename
                    'public_path' => $publicPath,   // storage/documents/2026/01/25/filename (accessible via URL)
                    'url' => asset($publicPath),    // Full URL: http://domain.com/storage/documents/2026/01/25/filename
                    'size' => $file->getSize(),
                    'mime_type' => $file->getMimeType(),
                    'uploaded_at' => now()->toDateTimeString()
                ];
            }
            $data['Doc_Upload'] = $uploadedFiles;
        }

        $document = DocumentLibrary::create($data);

        return response()->json([
            'success' => true,
            'data' => $document,
            'message' => 'Document created successfully.'
        ], 201);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error creating document.',
            'error' => $e->getMessage()
        ], 500);
    }
}


    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try {
            $document = DocumentLibrary::with('functionType', 'tenderEntries')
                ->findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => $document,
                'message' => 'Document retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Document not found.'
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $document = DocumentLibrary::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'Menu_Id' => 'sometimes|exists:function_types,id',
                'Description' => 'sometimes|string',
                'Doc_Upload' => 'nullable|array',
                'Doc_Upload.*' => 'sometimes|file|max:204800',
                'Serial_No' => 'nullable|integer'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->only(['Menu_Id', 'Description', 'Serial_No']);

            // Handle file updates
            if ($request->hasFile('Doc_Upload')) {
                // Delete old files
                if ($document->Doc_Upload) {
                    foreach ($document->Doc_Upload as $oldFile) {
                        Storage::disk('public')->delete($oldFile['path']);
                    }
                }

                // Upload new files
                $uploadedFiles = [];
                foreach ($request->file('Doc_Upload') as $file) {
                    $filename = Str::uuid() . '.' . $file->getClientOriginalExtension();
                    $path = $file->storeAs('documents', $filename, 'public');

                    $uploadedFiles[] = [
                        'filename' => $filename,
                        'original_name' => $file->getClientOriginalName(),
                        'path' => $path,
                        'size' => $file->getSize(),
                        'mime_type' => $file->getMimeType(),
                        'uploaded_at' => now()->toDateTimeString()
                    ];
                }
                $data['Doc_Upload'] = $uploadedFiles;
            }

            $document->update($data);

            return response()->json([
                'success' => true,
                'data' => $document,
                'message' => 'Document updated successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating document.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $document = DocumentLibrary::findOrFail($id);

            // Delete associated files
            if ($document->Doc_Upload) {
                foreach ($document->Doc_Upload as $file) {
                    Storage::disk('public')->delete($file['path']);
                }
            }

            $document->delete();

            return response()->json([
                'success' => true,
                'message' => 'Document deleted successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting document.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download document file
     */
    public function downloadFile($id, $fileIndex)
    {
        try {
            $document = DocumentLibrary::findOrFail($id);

            if (!$document->Doc_Upload || !isset($document->Doc_Upload[$fileIndex])) {
                return response()->json([
                    'success' => false,
                    'message' => 'File not found.'
                ], 404);
            }

            $file = $document->Doc_Upload[$fileIndex];
            $path = storage_path('app/public/' . $file['path']);

            return response()->download($path, $file['original_name']);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error downloading file.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
