<?php

namespace App\Http\Controllers;

use App\Models\EducationMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class EducationMasterController extends Controller
{
    // 🔹 GET ALL
    public function index()
    {
        try {
            $data = EducationMaster::orderBy('education_id', 'desc')->get();

            return response()->json([
                'success' => true,
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch records',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'qual_code' => 'nullable|string|max:20|unique:education_master,qual_code',
                'qual_name' => 'required|string|max:100',
                'is_mandatory' => 'required|in:Y,N',
                'is_technical' => 'required|in:Y,N',
                'is_professional' => 'required|in:Y,N',
                'valid_for_recruitment' => 'required|in:Y,N'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $education = EducationMaster::create([
                'qual_code' => $request->qual_code,
                'qual_name' => $request->qual_name,
                'is_active' => $request->is_active ?? true,
                'is_mandatory' => $request->is_mandatory,
                'is_technical' => $request->is_technical,
                'is_professional' => $request->is_professional,
                'valid_for_recruitment' => $request->valid_for_recruitment
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Education record created successfully',
                'data' => $education
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Create failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $education = EducationMaster::find($id);

            if (!$education) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $education
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE (sometimes + update)
    public function update(Request $request, $id)
    {
        try {
            $education = EducationMaster::find($id);

            if (!$education) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'qual_code' => 'sometimes|required|string|max:20|unique:education_master,qual_code,' . $id . ',education_id',
                'qual_name' => 'sometimes|required|string|max:100',
                'is_active' => 'sometimes|boolean',
                'is_mandatory' => 'sometimes|required|in:Y,N',
                'is_technical' => 'sometimes|required|in:Y,N',
                'is_professional' => 'sometimes|required|in:Y,N',
                'valid_for_recruitment' => 'sometimes|required|in:Y,N'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $education->update($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Education record updated successfully',
                'data' => $education
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Update failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $education = EducationMaster::find($id);

            if (!$education) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $education->delete();

            return response()->json([
                'success' => true,
                'message' => 'Education record deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Delete failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
