<?php

namespace App\Http\Controllers;

use App\Models\EmployeeGradeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class EmployeeGradeMasterController extends Controller
{
    // 🔹 GET ALL
    public function index()
    {
        try {
            $data = EmployeeGradeMaster::orderBy('grade_id', 'desc')->get();

            return response()->json([
                'success' => true,
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch records',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'grade_code' => 'nullable|string|max:10|unique:employee_grade_master,grade_code',
                'grade_name' => 'required|string|max:50',
                'pay_band' => 'nullable|string|max:50',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $grade = EmployeeGradeMaster::create([
                'grade_code' => $request->grade_code,
                'grade_name' => $request->grade_name,
                'pay_band' => $request->pay_band,
                'is_active' => true
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Employee Grade created successfully',
                'data' => $grade
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Create failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $grade = EmployeeGradeMaster::find($id);

            if (!$grade) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $grade
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE (sometimes + update)
    public function update(Request $request, $id)
    {
        try {
            $grade = EmployeeGradeMaster::find($id);

            if (!$grade) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'grade_code' => 'sometimes|required|string|max:10|unique:employee_grade_master,grade_code,' . $id . ',grade_id',
                'grade_name' => 'sometimes|required|string|max:50',
                'pay_band' => 'sometimes|nullable|string|max:50',
                'is_active' => 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            // ✅ correct usage
            $grade->update($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Employee Grade updated successfully',
                'data' => $grade
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Update failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $grade = EmployeeGradeMaster::find($id);

            if (!$grade) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $grade->delete();

            return response()->json([
                'success' => true,
                'message' => 'Employee Grade deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Delete failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
