<?php

namespace App\Http\Controllers;

use App\Models\EmployeeQuotaMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class EmployeeQuotaMasterController extends Controller
{
    // 🔹 GET ALL
    public function index()
    {
        try {
            $data = EmployeeQuotaMaster::orderBy('quota_id', 'desc')->get();

            return response()->json([
                'success' => true,
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch records',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'quota_code' => 'nullable|string|max:20|unique:employee_quota_master,quota_code',
                'quota_name' => 'required|string|max:100',
                // percentage removed
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $quota = EmployeeQuotaMaster::create([
                'quota_code' => $request->quota_code,
                'quota_name' => $request->quota_name,
                // percentage removed
                'is_active' => true
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Employee Quota created successfully',
                'data' => $quota
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Create failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $quota = EmployeeQuotaMaster::find($id);

            if (!$quota) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $quota
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE (sometimes + update)
    public function update(Request $request, $id)
    {
        try {
            $quota = EmployeeQuotaMaster::find($id);

            if (!$quota) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'quota_code' => 'sometimes|required|string|max:20|unique:employee_quota_master,quota_code,' . $id . ',quota_id',
                'quota_name' => 'sometimes|required|string|max:100',
                // percentage removed
                'is_active' => 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            // ✅ correct update usage
            $quota->update($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Employee Quota updated successfully',
                'data' => $quota
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Update failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $quota = EmployeeQuotaMaster::find($id);

            if (!$quota) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $quota->delete();

            return response()->json([
                'success' => true,
                'message' => 'Employee Quota deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Delete failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
