<?php

namespace App\Http\Controllers;

use App\Models\EmployeeTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class EmployeeTypeMasterController extends Controller
{
    // 🔹 GET ALL
    public function index()
    {
        try {
            $data = EmployeeTypeMaster::orderBy('emp_type_id', 'desc')->get();

            return response()->json([
                'success' => true,
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch records',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'type_code' => 'nullable|string|max:20|unique:employee_type_master,type_code',
                'type_name' => 'required|string|max:100',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $type = EmployeeTypeMaster::create([
                'type_code' => $request->type_code,
                'type_name' => $request->type_name,
                'is_active' => true
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Employee Type created successfully',
                'data' => $type
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Create failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $type = EmployeeTypeMaster::find($id);

            if (!$type) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $type
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE (sometimes + update)
    public function update(Request $request, $id)
    {
        try {
            $type = EmployeeTypeMaster::find($id);

            if (!$type) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'type_code' => 'sometimes|required|string|max:20|unique:employee_type_master,type_code,' . $id . ',emp_type_id',
                'type_name' => 'sometimes|required|string|max:100',
                'is_active' => 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $type->update($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Employee Type updated successfully',
                'data' => $type
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Update failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $type = EmployeeTypeMaster::find($id);

            if (!$type) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $type->delete();

            return response()->json([
                'success' => true,
                'message' => 'Employee Type deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Delete failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
