<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\FasFundsTypeMaster;
use App\Models\GrantCategoryMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FasFundsTypeMasterController extends Controller
{
    public function index()
    {
        $fundsTypes = FasFundsTypeMaster::with('grantCategory')
            ->where('is_active', true)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $fundsTypes
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'funds_code' => 'nullable|string|max:20|unique:fas_funds_type_master,funds_code',
            'funds_name' => 'required|string|max:100',
            'fund_type' => 'nullable|string|max:50',  // Added validation
            'source_of_fund' => 'nullable|string|max:50',  // Added validation
            'grant_category_id' => 'nullable|exists:grant_category_masters,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        // Validate grant_category_id exists if provided
        if ($request->has('grant_category_id') && $request->grant_category_id) {
            $grantCategory = GrantCategoryMaster::find($request->grant_category_id);
            if (!$grantCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Grant category not found'
                ], 404);
            }
        }

        $fundsType = FasFundsTypeMaster::create([
            'funds_code' => $request->funds_code,
            'funds_name' => $request->funds_name,
            'fund_type' => $request->fund_type,           // Added
            'source_of_fund' => $request->source_of_fund, // Added
            'grant_category_id' => $request->grant_category_id,
            'is_active' => $request->has('is_active') ? $request->is_active : true
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Funds type created successfully',
            'data' => $fundsType->load('grantCategory')
        ], 201);
    }

    public function show($id)
    {
        $fundsType = FasFundsTypeMaster::with('grantCategory')->find($id);

        if (!$fundsType) {
            return response()->json([
                'success' => false,
                'message' => 'Funds type not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $fundsType
        ]);
    }

    public function update(Request $request, $id)
    {
        $fundsType = FasFundsTypeMaster::find($id);

        if (!$fundsType) {
            return response()->json([
                'success' => false,
                'message' => 'Funds type not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'funds_code' => 'nullable|string|max:20|unique:fas_funds_type_master,funds_code,' . $id . ',funds_type_id',
            'funds_name' => 'required|string|max:100',
            'fund_type' => 'nullable|string|max:50',           // Added validation
            'source_of_fund' => 'nullable|string|max:50',      // Added validation
            'grant_category_id' => 'nullable|exists:grant_category_masters,id',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        // Validate grant_category_id exists if provided
        if ($request->has('grant_category_id') && $request->grant_category_id) {
            $grantCategory = GrantCategoryMaster::find($request->grant_category_id);
            if (!$grantCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Grant category not found'
                ], 404);
            }
        }

        $fundsType->update([
            'funds_code' => $request->funds_code,
            'funds_name' => $request->funds_name,
            'fund_type' => $request->fund_type,           // Added
            'source_of_fund' => $request->source_of_fund, // Added
            'grant_category_id' => $request->grant_category_id,
            'is_active' => $request->has('is_active') ? $request->is_active : $fundsType->is_active
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Funds type updated successfully',
            'data' => $fundsType->load('grantCategory')
        ]);
    }

    public function destroy($id)
    {
        $fundsType = FasFundsTypeMaster::find($id);

        if (!$fundsType) {
            return response()->json([
                'success' => false,
                'message' => 'Funds type not found'
            ], 404);
        }

        $fundsType->update(['is_active' => false]);

        return response()->json([
            'success' => true,
            'message' => 'Funds type deactivated successfully'
        ]);
    }

    public function restore($id)
    {
        $fundsType = FasFundsTypeMaster::find($id);

        if (!$fundsType) {
            return response()->json([
                'success' => false,
                'message' => 'Funds type not found'
            ], 404);
        }

        $fundsType->update(['is_active' => true]);

        return response()->json([
            'success' => true,
            'message' => 'Funds type activated successfully',
            'data' => $fundsType->load('grantCategory')
        ]);
    }

    /**
     * Get funds types by grant category
     */
    public function getByGrantCategory($grantCategoryId)
    {
        $grantCategory = GrantCategoryMaster::find($grantCategoryId);

        if (!$grantCategory) {
            return response()->json([
                'success' => false,
                'message' => 'Grant category not found'
            ], 404);
        }

        $fundsTypes = FasFundsTypeMaster::with('grantCategory')
            ->where('grant_category_id', $grantCategoryId)
            ->where('is_active', true)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $fundsTypes,
            'grant_category' => $grantCategory
        ]);
    }

    /**
     * Get all funds types (including inactive)
     */
    public function getAll()
    {
        $fundsTypes = FasFundsTypeMaster::with('grantCategory')->get();

        return response()->json([
            'success' => true,
            'data' => $fundsTypes
        ]);
    }
}
