<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\FasGrantTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FasGrantTypeMasterController extends Controller
{
    public function index()
    {
        $grantTypes = FasGrantTypeMaster::with('grantAuthority')
            ->where('is_active', true)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $grantTypes
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'grant_code' => 'nullable|string|max:25|unique:fas_grant_type_master,grant_code',
            'grant_name' => 'required|string|max:100',
            'gt_nature' => 'nullable|string|max:25|in:Revenue,Capital',
            'gt_condition' => 'nullable|string|in:Y,N',
            'gt_util_from_date' => 'nullable|date|before_or_equal:gt_util_to_date',
            'gt_util_to_date' => 'nullable|date|after_or_equal:gt_util_from_date',
            'gt_act_mode' => 'nullable|string|max:50|in:Income,Liability,Differed',
            'gt_status' => 'nullable|string|max:25',
            'gr_au_id' => 'nullable|exists:gr_authority_masters,gr_au_id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $grantType = FasGrantTypeMaster::create(array_merge(
            $request->all(),
            ['is_active' => true]
        ));

        // ✅ UPDATED: Load grantAuthority relationship
        $grantType->load('grantAuthority');

        return response()->json([
            'success' => true,
            'message' => 'Grant type created successfully',
            'data' => $grantType // Now includes grantAuthority data
        ], 201);
    }

    public function show($id)
    {
        $grantType = FasGrantTypeMaster::with('grantAuthority')->find($id);

        if (!$grantType) {
            return response()->json([
                'success' => false,
                'message' => 'Grant type not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $grantType
        ]);
    }

    public function update(Request $request, $id)
    {
        $grantType = FasGrantTypeMaster::find($id);

        if (!$grantType) {
            return response()->json([
                'success' => false,
                'message' => 'Grant type not found'
            ], 404);
        }

        // Manual validation for unique grant_code
        if ($request->has('grant_code') && $request->grant_code !== $grantType->grant_code) {
            $exists = FasGrantTypeMaster::where('grant_code', $request->grant_code)
                ->where('grant_type_id', '!=', $id)
                ->exists();

            if ($exists) {
                return response()->json([
                    'success' => false,
                    'errors' => ['grant_code' => ['The grant code has already been taken.']]
                ], 422);
            }
        }

        $validator = Validator::make($request->all(), [
            'grant_code' => 'required|string|max:25',
            'grant_name' => 'required|string|max:100',
            'gt_nature' => 'nullable|string|max:25|in:Revenue,Capital',
            'gt_condition' => 'nullable|string|in:Y,N',
            'gt_util_from_date' => 'nullable|date|before_or_equal:gt_util_to_date',
            'gt_util_to_date' => 'nullable|date|after_or_equal:gt_util_from_date',
            'gt_act_mode' => 'nullable|string|max:50|in:Income,Liability,Differed',
            'gt_status' => 'nullable|string|max:25',
            'gr_au_id' => 'nullable|exists:gr_authority_masters,gr_au_id',
            'is_active' => 'nullable|boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $grantType->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Grant type updated successfully',
            'data' => $grantType
        ]);
    }

    public function destroy($id)
    {
        $grantType = FasGrantTypeMaster::find($id);

        if (!$grantType) {
            return response()->json([
                'success' => false,
                'message' => 'Grant type not found'
            ], 404);
        }

        $grantType->update(['is_active' => false]);

        return response()->json([
            'success' => true,
            'message' => 'Grant type deactivated successfully'
        ]);
    }

    public function restore($id)
    {
        $grantType = FasGrantTypeMaster::find($id);

        if (!$grantType) {
            return response()->json([
                'success' => false,
                'message' => 'Grant type not found'
            ], 404);
        }

        $grantType->update(['is_active' => true]);

        return response()->json([
            'success' => true,
            'message' => 'Grant type activated successfully',
            'data' => $grantType
        ]);
    }
}
