<?php

namespace App\Http\Controllers;

use App\Models\FasLoanDocumentMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class FasLoanDocumentMasterController extends Controller
{
    // 🔹 GET ALL
    public function index()
    {
        try {
            $data = FasLoanDocumentMaster::with('loanType')->orderBy('doc_id', 'desc')->get();
            return response()->json(['success' => true, 'data' => $data], 200);
        } catch (Throwable $e) {
            return response()->json(['success' => false, 'message' => 'Failed to fetch records', 'error' => $e->getMessage()], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'doc_code' => 'nullable|string|max:20|unique:fas_loan_document_master,doc_code',
                'doc_name' => 'required|string|max:100',
                'loan_type_id' => 'nullable|exists:fas_loan_type_master,loan_type_id',
                'is_mandatory' => 'sometimes|boolean',
                'is_active' => 'sometimes|boolean'
            ]);

            if ($validator->fails())
                return response()->json(['success' => false, 'errors' => $validator->errors()], 422);

            $document = FasLoanDocumentMaster::create($validator->validated());

            return response()->json(['success' => true, 'message' => 'Loan Document created successfully', 'data' => $document], 201);
        } catch (Throwable $e) {
            return response()->json(['success' => false, 'message' => 'Create failed', 'error' => $e->getMessage()], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $document = FasLoanDocumentMaster::with('loanType')->find($id);

            if (!$document)
                return response()->json(['success' => false, 'message' => 'Record not found'], 404);

            return response()->json(['success' => true, 'data' => $document], 200);
        } catch (Throwable $e) {
            return response()->json(['success' => false, 'error' => $e->getMessage()], 500);
        }
    }

    // 🔹 UPDATE
    public function update(Request $request, $id)
    {
        try {
            $document = FasLoanDocumentMaster::find($id);

            if (!$document)
                return response()->json(['success' => false, 'message' => 'Record not found'], 404);

            $validator = Validator::make($request->all(), [
                'doc_code' => 'sometimes|required|string|max:20|unique:fas_loan_document_master,doc_code,' . $id . ',doc_id',
                'doc_name' => 'sometimes|required|string|max:100',
                'loan_type_id' => 'sometimes|nullable|exists:fas_loan_type_master,loan_type_id',
                'is_mandatory' => 'sometimes|boolean',
                'is_active' => 'sometimes|boolean'
            ]);

            if ($validator->fails())
                return response()->json(['success' => false, 'errors' => $validator->errors()], 422);

            $document->update($validator->validated());

            return response()->json(['success' => true, 'message' => 'Loan Document updated successfully', 'data' => $document], 200);
        } catch (Throwable $e) {
            return response()->json(['success' => false, 'message' => 'Update failed', 'error' => $e->getMessage()], 500);
        }
    }

    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $document = FasLoanDocumentMaster::find($id);

            if (!$document)
                return response()->json(['success' => false, 'message' => 'Record not found'], 404);

            $document->delete();

            return response()->json(['success' => true, 'message' => 'Loan Document deleted successfully'], 200);
        } catch (Throwable $e) {
            return response()->json(['success' => false, 'message' => 'Delete failed', 'error' => $e->getMessage()], 500);
        }
    }
}
