<?php

namespace App\Http\Controllers;

use App\Models\FinanTrn;
use App\Models\LedgerMaster;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class FinanTrnController extends Controller
{
    public function index(Request $request)
    {
        $query = FinanTrn::with(['organisation', 'ledger', 'refLedger', 'agent'])
            ->orderBy('vou_dt', 'asc');

        if ($request->has('vou_rf') && !empty($request->vou_rf)) {
            $query->where('vou_rf', 'LIKE', '%' . $request->vou_rf . '%');
        }

        $finanTrns = $query->get();

        return response()->json($finanTrns, 200);
    }
    public function show($id)
    {
        $finanTrn = FinanTrn::with(['organisation', 'ledger', 'refLedger', 'agent'])
            ->findOrFail($id);

        return response()->json($finanTrn, 200);
    }

    public function findByVouRf(Request $request)
    {
        $request->validate([
            'vou_rf' => 'required|string'
        ]);

        $finanTrn = FinanTrn::with(['organisation', 'ledger',  'agent'])
            ->where('vou_rf', $request->vou_rf)
            ->firstOrFail();

        return response()->json($finanTrn, 200);
    }
    public function store(Request $request)
    {
        // Validation rules based on Excel schema
        $validator = Validator::make($request->all(), [
            'Org_Id' => 'required|integer|exists:organisation_settings,Org_Id',
            'Lg_Id' => 'required|integer|exists:ledger_master,Lg_Id',
            'vou_ty' => 'nullable|string|max:3',
            'vou_id' => 'nullable|integer',
            'vou_sr' => 'nullable|integer|min:1',
            'vou_ln' => 'nullable|integer|min:1',
            'vou_rf' => 'nullable|string|max:150',
            'vou_dt' => 'nullable|date',
            'inq_id' => 'nullable|numeric',
            'ag_id' => 'nullable|integer|exists:subledg_master,SL_Id',
            'inst_no' => 'nullable|integer',
            'agt_amt' => 'nullable|numeric',
            'cr_amt' => 'nullable|numeric|between:0,9999999999.99',
            'dr_amt' => 'nullable|numeric|between:0,9999999999.99',
            'ref_id' => 'nullable|numeric',
            'chq_no' => 'nullable|string|max:50',
            'chq_dt' => 'nullable|date',
            'clr_dt' => 'nullable|date',
            'bk_id' => 'nullable|integer',
            'lot_no' => 'nullable|numeric',
            'rmk1' => 'nullable|string|max:150',
            'rmk2' => 'nullable|string|max:150',
            'rmk3' => 'nullable|string|max:150',
            'rmk4' => 'nullable|string|max:150',
            'rmk5' => 'nullable|string|max:150',
            'chq_mode' => 'nullable|string|max:1',
            'r_mode' => 'nullable|string|max:2',
            'sms_stus' => 'nullable|string|max:1',
            'email_stus' => 'nullable|string|max:1',
            'sm_id' => 'nullable|integer',
            'an_id' => 'nullable|string|max:40',
            'bfr1' => 'nullable|string|max:50',
            'bfr2' => 'nullable|string|max:50',
            'op_id' => 'nullable|integer',
            'dt_tm' => 'nullable|string|max:10',
            'pay_mode' => 'nullable|string|max:5',
            'exp_ref' => 'nullable|integer',
        ]);

        // Custom validation for cr_amt or dr_amt
        $validator->after(function ($validator) use ($request) {
            if (!$request->has('cr_amt') && !$request->has('dr_amt')) {
                $validator->errors()->add('cr_amt', 'Either cr_amt or dr_amt must be provided.');
                $validator->errors()->add('dr_amt', 'Either cr_amt or dr_amt must be provided.');
            }
            if ($request->has('cr_amt') && $request->has('dr_amt') && $request->cr_amt !== null && $request->dr_amt !== null) {
                $validator->errors()->add('cr_amt', 'Only one of cr_amt or dr_amt can be provided.');
                $validator->errors()->add('dr_amt', 'Only one of cr_amt or dr_amt can be provided.');
            }
        });

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Determine transaction type and set vou_ty
        $isCredit = $request->has('cr_amt') && $request->cr_amt !== null;
        $data = $request->all();
        // $data['vou_ty'] = $isCredit ? 'PAY' : 'RCP';

        // Set fallbacks for nullable fields
        $data['vou_id'] = isset($data['vou_id']) ? $data['vou_id'] : (FinanTrn::max('vou_id') + 1 ?? 1);
        $data['vou_sr'] = isset($data['vou_sr']) ? $data['vou_sr'] : 1;
        $data['vou_ln'] = isset($data['vou_ln']) ? $data['vou_ln'] : 1;
        $data['v_mode'] = 'G'; // Hardcode v_mode to 'G'


        if (!isset($data['ref_id']) && isset($data['Lg_Id'])) {
            $data['ref_id'] = $data['Lg_Id'];
        }


        $data['vou_rf'] = $request->input('vou_rf');

        // First entry
        $finanTrn = FinanTrn::create($data);

        // Second entry (same vou_ty, opposite amount, Lg_Id = null)
        $secondEntry = $data;
        $secondEntry['Lg_Id'] = null; // Blank for second entry
        $secondEntry['vou_ty'];
        $secondEntry['vou_ln'] = $data['vou_ln'] + 1;
        $secondEntry['vou_sr'] = $data['vou_sr'] + 1;
        $secondEntry['cr_amt'] = $isCredit ? null : $data['dr_amt'];
        $secondEntry['dr_amt'] = $isCredit ? $data['cr_amt'] : null;

        $secondEntry['vou_rf'] = $data['vou_rf'];

        $secondEntry['ref_id'] = $data['Lg_Id'];

        FinanTrn::create($secondEntry);

        return response()->json($finanTrn->load(['organisation', 'ledger', 'agent']), 201);
    }

public function storeSingle(Request $request)
{
    DB::beginTransaction();

    try {

        $validator = Validator::make($request->all(), [
            'Org_Id'  => 'required|exists:organisation_settings,Org_Id',
            'Lg_Id'   => 'required|exists:ledger_master,Lg_Id',
            'ag_id' => 'nullable|integer|exists:subledg_master,SL_Id',
            'vou_dt' => 'required|date',
            'vou_ty' => 'required|string|max:10',
            'vou_rf' => 'nullable|string|max:50',
            'dr_amt' => 'nullable|numeric|min:0',
            'cr_amt' => 'nullable|numeric|min:0',
            'v_mode' => 'nullable|string|max:1', // ✅ NEW (no effect)
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => $validator->errors()->first()
            ], 422);
        }

        $data = $request->all();

        // Defaults (existing logic safe)
        $data['vou_id'] = $data['vou_id'] ?? (FinanTrn::max('vou_id') + 1 ?? 1);
        $data['vou_sr'] = $data['vou_sr'] ?? 1;
        $data['vou_ln'] = $data['vou_ln'] ?? 1;

        // ✅ v_mode from request (default G)
        $data['v_mode'] = $request->input('v_mode', 'G');

        // --------------------
        // First Entry
        // --------------------
        $firstEntry = FinanTrn::create($data);

        // --------------------
        // Second (Auto) Entry
        // --------------------
        // $secondEntryData = [
        //     'Org_Id' => $data['Org_Id'],
        //     'Lg_Id'  => null,
        //     'vou_dt' => $data['vou_dt'],
        //     'vou_id' => $data['vou_id'],
        //     'vou_sr' => $data['vou_sr'] + 1,
        //     'vou_ln' => $data['vou_ln'] + 1,
        //     'vou_ty' => $data['vou_ty'],
        //     'vou_rf' => $data['vou_rf'] ?? null,
        //     'dr_amt' => $data['cr_amt'] ?? 0,
        //     'cr_amt' => $data['dr_amt'] ?? 0,
        //     'v_mode' => $data['v_mode'], // ✅ SAME MODE
        //     'ref_id' => $data['Lg_Id'],
        // ];

        // FinanTrn::create($secondEntryData);

        DB::commit();

        return response()->json([
            'status' => 'success',
            'data'   => $firstEntry
        ], 201);

    } catch (Exception $e) {
        DB::rollBack();
        return response()->json([
            'status'  => 'error',
            'message' => $e->getMessage()
        ], 500);
    }
}


  public function update(Request $request, $id)
{
    DB::beginTransaction();

    try {

        $finanTrn = FinanTrn::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Org_Id'  => 'required|exists:organisation_settings,Org_Id',
            'Lg_Id'   => 'required|exists:ledger_master,Lg_Id',
            'ag_id' => 'nullable|integer|exists:subledg_master,SL_Id',
            'vou_dt' => 'required|date',
            'vou_ty' => 'required|string|max:10',
            'vou_rf' => 'nullable|string|max:50',
            'dr_amt' => 'nullable|numeric|min:0',
            'cr_amt' => 'nullable|numeric|min:0',
            'v_mode' => 'nullable|string|max:1', // ✅ NEW
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status'  => 'error',
                'message' => $validator->errors()->first()
            ], 422);
        }

        $data = $request->all();

        // Keep existing values safe
        $data['vou_id'] = $data['vou_id'] ?? $finanTrn->vou_id;
        $data['vou_sr'] = $data['vou_sr'] ?? $finanTrn->vou_sr;
        $data['vou_ln'] = $data['vou_ln'] ?? $finanTrn->vou_ln;

        // ✅ v_mode from request OR existing OR default
        $data['v_mode'] = $request->input('v_mode', $finanTrn->v_mode ?? 'G');

        // --------------------
        // Update First Entry
        // --------------------
        $finanTrn->update($data);

        // --------------------
        // Update Second Entry
        // --------------------
        // FinanTrn::where('vou_id', $finanTrn->vou_id)
        //     ->where('vou_ln', $finanTrn->vou_ln + 1)
        //     ->update([
        //         'dr_amt' => $data['cr_amt'] ?? 0,
        //         'cr_amt' => $data['dr_amt'] ?? 0,
        //         'v_mode' => $data['v_mode'], // ✅ SAME MODE
        //     ]);

        DB::commit();

        return response()->json([
            'status' => 'success',
            'data'   => $finanTrn
        ], 200);

    } catch (Exception $e) {
        DB::rollBack();
        return response()->json([
            'status'  => 'error',
            'message' => $e->getMessage()
        ], 500);
    }
}


    public function destroy($id)
    {
        $finanTrn = FinanTrn::findOrFail($id);
        // Delete the second entry if it exists
        FinanTrn::where('vou_id', $finanTrn->vou_id)
            ->where('vou_ln', $finanTrn->vou_ln + 1)
            ->delete();
        $finanTrn->delete();
        return response()->json(null, 204);
    }

    public function getFinancialSummary(Request $request)
    {
        // Validate request parameters
        $validator = Validator::make($request->all(), [
            'Org_Id' => 'nullable|integer|exists:organisation_settings,Org_Id',
            'Lg_Id' => 'nullable|integer|exists:ledger_master,Lg_Id',
            'Gr_Id' => 'nullable|integer|exists:ledg_grps,Gr_Id',
            'ag_id' => 'nullable|integer|exists:subledg_master,SL_Id',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Build the query - updated for new structure
        $query = DB::table('finan_trns as FT')
            ->select(
                'LG.gr_code AS GroupCode',
                'LG.gr_nm AS GroupName',
                'LG.gr_stt AS GroupStatus',
                'FT.Org_Id',
                'OS.org_name',
                DB::raw('SUM(COALESCE(FT.cr_amt, 0)) AS TotalCredit'),
                DB::raw('SUM(COALESCE(FT.dr_amt, 0)) AS TotalDebit'),
                DB::raw('(SUM(COALESCE(FT.dr_amt, 0)) - SUM(COALESCE(FT.cr_amt, 0))) AS NetBalance')
            )
            ->leftJoin('ledger_master as LM', 'FT.Lg_Id', '=', 'LM.Lg_Id')
            ->leftJoin('ledg_grps as LG', function ($join) {
                $join->on('LM.Lg_Gr', '=', 'LG.gr_code')
                    ->orOn('LM.Gr_Id', '=', 'LG.Gr_Id');
            })
            ->leftJoin('organisation_settings as OS', 'FT.Org_Id', '=', 'OS.Org_Id')
            ->whereNotNull('FT.Lg_Id')
            ->groupBy('LG.gr_code', 'LG.gr_nm', 'LG.gr_stt', 'FT.Org_Id', 'OS.org_name');

        // Apply filters
        if ($request->filled('Org_Id')) {
            $query->where('FT.Org_Id', $request->Org_Id);
        }

        if ($request->filled('Lg_Id')) {
            $query->where('FT.Lg_Id', $request->Lg_Id);
        }

        if ($request->filled('Gr_Id')) {
            $query->where('LM.Gr_Id', $request->Gr_Id);
        }

        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('FT.vou_dt', [$request->start_date, $request->end_date]);
        }

        // Get all results without pagination
        $results = $query->get();

        // Calculate totals
        $totalCredit = $results->sum('TotalCredit');
        $totalDebit = $results->sum('TotalDebit');
        $totalNetBalance = $results->sum('NetBalance');

        // Prepare response
        $response = [
            'data' => $results,
            'summary' => [
                'total_credit' => round($totalCredit, 2),
                'total_debit' => round($totalDebit, 2),
                'net_balance' => round($totalNetBalance, 2),
                'total_records' => $results->count()
            ]
        ];

        return response()->json($response, 200);
    }

    public function getTransactionDetails(Request $request)
    {
        // Validation rules for query parameters
        $validator = Validator::make($request->all(), [
            'Org_Id' => 'nullable|integer|exists:organisation_settings,Org_Id',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'page' => 'integer|min:1',
            'per_page' => 'integer|min:1|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // Set pagination parameters
        $page = $request->input('page', 1);
        $perPage = $request->input('per_page', 25);
        $offset = ($page - 1) * $perPage;

        // Build the base query for first entries
        $query = FinanTrn::select(
            'finan_trns.vou_id',
            'finan_trns.vou_dt as Ft_Date',
            'finan_trns.vou_rf as RefPrName',
            'finan_trns.cr_amt as CrAmt',
            'finan_trns.dr_amt as DrAmt',
            'finan_trns.Lg_Id',
            'finan_trns.ref_id', // ref_id को भी select करें
            DB::raw('CONCAT(COALESCE(finan_trns.rmk1, ""), " ",
                            COALESCE(finan_trns.rmk2, ""), " ",
                            COALESCE(finan_trns.rmk3, ""), " ",
                            COALESCE(finan_trns.rmk4, ""), " ",
                            COALESCE(finan_trns.rmk5, "")) as Description')
        )
            ->whereIn('finan_trns.Ft_Id', function ($subQuery) {
                $subQuery->select(DB::raw('MIN(Ft_Id)'))
                    ->from('finan_trns')
                    ->groupBy('vou_id');
            });

        // Apply filters
        if ($request->filled('Org_Id')) {
            $query->where('finan_trns.Org_Id', $request->Org_Id);
        }

        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('finan_trns.vou_dt', [$request->start_date, $request->end_date]);
        }

        // Order by vou_dt for consistent balance calculation
        $query->orderBy('finan_trns.vou_dt', 'asc')
            ->orderBy('finan_trns.vou_id', 'asc');

        // Fetch transactions
        $transactions = $query->skip($offset)->take($perPage)->get();

        // Calculate running balance for each transaction
        $results = [];
        $balance = 0;
        $previousLgId = null;

        foreach ($transactions as $transaction) {
            // Reset balance if Lg_Id changes
            if ($previousLgId !== $transaction->Lg_Id) {
                $balance = 0;
                $openingBalance = FinanTrn::where('Lg_Id', $transaction->Lg_Id)
                    ->where('vou_dt', '<', $transaction->Ft_Date)
                    ->selectRaw('SUM(COALESCE(dr_amt, 0)) - SUM(COALESCE(cr_amt, 0)) as balance')
                    ->value('balance') ?? 0;
                $balance = $openingBalance;
                $previousLgId = $transaction->Lg_Id;
            }

            // Update balance for current transaction
            $balance += ($transaction->DrAmt ?? 0) - ($transaction->CrAmt ?? 0);

            $results[] = [
                'Ft_Date' => $transaction->Ft_Date,
                'RefPrName' => $transaction->RefPrName,
                'CrAmt' => $transaction->CrAmt,
                'DrAmt' => $transaction->DrAmt,
                'Balance' => round($balance, 2),
                'Description' => trim($transaction->Description),
                'Lg_Id' => $transaction->Lg_Id,
                'ref_id' => $transaction->ref_id,
            ];
        }

        // Get total count for pagination
        $totalQuery = FinanTrn::whereIn('Ft_Id', function ($subQuery) {
            $subQuery->select(DB::raw('MIN(Ft_Id)'))
                ->from('finan_trns')
                ->groupBy('vou_id');
        });

        if ($request->filled('Org_Id')) {
            $totalQuery->where('Org_Id', $request->Org_Id);
        }

        if ($request->filled('start_date') && $request->filled('end_date')) {
            $totalQuery->whereBetween('vou_dt', [$request->start_date, $request->end_date]);
        }

        $total = $totalQuery->count();

        // Prepare response
        $response = [
            'data' => $results,
            'pagination' => [
                'current_page' => $page,
                'per_page' => $perPage,
                'total' => $total,
                'last_page' => ceil($total / $perPage),
            ],
        ];

        return response()->json($response, 200);
    }



}
