<?php

namespace App\Http\Controllers;

use App\Models\GradePayMaster;
use App\Models\PayCommissionMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class GradePayMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $gradePays = GradePayMaster::with('commission')->get();
        return response()->json([
            'success' => true,
            'data' => $gradePays
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'commission_id' => 'required|exists:pay_commission_masters,commission_id',
            'grade_pay' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $gradePay = GradePayMaster::create($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Grade Pay created successfully',
            'data' => $gradePay->load('commission')
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $gradePay = GradePayMaster::with('commission')->find($id);

        if (!$gradePay) {
            return response()->json([
                'success' => false,
                'message' => 'Grade Pay not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $gradePay
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $gradePay = GradePayMaster::find($id);

        if (!$gradePay) {
            return response()->json([
                'success' => false,
                'message' => 'Grade Pay not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'commission_id' => 'exists:pay_commission_masters,commission_id',
            'grade_pay' => 'numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $gradePay->update($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Grade Pay updated successfully',
            'data' => $gradePay->load('commission')
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $gradePay = GradePayMaster::find($id);

        if (!$gradePay) {
            return response()->json([
                'success' => false,
                'message' => 'Grade Pay not found'
            ], 404);
        }

        $gradePay->delete();

        return response()->json([
            'success' => true,
            'message' => 'Grade Pay deleted successfully'
        ]);
    }

    /**
     * Get grade pays by commission
     */
    public function getByCommission(string $commissionId)
    {
        $commission = PayCommissionMaster::find($commissionId);

        if (!$commission) {
            return response()->json([
                'success' => false,
                'message' => 'Commission not found'
            ], 404);
        }

        $gradePays = GradePayMaster::where('commission_id', $commissionId)->get();

        return response()->json([
            'success' => true,
            'data' => $gradePays
        ]);
    }
}
