<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\GrantCategoryMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class GrantCategoryMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        try {
            $grantCategories = GrantCategoryMaster::orderBy('created_at', 'desc')->get();
            return response()->json([
                'success' => true,
                'message' => 'Grant categories retrieved successfully',
                'data' => $grantCategories
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve grant categories',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:100',
                'code' => 'nullable|string|max:100|unique:grant_category_masters,code',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $grantCategory = GrantCategoryMaster::create($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Grant category created successfully',
                'data' => $grantCategory
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create grant category',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $grantCategory = GrantCategoryMaster::find($id);

            if (!$grantCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Grant category not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Grant category retrieved successfully',
                'data' => $grantCategory
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve grant category',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            $grantCategory = GrantCategoryMaster::find($id);

            if (!$grantCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Grant category not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:100',
                'code' => 'nullable|string|max:100|unique:grant_category_masters,code,' . $id,
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $grantCategory->update($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Grant category updated successfully',
                'data' => $grantCategory
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update grant category',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $grantCategory = GrantCategoryMaster::find($id);

            if (!$grantCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Grant category not found'
                ], 404);
            }

            $grantCategory->delete();

            return response()->json([
                'success' => true,
                'message' => 'Grant category deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete grant category',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Restore soft deleted grant category.
     */
    public function restore(string $id)
    {
        try {
            $grantCategory = GrantCategoryMaster::withTrashed()->find($id);

            if (!$grantCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Grant category not found'
                ], 404);
            }

            if (!$grantCategory->trashed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Grant category is not deleted'
                ], 400);
            }

            $grantCategory->restore();

            return response()->json([
                'success' => true,
                'message' => 'Grant category restored successfully',
                'data' => $grantCategory
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to restore grant category',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get trashed grant categories.
     */
    public function trashed()
    {
        try {
            $grantCategories = GrantCategoryMaster::onlyTrashed()->orderBy('deleted_at', 'desc')->get();

            return response()->json([
                'success' => true,
                'message' => 'Trashed grant categories retrieved successfully',
                'data' => $grantCategories
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve trashed grant categories',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Force delete grant category.
     */
    public function forceDelete(string $id)
    {
        try {
            $grantCategory = GrantCategoryMaster::withTrashed()->find($id);

            if (!$grantCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Grant category not found'
                ], 404);
            }

            $grantCategory->forceDelete();

            return response()->json([
                'success' => true,
                'message' => 'Grant category permanently deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to permanently delete grant category',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
