<?php

namespace App\Http\Controllers;

use App\Models\GrantDivisionMapping;
use App\Models\FasGrantTypeMaster;
use App\Models\MainDepartment; // Changed from Department
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class GrantDivisionMappingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $mappings = GrantDivisionMapping::with(['grantType'])
            ->where('status', 1)
            ->get()
            ->map(function ($mapping) {
                // Load all departments from dept_ids
                $mapping->all_departments = $mapping->departments();
                return $mapping;
            });

        return response()->json([
            'success' => true,
            'message' => 'Grant division mappings retrieved successfully',
            'data' => $mappings
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'grant_ty_id' => 'required|exists:fas_grant_type_master,grant_type_id',
            'dept_ids' => 'required|array|min:1', // Required array of department IDs
            'dept_ids.*' => 'exists:main_departments,id', // Check each ID exists in main_departments
            'status' => 'sometimes|in:0,1'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        // Prepare department IDs
        $deptIds = array_unique($request->dept_ids);

        // Check if mapping already exists for this grant type with same departments
        $existingMappings = [];
        foreach ($deptIds as $deptId) {
            $existing = GrantDivisionMapping::where('grant_ty_id', $request->grant_ty_id)
                ->whereJsonContains('dept_ids', $deptId)
                ->first();

            if ($existing) {
                $existingMappings[] = [
                    'department_id' => $deptId,
                    'mapping' => $existing
                ];
            }
        }

        if (!empty($existingMappings)) {
            return response()->json([
                'success' => false,
                'message' => 'Some mappings already exist',
                'existing_mappings' => $existingMappings
            ], 409);
        }

        // Create mapping
        $mappingData = [
            'grant_ty_id' => $request->grant_ty_id,
            'dept_ids' => $deptIds,
            'status' => $request->status ?? 1
        ];

        $mapping = GrantDivisionMapping::create($mappingData);

        // Load relationships
        $mapping->load(['grantType']);
        $mapping->all_departments = $mapping->departments();

        return response()->json([
            'success' => true,
            'message' => 'Grant division mapping created successfully',
            'data' => $mapping
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $mapping = GrantDivisionMapping::with(['grantType'])
            ->find($id);

        if (!$mapping) {
            return response()->json([
                'success' => false,
                'message' => 'Grant division mapping not found'
            ], 404);
        }

        $mapping->all_departments = $mapping->departments();

        return response()->json([
            'success' => true,
            'message' => 'Grant division mapping retrieved successfully',
            'data' => $mapping
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $mapping = GrantDivisionMapping::find($id);

        if (!$mapping) {
            return response()->json([
                'success' => false,
                'message' => 'Grant division mapping not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'dept_ids' => 'sometimes|array|min:1',
            'dept_ids.*' => 'exists:main_departments,id',
            'status' => 'sometimes|in:0,1'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $updateData = [];

        // Update department IDs if provided
        if ($request->has('dept_ids')) {
            $deptIds = array_unique($request->dept_ids);
            $updateData['dept_ids'] = $deptIds;
        }

        if ($request->has('status')) {
            $updateData['status'] = $request->status;
        }

        if (!empty($updateData)) {
            $mapping->update($updateData);
        }

        // Load relationships
        $mapping->load(['grantType']);
        $mapping->all_departments = $mapping->departments();

        return response()->json([
            'success' => true,
            'message' => 'Grant division mapping updated successfully',
            'data' => $mapping
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $mapping = GrantDivisionMapping::find($id);

        if (!$mapping) {
            return response()->json([
                'success' => false,
                'message' => 'Grant division mapping not found'
            ], 404);
        }

        $mapping->update(['status' => 0]);

        return response()->json([
            'success' => true,
            'message' => 'Grant division mapping deactivated successfully'
        ]);
    }

    /**
     * Restore the specified resource.
     */
    public function restore($id)
    {
        $mapping = GrantDivisionMapping::find($id);

        if (!$mapping) {
            return response()->json([
                'success' => false,
                'message' => 'Grant division mapping not found'
            ], 404);
        }

        $mapping->update(['status' => 1]);

        $mapping->load(['grantType']);
        $mapping->all_departments = $mapping->departments();

        return response()->json([
            'success' => true,
            'message' => 'Grant division mapping activated successfully',
            'data' => $mapping
        ]);
    }

    /**
     * Get mappings by grant type
     */
    public function byGrantType($grantTypeId)
    {
        $grantType = FasGrantTypeMaster::find($grantTypeId);

        if (!$grantType) {
            return response()->json([
                'success' => false,
                'message' => 'Grant type not found'
            ], 404);
        }

        $mappings = GrantDivisionMapping::with(['grantType'])
            ->where('grant_ty_id', $grantTypeId)
            ->where('status', 1)
            ->get()
            ->map(function ($mapping) {
                $mapping->all_departments = $mapping->departments();
                return $mapping;
            });

        return response()->json([
            'success' => true,
            'message' => 'Mappings for grant type retrieved successfully',
            'grant_type' => $grantType,
            'data' => $mappings
        ]);
    }

    /**
     * Get mappings by department
     */
    public function byDepartment($departmentId)
    {
        $department = MainDepartment::find($departmentId);

        if (!$department) {
            return response()->json([
                'success' => false,
                'message' => 'Department not found'
            ], 404);
        }

        $mappings = GrantDivisionMapping::with(['grantType'])
            ->whereJsonContains('dept_ids', $departmentId)
            ->where('status', 1)
            ->get()
            ->map(function ($mapping) {
                $mapping->all_departments = $mapping->departments();
                return $mapping;
            });

        return response()->json([
            'success' => true,
            'message' => 'Mappings for department retrieved successfully',
            'department' => $department,
            'data' => $mappings
        ]);
    }

    /**
     * Add departments to an existing mapping
     */
    public function addDepartments(Request $request, $id)
    {
        $mapping = GrantDivisionMapping::find($id);

        if (!$mapping) {
            return response()->json([
                'success' => false,
                'message' => 'Grant division mapping not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'dept_ids' => 'required|array|min:1',
            'dept_ids.*' => 'exists:main_departments,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        // Get current department IDs
        $currentDeptIds = $mapping->dept_ids ?? [];

        // Merge with new department IDs
        $newDeptIds = array_unique($request->dept_ids);
        $mergedDeptIds = array_unique(array_merge($currentDeptIds, $newDeptIds));

        // Update the mapping
        $mapping->dept_ids = $mergedDeptIds;
        $mapping->save();

        $mapping->load(['grantType']);
        $mapping->all_departments = $mapping->departments();

        return response()->json([
            'success' => true,
            'message' => 'Departments added successfully',
            'added_count' => count(array_diff($newDeptIds, $currentDeptIds)),
            'total_departments' => count($mergedDeptIds),
            'data' => $mapping
        ]);
    }

    /**
     * Remove departments from an existing mapping
     */
    public function removeDepartments(Request $request, $id)
    {
        $mapping = GrantDivisionMapping::find($id);

        if (!$mapping) {
            return response()->json([
                'success' => false,
                'message' => 'Grant division mapping not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'dept_ids' => 'required|array|min:1',
            'dept_ids.*' => 'exists:main_departments,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        // Get current department IDs
        $currentDeptIds = $mapping->dept_ids ?? [];

        // Remove specified departments
        $removeDeptIds = $request->dept_ids;
        $updatedDeptIds = array_diff($currentDeptIds, $removeDeptIds);

        // If no departments left, deactivate the mapping
        if (empty($updatedDeptIds)) {
            $mapping->update(['status' => 0]);

            return response()->json([
                'success' => true,
                'message' => 'All departments removed. Mapping deactivated.',
                'data' => $mapping
            ]);
        }

        // Update the mapping
        $mapping->dept_ids = array_values($updatedDeptIds); // Re-index
        $mapping->save();

        $mapping->load(['grantType']);
        $mapping->all_departments = $mapping->departments();

        return response()->json([
            'success' => true,
            'message' => 'Departments removed successfully',
            'removed_count' => count(array_intersect($currentDeptIds, $removeDeptIds)),
            'remaining_departments' => count($updatedDeptIds),
            'data' => $mapping
        ]);
    }

    /**
     * Bulk create mappings
     */
    public function bulkStore(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'grant_ty_id' => 'required|exists:fas_grant_type_master,grant_type_id',
            'dept_ids' => 'required|array|min:1',
            'dept_ids.*' => 'exists:main_departments,id',
            'status' => 'sometimes|in:0,1'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        $deptIds = array_unique($request->dept_ids);

        DB::beginTransaction();
        try {
            // Create a single mapping with all department IDs
            $mappingData = [
                'grant_ty_id' => $request->grant_ty_id,
                'dept_ids' => $deptIds,
                'status' => $request->status ?? 1
            ];

            // Check if mapping already exists for this grant type
            $existing = GrantDivisionMapping::where('grant_ty_id', $request->grant_ty_id)
                ->first();

            if ($existing) {
                // Update existing mapping with new departments
                $currentDeptIds = $existing->dept_ids ?? [];
                $allDeptIds = array_unique(array_merge($currentDeptIds, $deptIds));

                $existing->dept_ids = $allDeptIds;
                $existing->save();
                $mapping = $existing;
            } else {
                // Create new mapping
                $mapping = GrantDivisionMapping::create($mappingData);
            }

            DB::commit();

            $mapping->load(['grantType']);
            $mapping->all_departments = $mapping->departments();

            return response()->json([
                'success' => true,
                'message' => 'Bulk mapping created/updated successfully',
                'total_departments' => count($deptIds),
                'data' => $mapping
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to create bulk mapping',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
