<?php

namespace App\Http\Controllers;

use App\Models\InquiryMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class InquiryMasterController extends Controller
{
    // 🔹 GET ALL
    public function index()
    {
        try {
            $data = InquiryMaster::with('department')->orderBy('inquiry_id', 'desc')->get();

            return response()->json([
                'success' => true,
                'message' => 'Inquiries fetched successfully',
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $data = $request->all();

            // Handle both Dept_Id and department_id inputs
            if ($request->has('Dept_Id')) {
                $data['department_id'] = $request->input('Dept_Id');
            } elseif ($request->has('department_id')) {
                $data['department_id'] = $request->input('department_id');
            }

            $validator = Validator::make($data, [
                'inquiry_no'   => 'nullable|string|max:30|unique:inquiry_master,inquiry_no',
                'citizen_name' => 'nullable|string|max:100',
                'mobile'       => 'nullable|string|max:15',
                'subject'      => 'required|string|max:200',
                'inquiry_type' => 'nullable|string|max:20',
                'department_id' => 'nullable|exists:main_departments,id', // Updated table reference
                'status'       => 'nullable|string|max:20',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Create inquiry with department_id
            $inquiry = InquiryMaster::create([
                'inquiry_no' => $data['inquiry_no'] ?? null,
                'citizen_name' => $data['citizen_name'] ?? null,
                'mobile' => $data['mobile'] ?? null,
                'subject' => $data['subject'],
                'inquiry_type' => $data['inquiry_type'] ?? null,
                'department_id' => $data['department_id'] ?? null,
                'status' => $data['status'] ?? 'PENDING',
            ]);

            // Load department relation
            $inquiry->load('department');

            return response()->json([
                'success' => true,
                'message' => 'Inquiry created successfully',
                'data' => $inquiry
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create inquiry',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $inquiry = InquiryMaster::with('department')->find($id);

            if (!$inquiry) {
                return response()->json([
                    'success' => false,
                    'message' => 'Inquiry not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $inquiry
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching inquiry',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE
    public function update(Request $request, $id)
    {
        try {
            $inquiry = InquiryMaster::find($id);

            if (!$inquiry) {
                return response()->json([
                    'success' => false,
                    'message' => 'Inquiry not found'
                ], 404);
            }

            $data = $request->all();

            // Handle both Dept_Id and department_id inputs
            if ($request->has('Dept_Id')) {
                $data['department_id'] = $request->input('Dept_Id');
            } elseif ($request->has('department_id')) {
                $data['department_id'] = $request->input('department_id');
            }

            $validator = Validator::make($data, [
                'inquiry_no'   => 'sometimes|required|string|max:30|unique:inquiry_master,inquiry_no,' . $id . ',inquiry_id',
                'citizen_name' => 'sometimes|nullable|string|max:100',
                'mobile'       => 'sometimes|nullable|string|max:15',
                'subject'      => 'sometimes|required|string|max:200',
                'inquiry_type' => 'sometimes|nullable|string|max:20',
                'department_id' => 'sometimes|nullable|exists:main_departments,id', // Updated table reference
                'status'       => 'sometimes|string|max:20',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Prepare update data
            $updateData = [];

            if ($request->has('inquiry_no')) {
                $updateData['inquiry_no'] = $request->input('inquiry_no');
            }
            if ($request->has('citizen_name')) {
                $updateData['citizen_name'] = $request->input('citizen_name');
            }
            if ($request->has('mobile')) {
                $updateData['mobile'] = $request->input('mobile');
            }
            if ($request->has('subject')) {
                $updateData['subject'] = $request->input('subject');
            }
            if ($request->has('inquiry_type')) {
                $updateData['inquiry_type'] = $request->input('inquiry_type');
            }
            // Handle department_id from both inputs
            if ($request->has('Dept_Id')) {
                $updateData['department_id'] = $request->input('Dept_Id');
            } elseif ($request->has('department_id')) {
                $updateData['department_id'] = $request->input('department_id');
            }
            if ($request->has('status')) {
                $updateData['status'] = $request->input('status');
            }

            $inquiry->update($updateData);

            // Reload with department
            $inquiry->load('department');

            return response()->json([
                'success' => true,
                'message' => 'Inquiry updated successfully',
                'data' => $inquiry
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update inquiry',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 DELETE (Hard delete – no is_active column)
    public function destroy($id)
    {
        try {
            $inquiry = InquiryMaster::find($id);

            if (!$inquiry) {
                return response()->json([
                    'success' => false,
                    'message' => 'Inquiry not found'
                ], 404);
            }

            $inquiry->delete();

            return response()->json([
                'success' => true,
                'message' => 'Inquiry deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete inquiry',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
