<?php

namespace App\Http\Controllers;

use App\Models\LeaveTypeMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class LeaveTypeMasterController extends Controller
{
    // 🔹 GET ALL
    public function index()
    {
        try {
            $data = LeaveTypeMaster::where('is_active', true)->get();

            return response()->json([
                'success' => true,
                'message' => 'Leave types fetched successfully',
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'leave_code' => 'nullable|string|max:20|unique:leave_type_master,leave_code',
                'leave_name' => 'required|string|max:100',
                'max_days'   => 'nullable|integer|min:1',
                'is_paid'    => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors'  => $validator->errors()
                ], 422);
            }

            $leave = LeaveTypeMaster::create($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Leave type created successfully',
                'data' => $leave
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create leave type',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW BY ID
    public function show($id)
    {
        try {
            $leave = LeaveTypeMaster::find($id);

            if (!$leave) {
                return response()->json([
                    'success' => false,
                    'message' => 'Leave type not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $leave
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching leave type',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE
    public function update(Request $request, $id)
    {
        try {
            $leave = LeaveTypeMaster::find($id);

            if (!$leave) {
                return response()->json([
                    'success' => false,
                    'message' => 'Leave type not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'leave_code' => 'sometimes|required|string|max:20|unique:leave_type_master,leave_code,' . $id . ',leave_type_id',
                'leave_name' => 'sometimes|required|string|max:100',
                'max_days'   => 'sometimes|nullable|integer|min:1',
                'is_paid'    => 'sometimes|boolean',
                'is_active'  => 'sometimes|boolean',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            // 🔹 Update only provided fields
            $leave->update($request->only([
                'leave_code',
                'leave_name',
                'max_days',
                'is_paid',
                'is_active'
            ]));

            return response()->json([
                'success' => true,
                'message' => 'Leave type updated successfully',
                'data' => $leave
            ], 200);
        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update leave type',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    // 🔹 DELETE (Soft via is_active)
    public function destroy($id)
    {
        try {
            $leave = LeaveTypeMaster::find($id);

            if (!$leave) {
                return response()->json([
                    'success' => false,
                    'message' => 'Leave type not found'
                ], 404);
            }

            $leave->update(['is_active' => false]);

            return response()->json([
                'success' => true,
                'message' => 'Leave type deactivated successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete leave type',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
