<?php

namespace App\Http\Controllers;

use App\Models\LedgGrp;
use App\Models\OrganisationSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Throwable;

class LedgGrpController extends Controller
{
    /**
     * GET all ledger groups
     */
    public function index(Request $request)
    {
        try {
            $Org_Id = $request->input('Org_Id');

            $ledgGrps = LedgGrp::query();

            // Filter by organization if provided
            if ($Org_Id) {
                $ledgGrps->where('Org_Id', $Org_Id);
            }

            $data = $ledgGrps->orderBy('gr_stt')->orderBy('gr_sr')->get();

            return response()->json([
                'status' => 'success',
                'message' => 'Ledger groups fetched successfully',
                'data' => $data
            ]);
        } catch (\Throwable $e) {
            Log::error('Fetch Ledger Groups Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch ledger groups',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * GET one ledger group
     */
    public function show($id)
    {
        try {
            $data = LedgGrp::find($id);

            if (!$data) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Ledger group not found'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Ledger group fetched successfully',
                'data' => $data
            ]);
        } catch (Throwable $e) {
            Log::error('Show Ledger Group Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch ledger group',
                'error' => $e->getMessage()
            ], 500);
        }
    }


public function store(Request $request)
{
    try {

        $grsh = $request->has('gr_stt') ? (string) $request->input('gr_stt') : '1';


        $grsrn = $request->has('gr_sr')
            ? (string) $request->input('gr_sr')
            : (string) LedgGrp::where('gr_stt', $grsh)->count() + 1;


        $prefix = str_pad($grsrn, 2, '0', STR_PAD_LEFT);
        $codeFormat = $prefix . '000000';


        $data = [
            'Org_Id'        => $request->input('Org_Id'),
            'gr_nm'         => $request->input('gr_nm'),
            'up_gr'         => $request->input('up_gr', $codeFormat),
            'gr_stt'        => $request->input('gr_stt'),
            'gr_sr'         => $request->input('gr_sr', $grsrn),
            'gr_alt'        => $request->input('gr_alt'),
            'gr_al_c'       => $request->input('gr_al_c'),
            'GIFI_Code'     => $request->input('GIFI_Code'),
            'is_del'        => $request->input('is_del', 'N'),
            'gr_exp'        => $request->input('gr_exp', 'N'),
            'gr_buk'        => $request->input('gr_buk', 'N'),
            'gr_prt'        => $request->input('gr_prt', 'N'),
            'op_id'         => $request->input('op_id', 1),
            'dt_tm'         => $request->input('dt_tm', date('Ymd')),
            'naof_grp'      => $request->input('naof_grp'),
            'credition'     => $request->input('credition'),
            'eff_gp'        => $request->input('eff_gp'),
            'eff_sub_led'   => $request->input('eff_sub_led'),

            'gr_code'       => $request->input('gr_code', $codeFormat),
        ];


        $data['gr_stt'] = $grsh;
        $data['gr_sr'] = $grsrn;


        if (!$request->has('up_gr')) {
            $data['up_gr'] = $request->input('up_gr', $codeFormat);
        }


        if (!$request->has('Org_Id') || !$request->has('gr_nm') || !$request->has('gr_stt')) {
            return response()->json([
                'status' => 'error',
                'message' => 'Org_Id, gr_nm and gr_stt are required'
            ], 422);
        }

        // Check if organization exists
        $org = OrganisationSetting::where('Org_Id', $request->input('Org_Id'))->first();
        if (!$org) {
            return response()->json([
                'status' => 'error',
                'message' => 'Organization not found'
            ], 404);
        }

        $ledgGrp = LedgGrp::create($data);

        return response()->json([
            'status' => 'success',
            'message' => 'Ledger group created successfully',
            'data' => $ledgGrp,
            'auto_calculated' => [
                'gr_stt' => $data['gr_stt'], // Grsh की जगह gr_stt
                'gr_sr' => $data['gr_sr'], // Grsrn की जगह gr_sr
                'gr_code' => $data['gr_code'], // Grcod की जगह gr_code
                'up_gr' => $data['up_gr'] // Upgrp की जगह up_gr
            ]
        ], 201);

    } catch (ValidationException $e) {
        return response()->json([
            'status' => 'error',
            'message' => 'Validation failed',
            'errors' => $e->errors()
        ], 422);
    } catch (Throwable $e) {
        Log::error('Create Ledger Group Error: ' . $e->getMessage());
        return response()->json([
            'status' => 'error',
            'message' => 'Unable to create ledger group',
            'error' => $e->getMessage()
        ], 500);
    }
}
 /**
 * UPDATE - FunctionTypeController का logic apply करें
 */
public function update(Request $request, $id)
{
    try {
        $ledgGrp = LedgGrp::find($id);

        if (!$ledgGrp) {
            return response()->json([
                'status' => 'error',
                'message' => 'Ledger group not found'
            ], 404);
        }

        // Validation rules
        $validatedData = $request->validate([
            // Original LedgGrp fields
            'gr_nm'         => 'sometimes|nullable|string|max:75',
            'up_gr'         => 'sometimes|nullable|string|max:25',
            'gr_sr'         => 'sometimes|nullable|integer',
            'gr_alt'        => 'sometimes|nullable|string|max:15',
            'gr_al_c'       => 'sometimes|nullable|string|max:75',
            'GIFI_Code'     => 'sometimes|nullable|string|max:10',
            'is_del'        => 'sometimes|nullable|string',
            'gr_exp'        => 'sometimes|nullable|string|in:Y,N',
            'gr_buk'        => 'sometimes|nullable|string|in:Y,N',
            'gr_prt'        => 'sometimes|nullable|string|in:Y,N',
            'op_id'         => 'sometimes|nullable|integer',
            'dt_tm'         => 'sometimes|nullable|string|max:10',
            'naof_grp'      => 'sometimes|nullable|string|max:10',
            'credition'     => 'sometimes|nullable|string|max:50',
            'eff_gp'        => 'sometimes|nullable|string|max:50',
            'eff_sub_led'   => 'sometimes|nullable|string|max:50',
            'gr_code'       => 'sometimes|nullable|string|max:25',
        ]);

        // सिर्फ validated data से update करें
        $ledgGrp->update($validatedData);

        return response()->json([
            'status' => 'success',
            'message' => 'Ledger group updated successfully',
            'data' => $ledgGrp
        ]);
    } catch (ValidationException $e) {
        return response()->json([
            'status' => 'error',
            'message' => 'Validation failed',
            'errors' => $e->errors()
        ], 422);
    } catch (Throwable $e) {
        Log::error('Update Ledger Group Error: ' . $e->getMessage());
        return response()->json([
            'status' => 'error',
            'message' => 'Unable to update ledger group',
            'error' => $e->getMessage()
        ], 500);
    }
}


public function destroy($id)
{
    try {
        $ledgGrp = LedgGrp::find($id);

        if (!$ledgGrp) {
            return response()->json([
                'status' => 'error',
                'message' => 'Ledger group not found'
            ], 404);
        }

        // PERMANENT DELETE - database से पूरी तरह हट जाएगा
        $ledgGrp->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Ledger group permanently deleted from database'
        ]);
    } catch (Throwable $e) {
        Log::error('Delete Ledger Group Error: ' . $e->getMessage());
        return response()->json([
            'status' => 'error',
            'message' => 'Unable to delete ledger group',
            'error' => $e->getMessage()
        ], 500);
    }
}

    /**
     * GET by Organization
     */
    public function getByOrganization($orgId)
    {
        try {
            $groups = LedgGrp::where('Org_Id', $orgId)
                ->orderBy('gr_stt')
                ->orderBy('gr_sr')
                ->get();

            return response()->json([
                'status' => 'success',
                'message' => 'Groups fetched successfully',
                'data' => $groups,
                'count' => $groups->count()
            ]);
        } catch (Throwable $e) {
            Log::error('Get By Organization Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch groups',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
