<?php

namespace App\Http\Controllers;

use App\Models\OrganisationSetting;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class OrganisationSettingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        try {
            $organisationSettings = OrganisationSetting::all();

            return response()->json([
                'success' => true,
                'message' => 'Organisation settings retrieved successfully',
                'data' => $organisationSettings
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve organisation settings',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

public function store(Request $request)
{
    try {
        // ✅ Validation
        $validator = Validator::make($request->all(), OrganisationSetting::createRules());

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], Response::HTTP_BAD_REQUEST);
        }

        $data = $validator->validated();

        // ✅ Handle Logo Upload (form-data)
        if ($request->hasFile('Org_Logo')) {
            $file = $request->file('Org_Logo');
            $filename = 'org_' . time() . '_' . Str::random(8) . '.' . $file->getClientOriginalExtension();

            // Store in storage/app/public/organisation/logo
            $file->storeAs('public/organisation/logo', $filename);

            // Store full URL in DB
            $data['Org_Logo'] = url('storage/organisation/logo/' . $filename);
        }

        // ✅ Create Organisation Setting
        $organisationSetting = OrganisationSetting::create($data);

        return response()->json([
            'success' => true,
            'message' => 'Organisation setting created successfully',
            'data' => $organisationSetting
        ], Response::HTTP_CREATED);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to create organisation setting',
            'error' => $e->getMessage()
        ], Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $organisationSetting = OrganisationSetting::find($id);

            if (!$organisationSetting) {
                return response()->json([
                    'success' => false,
                    'message' => 'Organisation setting not found'
                ], Response::HTTP_NOT_FOUND);
            }

            return response()->json([
                'success' => true,
                'message' => 'Organisation setting retrieved successfully',
                'data' => $organisationSetting
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve organisation setting',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

 public function update(Request $request, string $id)
{
    try {
        // ✅ Pehle check karein kitne records hain
        $existingCount = OrganisationSetting::count();

        // ✅ Agar koi record nahi hai (0), toh CREATE karein
        if ($existingCount === 0) {
            return $this->createFirstOrganization($request);
        }

        // ✅ Agar ek record hai, toh usi ko UPDATE karein (ID se check karein)
        $organisationSetting = OrganisationSetting::first(); // Pehla record lelo

        // ✅ Agar URL me different ID diya hai, toh error de do
        if ($organisationSetting->Org_Id != $id) {
            return response()->json([
                'success' => false,
                'message' => 'Only one organization setting is allowed. Update the existing one.',
                'existing_id' => $organisationSetting->Org_Id
            ], Response::HTTP_BAD_REQUEST);
        }

        // ✅ Validation for UPDATE
        $validator = Validator::make($request->all(), OrganisationSetting::updateRules($organisationSetting->Org_Id));

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], Response::HTTP_BAD_REQUEST);
        }

        $data = $validator->validated();

        // ✅ Handle Logo Update
        if (!empty($request->Org_Logo)) {
            // Delete old logo if exists
            if ($organisationSetting->Org_Logo) {
                $oldPath = str_replace(url('/storage') . '/', '', $organisationSetting->Org_Logo);
                if (Storage::exists('public/' . $oldPath)) {
                    Storage::delete('public/' . $oldPath);
                }
            }

            $filename = null;

            if (str_contains($request->Org_Logo, 'base64')) {
                $base64 = preg_replace('#^data:image/\w+;base64,#i', '', $request->Org_Logo);
                $image = base64_decode($base64);
                $filename = 'org_' . time() . '_' . Str::random(8) . '.png';
                Storage::put('public/organisation/logo/' . $filename, $image);
            } elseif ($request->hasFile('Org_Logo')) {
                $file = $request->file('Org_Logo');
                $filename = 'org_' . time() . '_' . Str::random(8) . '.' . $file->getClientOriginalExtension();
                $file->storeAs('public/organisation/logo', $filename);
            }

            if ($filename) {
                $data['Org_Logo'] = url('storage/organisation/logo/' . $filename);
            }
        }

        // ✅ Update the single record
        $organisationSetting->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Organization setting updated successfully',
            'data' => $organisationSetting,
            'action' => 'updated'
        ], Response::HTTP_OK);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to process organization setting',
            'error' => $e->getMessage()
        ], Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}

/**
 * Create the first (and only) organization setting
 */
private function createFirstOrganization(Request $request)
{
    try {
        $validator = Validator::make($request->all(), OrganisationSetting::createRules());

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], Response::HTTP_BAD_REQUEST);
        }

        $data = $validator->validated();

        // ✅ Handle Logo Upload for CREATE
        if ($request->hasFile('Org_Logo')) {
            $file = $request->file('Org_Logo');
            $filename = 'org_' . time() . '_' . Str::random(8) . '.' . $file->getClientOriginalExtension();
            $file->storeAs('public/organisation/logo', $filename);
            $data['Org_Logo'] = url('storage/organisation/logo/' . $filename);
        }
        elseif (!empty($request->Org_Logo) && str_contains($request->Org_Logo, 'base64')) {
            $base64 = preg_replace('#^data:image/\w+;base64,#i', '', $request->Org_Logo);
            $image = base64_decode($base64);
            $filename = 'org_' . time() . '_' . Str::random(8) . '.png';
            Storage::put('public/organisation/logo/' . $filename, $image);
            $data['Org_Logo'] = url('storage/organisation/logo/' . $filename);
        }

        // ✅ Create first organization
        $organisationSetting = OrganisationSetting::create($data);

        return response()->json([
            'success' => true,
            'message' => 'First organization setting created successfully',
            'data' => $organisationSetting,
            'action' => 'created'
        ], Response::HTTP_CREATED);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to create organization setting',
            'error' => $e->getMessage()
        ], Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $organisationSetting = OrganisationSetting::find($id);

            if (!$organisationSetting) {
                return response()->json([
                    'success' => false,
                    'message' => 'Organisation setting not found'
                ], Response::HTTP_NOT_FOUND);
            }

            $organisationSetting->delete();

            return response()->json([
                'success' => true,
                'message' => 'Organisation setting deleted successfully'
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete organisation setting',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Restore the specified soft deleted resource.
     */
    public function restore(string $id)
    {
        try {
            $organisationSetting = OrganisationSetting::withTrashed()->where('Org_Id', $id)->first();

            if (!$organisationSetting) {
                return response()->json([
                    'success' => false,
                    'message' => 'Organisation setting not found'
                ], Response::HTTP_NOT_FOUND);
            }

            if (!$organisationSetting->trashed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Organisation setting is not deleted'
                ], Response::HTTP_BAD_REQUEST);
            }

            $organisationSetting->restore();

            return response()->json([
                'success' => true,
                'message' => 'Organisation setting restored successfully',
                'data' => $organisationSetting
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to restore organisation setting',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Permanently delete the specified resource.
     */
    public function forceDelete(string $id)
    {
        try {
            $organisationSetting = OrganisationSetting::withTrashed()->where('Org_Id', $id)->first();

            if (!$organisationSetting) {
                return response()->json([
                    'success' => false,
                    'message' => 'Organisation setting not found'
                ], Response::HTTP_NOT_FOUND);
            }

            $organisationSetting->forceDelete();

            return response()->json([
                'success' => true,
                'message' => 'Organisation setting permanently deleted successfully'
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to permanently delete organisation setting',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Get all trashed organisation settings.
     */
    public function trashed()
    {
        try {
            $trashedSettings = OrganisationSetting::onlyTrashed()->get();

            return response()->json([
                'success' => true,
                'message' => 'Deleted organisation settings retrieved successfully',
                'data' => $trashedSettings
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve deleted organisation settings',
                'error' => $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }
}
