<?php

namespace App\Http\Controllers;

use App\Models\PayMatrixCellMaster;
use App\Models\PayMatrixLevelMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PayMatrixCellMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $cells = PayMatrixCellMaster::with('level')->get();
        return response()->json([
            'success' => true,
            'data' => $cells
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'level_id' => 'required|exists:pay_matrix_level_masters,level_id',
            'cell_index' => 'required|integer|min:1',
            'basic_pay' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $cell = PayMatrixCellMaster::create($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Matrix Cell created successfully',
            'data' => $cell->load('level')
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $cell = PayMatrixCellMaster::with('level')->find($id);

        if (!$cell) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Matrix Cell not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $cell
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $cell = PayMatrixCellMaster::find($id);

        if (!$cell) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Matrix Cell not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'level_id' => 'exists:pay_matrix_level_masters,level_id',
            'cell_index' => 'integer|min:1',
            'basic_pay' => 'numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $cell->update($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Matrix Cell updated successfully',
            'data' => $cell->load('level')
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $cell = PayMatrixCellMaster::find($id);

        if (!$cell) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Matrix Cell not found'
            ], 404);
        }

        $cell->delete();

        return response()->json([
            'success' => true,
            'message' => 'Pay Matrix Cell deleted successfully'
        ]);
    }

    /**
     * Get cells by level
     */
    public function getByLevel(string $levelId)
    {
        $level = PayMatrixLevelMaster::find($levelId);

        if (!$level) {
            return response()->json([
                'success' => false,
                'message' => 'Level not found'
            ], 404);
        }

        $cells = PayMatrixCellMaster::where('level_id', $levelId)
            ->orderBy('cell_index')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $cells
        ]);
    }
}
