<?php

namespace App\Http\Controllers;

use App\Models\PayRateMaster;
use App\Models\EmployeeTypeMaster; // ADDED
use App\Models\LedgerMaster;
use App\Models\PayrollMaster;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class PayRateMasterController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        try {
            $perPage = $request->get('per_page', 10);
            $search = $request->get('search', '');
            $withTrashed = $request->get('with_trashed', false);
            $sortBy = $request->get('sort_by', 'pay_rate_id');
            $sortOrder = $request->get('sort_order', 'desc');
            $empTypeId = $request->get('emp_type_id'); // CHANGED: tend_work_id से emp_type_id
            $date = $request->get('effective_date');

            $validSortColumns = ['pay_rate_id', 'effective_date', 'created_at', 'updated_at'];
            $sortBy = in_array($sortBy, $validSortColumns) ? $sortBy : 'pay_rate_id';
            $sortOrder = in_array(strtolower($sortOrder), ['asc', 'desc']) ? strtolower($sortOrder) : 'desc';

            // Load all related models - CHANGED: tenderWork से employeeType
            $query = PayRateMaster::with([
                'employeeType:emp_type_id,type_code,type_name,is_active' // CHANGED
            ]);

            if ($withTrashed) {
                $query->withTrashed();
            }

            if ($search) {
                $query->where(function ($q) use ($search) {
                    $q->where('emp_type_id', 'like', "%{$search}%") // CHANGED
                        ->orWhereHas('employeeType', function ($q1) use ($search) { // CHANGED
                            $q1->where('type_code', 'like', "%{$search}%")
                                ->orWhere('type_name', 'like', "%{$search}%");
                        })
                        ->orWhereJsonContains('pay_parameters', [['applicable_criteria_name' => $search]]);
                });
            }

            if ($empTypeId) { // CHANGED
                $query->where('emp_type_id', $empTypeId);
            }

            if ($date) {
                $query->whereDate('effective_date', $date);
            }

            $query->orderBy($sortBy, $sortOrder);

            $payRates = $query->paginate($perPage);

            // Load payroll and ledger data for each pay rate
            $transformedData = $payRates->getCollection()->map(function ($item) {
                return $this->transformPayRateWithDetails($item);
            });

            $payRates->setCollection($transformedData);

            return response()->json([
                'success' => true,
                'message' => 'Pay Rate Master retrieved successfully.',
                'data' => $payRates->items(),
                'meta' => [
                    'current_page' => $payRates->currentPage(),
                    'last_page' => $payRates->lastPage(),
                    'per_page' => $payRates->perPage(),
                    'total' => $payRates->total(),
                    'from' => $payRates->firstItem(),
                    'to' => $payRates->lastItem(),
                    'sort_by' => $sortBy,
                    'sort_order' => $sortOrder
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve Pay Rate Master.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make(
            $request->all(),
            PayRateMaster::createRules(),
            PayRateMaster::validationMessages()
        );

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed.',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Validate and format pay parameters
            $payParameters = $this->validateAndFormatPayParameters($request->pay_parameters);

            if (empty($payParameters)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No valid pay parameters provided.'
                ], 422);
            }

            // Check for exact duplicate - CHANGED: tend_work_id से emp_type_id
            $existing = PayRateMaster::isExactDuplicate(
                $request->emp_type_id, // CHANGED
                $request->effective_date,
                $payParameters
            );

            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'An identical pay parameter set already exists.',
                    'error_type' => 'duplicate'
                ], 409);
            }

            $payRate = PayRateMaster::create([
                'emp_type_id' => $request->emp_type_id, // CHANGED
                'effective_date' => $request->effective_date,
                'pay_parameters' => $payParameters
            ]);

            // Load all related data - CHANGED
            $payRate->load(['employeeType']);

            return response()->json([
                'success' => true,
                'message' => 'Pay rate created successfully.',
                'data' => $this->transformPayRateWithDetails($payRate)
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create pay rate.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): JsonResponse
    {
        try {
            $payRate = PayRateMaster::with([
                'employeeType:emp_type_id,type_code,type_name,is_active' // CHANGED
            ])->withTrashed()->find($id);

            if (!$payRate) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pay rate not found.'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Pay rate retrieved successfully.',
                'data' => $this->transformPayRateWithDetails($payRate)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve pay rate.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Transform pay rate data with all related details - CHANGED
     */
    private function transformPayRateWithDetails($payRate): array
    {
        $payParameters = [];
        if (is_array($payRate->pay_parameters)) {
            foreach ($payRate->pay_parameters as $param) {
                $payroll = PayrollMaster::where('payroll_id', $param['payroll_id'])->first();
                $ledger = LedgerMaster::where('Lg_Id', $param['Lg_Id'])->first();

                $payParameters[] = [
                    'payroll_id' => $param['payroll_id'] ?? null,
                    'payroll_details' => $payroll ? [
                        'payroll_id' => $payroll->payroll_id,
                        'payroll_code' => $payroll->payroll_code,
                        'payroll_name' => $payroll->payroll_name,
                        'payroll_percentage' => $payroll->payroll_percentage,
                        'payroll_percentage_formatted' => number_format($payroll->payroll_percentage, 2) . '%'
                    ] : null,
                    'applicable_criteria_name' => $param['applicable_criteria_name'] ?? '',
                    'add_deduct' => $param['add_deduct'] ?? 0,
                    'add_deduct_text' => PayRateMaster::ADD_DEDUCT_TEXTS[$param['add_deduct'] ?? 0] ?? 'Unknown',
                    'Lg_Id' => $param['Lg_Id'] ?? null,
                    'ledger_details' => $ledger ? [
                        'Lg_Id' => $ledger->Lg_Id,
                        'Lg_Code' => $ledger->Lg_Code,
                        'Lg_Name' => $ledger->Lg_Name,
                        'Lg_Address' => $ledger->Lg_Address,
                        'Lg_City' => $ledger->Lg_City,
                        'Lg_Pin' => $ledger->Lg_Pin,
                        'Lg_Phone' => $ledger->Lg_Phone,
                        'Lg_Email' => $ledger->Lg_Email,
                        'Lg_Status' => $ledger->Lg_Status,
                        'status_text' => $ledger->getStatusTextAttribute(),
                        'full_address' => $ledger->getFullAddressAttribute()
                    ] : null,
                    'percentage' => isset($param['percentage']) ? (float)$param['percentage'] : null,
                    'percentage_formatted' => isset($param['percentage']) ? number_format($param['percentage'], 2) . '%' : null,
                    'amount_per_day' => isset($param['amount_per_day']) ? (float)$param['amount_per_day'] : null,
                    'amount_per_day_formatted' => isset($param['amount_per_day']) ? '₹' . number_format($param['amount_per_day'], 2) : null,
                    'effective_percentage' => isset($param['percentage']) ?
                        (($param['add_deduct'] == PayRateMaster::DEDUCT ? -$param['percentage'] : $param['percentage'])) : 0,
                    'effective_amount_per_day' => isset($param['amount_per_day']) ?
                        (($param['add_deduct'] == PayRateMaster::DEDUCT ? -$param['amount_per_day'] : $param['amount_per_day'])) : 0,
                    'calculation_formula' => $param['calculation_formula'] ?? null,
                ];
            }
        }

        return [
            'pay_rate_id' => $payRate->pay_rate_id,
            'emp_type_id' => $payRate->emp_type_id, // CHANGED
            'employee_type' => $payRate->employeeType ? [ // CHANGED
                'emp_type_id' => $payRate->employeeType->emp_type_id,
                'type_code' => $payRate->employeeType->type_code,
                'type_name' => $payRate->employeeType->type_name,
                'is_active' => $payRate->employeeType->is_active,
                'status_text' => $payRate->employeeType->status_text
            ] : null,
            'effective_date' => $payRate->effective_date ? $payRate->effective_date->format('Y-m-d') : null,
            'effective_date_formatted' => $payRate->effective_date ? $payRate->effective_date->format('d-m-Y') : null,
            'pay_parameters' => $payParameters,
            'total_components' => $payRate->total_components,
            'total_percentage' => $payRate->total_percentage,
            'total_percentage_formatted' => number_format($payRate->total_percentage, 2) . '%',
            'total_amount_per_day' => $payRate->total_amount_per_day,
            'total_amount_per_day_formatted' => '₹' . number_format($payRate->total_amount_per_day, 2),
            'created_at' => $payRate->created_at ? $payRate->created_at->format('Y-m-d H:i:s') : null,
            'updated_at' => $payRate->updated_at ? $payRate->updated_at->format('Y-m-d H:i:s') : null,
            'deleted_at' => $payRate->deleted_at ? $payRate->deleted_at->format('Y-m-d H:i:s') : null,
            'is_deleted' => !is_null($payRate->deleted_at),
            'is_active' => is_null($payRate->deleted_at)
        ];
    }

    /**
     * Update the specified resource in storage. - CHANGED
     */
    public function update(Request $request, string $id): JsonResponse
    {
        try {
            $payRate = PayRateMaster::find($id);

            if (!$payRate) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pay rate not found.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'emp_type_id' => 'sometimes|required|exists:employee_type_master,emp_type_id', // CHANGED
                'effective_date' => 'sometimes|required|date|date_format:Y-m-d',
                'pay_parameters' => 'sometimes|required|array|min:1',
                'pay_parameters.*.payroll_id' => 'required_with:pay_parameters|exists:payroll_master,payroll_id',
                'pay_parameters.*.applicable_criteria_name' => 'required_with:pay_parameters|string|max:100',
                'pay_parameters.*.add_deduct' => 'required_with:pay_parameters|in:0,1',
                'pay_parameters.*.Lg_Id' => 'required_with:pay_parameters|exists:ledger_master,Lg_Id',
                'pay_parameters.*.percentage' => 'nullable|numeric|min:0|max:100|regex:/^\d+(\.\d{1,2})?$/',
                'pay_parameters.*.amount_per_day' => 'nullable|numeric|min:0|max:999999.99|regex:/^\d+(\.\d{1,2})?$/',
                'pay_parameters.*.calculation_formula' => 'nullable|string|max:255'
            ], PayRateMaster::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $updateData = [];
            $fields = ['emp_type_id', 'effective_date', 'pay_parameters']; // CHANGED

            foreach ($fields as $field) {
                if ($request->has($field)) {
                    $updateData[$field] = $request->$field;
                }
            }

            // If updating pay_parameters, validate and format them
            if (isset($updateData['pay_parameters'])) {
                $payParameters = $this->validateAndFormatPayParameters($updateData['pay_parameters']);
                if (empty($payParameters)) {
                    return response()->json([
                        'success' => false,
                        'message' => 'No valid pay parameters provided.'
                    ], 422);
                }
                $updateData['pay_parameters'] = $payParameters;
            }

            // Check for duplicate if updating - CHANGED
            if (!empty($updateData)) {
                $empTypeId = $updateData['emp_type_id'] ?? $payRate->emp_type_id; // CHANGED
                $effectiveDate = $updateData['effective_date'] ?? $payRate->effective_date;
                $payParameters = $updateData['pay_parameters'] ?? $payRate->pay_parameters;

                $isDuplicate = PayRateMaster::isExactDuplicate(
                    $empTypeId, // CHANGED
                    $effectiveDate,
                    $payParameters,
                    $id
                );

                if ($isDuplicate) {
                    return response()->json([
                        'success' => false,
                        'message' => 'An identical pay parameter set already exists.',
                        'error_type' => 'duplicate'
                    ], 409);
                }
            }

            $payRate->update($updateData);
            $payRate->refresh();
            $payRate->load(['employeeType']); // CHANGED

            return response()->json([
                'success' => true,
                'message' => 'Pay rate updated successfully.',
                'data' => $this->transformPayRate($payRate)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update pay rate.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $payRate = PayRateMaster::find($id);

            if (!$payRate) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pay rate not found.'
                ], 404);
            }

            $payRate->delete();

            return response()->json([
                'success' => true,
                'message' => 'Pay rate deleted successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete pay rate.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get pay rates by employee type and date - CHANGED
     */
    public function getByEmployeeTypeAndDate($empTypeId, $date): JsonResponse // CHANGED
    {
        try {
            $validator = Validator::make([
                'emp_type_id' => $empTypeId, // CHANGED
                'effective_date' => $date
            ], [
                'emp_type_id' => 'required|exists:employee_type_master,emp_type_id', // CHANGED
                'effective_date' => 'required|date|date_format:Y-m-d'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $payRates = PayRateMaster::with([
                'employeeType:emp_type_id,type_code,type_name,is_active' // CHANGED
            ])
                ->where('emp_type_id', $empTypeId) // CHANGED
                ->whereDate('effective_date', $date)
                ->whereNull('deleted_at')
                ->orderBy('effective_date', 'desc')
                ->get();

            $transformedData = $payRates->map(function ($item) {
                return $this->transformPayRate($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'Pay rates retrieved successfully.',
                'data' => $transformedData,
                'meta' => [
                    'emp_type_id' => $empTypeId, // CHANGED
                    'effective_date' => $date,
                    'total_entries' => $payRates->count()
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve pay rates.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Calculate total amount for number of days
     */
    public function calculateTotalAmount(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'pay_rate_id' => 'required|exists:pay_rate_master,pay_rate_id',
                'number_of_days' => 'required|integer|min:1',
                'base_amount' => 'nullable|numeric|min:0'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $payRate = PayRateMaster::find($request->pay_rate_id);
            $numberOfDays = $request->number_of_days;
            $baseAmount = $request->base_amount ?? 0;

            $totalAmountPerDay = $payRate->total_amount_per_day;
            $totalAmount = $payRate->calculateTotalAmount($numberOfDays);

            $totalPercentageAmount = $payRate->calculateTotalPercentageAmount($baseAmount);
            $grandTotal = $totalAmount + $totalPercentageAmount;

            return response()->json([
                'success' => true,
                'message' => 'Calculation completed successfully.',
                'data' => [
                    'pay_rate_id' => $payRate->pay_rate_id,
                    'emp_type_id' => $payRate->emp_type_id, // CHANGED
                    'effective_date' => $payRate->effective_date->format('Y-m-d'),
                    'number_of_days' => $numberOfDays,
                    'base_amount' => $baseAmount,
                    'total_amount_per_day' => $totalAmountPerDay,
                    'total_amount_per_day_formatted' => '₹' . number_format($totalAmountPerDay, 2),
                    'total_days_amount' => $totalAmount,
                    'total_days_amount_formatted' => '₹' . number_format($totalAmount, 2),
                    'total_percentage_amount' => $totalPercentageAmount,
                    'total_percentage_amount_formatted' => '₹' . number_format($totalPercentageAmount, 2),
                    'grand_total' => $grandTotal,
                    'grand_total_formatted' => '₹' . number_format($grandTotal, 2)
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to calculate total amount.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Transform pay rate data for API response. - CHANGED
     */
    private function transformPayRate($payRate): array
    {
        $payParameters = [];
        if (is_array($payRate->pay_parameters)) {
            foreach ($payRate->pay_parameters as $param) {
                $payParameters[] = PayRateMaster::transformParameterItem($param);
            }
        }

        return [
            'pay_rate_id' => $payRate->pay_rate_id,
            'emp_type_id' => $payRate->emp_type_id, // CHANGED
            'employee_type' => $payRate->employeeType ? [ // CHANGED
                'emp_type_id' => $payRate->employeeType->emp_type_id,
                'type_code' => $payRate->employeeType->type_code,
                'type_name' => $payRate->employeeType->type_name,
                'is_active' => $payRate->employeeType->is_active,
                'status_text' => $payRate->employeeType->status_text
            ] : null,
            'effective_date' => $payRate->effective_date ? $payRate->effective_date->format('Y-m-d') : null,
            'effective_date_formatted' => $payRate->effective_date ? $payRate->effective_date->format('d-m-Y') : null,
            'pay_parameters' => $payParameters,
            'total_components' => $payRate->total_components,
            'total_percentage' => $payRate->total_percentage,
            'total_percentage_formatted' => number_format($payRate->total_percentage, 2) . '%',
            'total_amount_per_day' => $payRate->total_amount_per_day,
            'total_amount_per_day_formatted' => '₹' . number_format($payRate->total_amount_per_day, 2),
            'created_at' => $payRate->created_at ? $payRate->created_at->format('Y-m-d H:i:s') : null,
            'updated_at' => $payRate->updated_at ? $payRate->updated_at->format('Y-m-d H:i:s') : null,
            'deleted_at' => $payRate->deleted_at ? $payRate->deleted_at->format('Y-m-d H:i:s') : null,
            'is_deleted' => !is_null($payRate->deleted_at),
            'is_active' => is_null($payRate->deleted_at)
        ];
    }

    /**
     * Validate and format pay parameters
     */
    private function validateAndFormatPayParameters(array $payParameters): array
    {
        $formatted = [];

        foreach ($payParameters as $param) {
            // Validate required fields
            if (!isset($param['payroll_id'], $param['applicable_criteria_name'], $param['add_deduct'], $param['Lg_Id'])) {
                continue;
            }

            // Format the parameter
            $formattedParam = [
                'payroll_id' => (int)$param['payroll_id'],
                'applicable_criteria_name' => (string)$param['applicable_criteria_name'],
                'add_deduct' => (int)$param['add_deduct'],
                'Lg_Id' => (int)$param['Lg_Id']
            ];

            // Add optional fields if present
            if (isset($param['percentage'])) {
                $formattedParam['percentage'] = (float)$param['percentage'];
            }

            if (isset($param['amount_per_day'])) {
                $formattedParam['amount_per_day'] = (float)$param['amount_per_day'];
            }

            if (isset($param['calculation_formula'])) {
                $formattedParam['calculation_formula'] = (string)$param['calculation_formula'];
            }

            $formatted[] = $formattedParam;
        }

        return $formatted;
    }

    /**
     * Restore soft deleted pay rate
     */
    public function restore(string $id): JsonResponse
    {
        try {
            $payRate = PayRateMaster::withTrashed()->find($id);

            if (!$payRate) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pay rate not found.'
                ], 404);
            }

            if (!$payRate->trashed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pay rate is not deleted.'
                ], 400);
            }

            $payRate->restore();
            $payRate->load(['employeeType']); // CHANGED

            return response()->json([
                'success' => true,
                'message' => 'Pay rate restored successfully.',
                'data' => $this->transformPayRate($payRate)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to restore pay rate.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Permanently delete pay rate
     */
    public function forceDelete(string $id): JsonResponse
    {
        try {
            $payRate = PayRateMaster::withTrashed()->find($id);

            if (!$payRate) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pay rate not found.'
                ], 404);
            }

            $payRate->forceDelete();

            return response()->json([
                'success' => true,
                'message' => 'Pay rate permanently deleted successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to permanently delete pay rate.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }
}
