<?php

namespace App\Http\Controllers;

use App\Models\PayScaleMaster;
use App\Models\PayCommissionMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PayScaleMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $payScales = PayScaleMaster::with('commission')->get();
        return response()->json([
            'success' => true,
            'data' => $payScales
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'commission_id' => 'required|exists:pay_commission_masters,commission_id',
            'scale_code' => 'required|string|max:20',
            'start_basic' => 'required|numeric|min:0',
            'increment_amount' => 'required|numeric|min:0',
            'increment_steps' => 'required|integer|min:1',
            'max_basic' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $payScale = PayScaleMaster::create($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Scale created successfully',
            'data' => $payScale->load('commission')
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $payScale = PayScaleMaster::with('commission')->find($id);

        if (!$payScale) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Scale not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $payScale
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $payScale = PayScaleMaster::find($id);

        if (!$payScale) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Scale not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'commission_id' => 'exists:pay_commission_masters,commission_id',
            'scale_code' => 'string|max:20',
            'start_basic' => 'numeric|min:0',
            'increment_amount' => 'numeric|min:0',
            'increment_steps' => 'integer|min:1',
            'max_basic' => 'numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $payScale->update($validator->validated());

        return response()->json([
            'success' => true,
            'message' => 'Pay Scale updated successfully',
            'data' => $payScale->load('commission')
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $payScale = PayScaleMaster::find($id);

        if (!$payScale) {
            return response()->json([
                'success' => false,
                'message' => 'Pay Scale not found'
            ], 404);
        }

        $payScale->delete();

        return response()->json([
            'success' => true,
            'message' => 'Pay Scale deleted successfully'
        ]);
    }

    /**
     * Get pay scales by commission
     */
    public function getByCommission(string $commissionId)
    {
        $commission = PayCommissionMaster::find($commissionId);

        if (!$commission) {
            return response()->json([
                'success' => false,
                'message' => 'Commission not found'
            ], 404);
        }

        $payScales = PayScaleMaster::where('commission_id', $commissionId)->get();

        return response()->json([
            'success' => true,
            'data' => $payScales
        ]);
    }
}
