<?php

namespace App\Http\Controllers;

use App\Models\PaymentVoucher;
use App\Models\TendorAllocation;
use App\Models\TenderEntryMaster;
use App\Models\SubLedgerMaster;
use App\Models\NewTenderWork;
use App\Models\LedgerMaster;
use App\Models\VendorInvoiceParameter;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class PaymentVoucherController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = PaymentVoucher::with([
                'branch',
                'ledger',
                'subLedger',
                'tenderAllocation' => function($q) {
                    $q->with(['tenderEntry' => function($q2) {
                        $q2->with('tenderWork');
                    }]);
                },
                'tenderWork'
            ]);

            // Apply filters if provided
            if ($request->has('voucher_no')) {
                $query->byVoucherNumber($request->voucher_no);
            }

            if ($request->has('branch_id')) {
                $query->byBranch($request->branch_id);
            }

            if ($request->has('ledger_id')) {
                $query->byLedger($request->ledger_id);
            }

            if ($request->has('sub_ledger_id')) {
                $query->bySubLedger($request->sub_ledger_id);
            }

            if ($request->has('tender_work_id')) {
                $query->byTenderWork($request->tender_work_id);
            }

            if ($request->has('payment_status')) {
                $query->byPaymentStatus($request->payment_status);
            }

            if ($request->has('start_date') && $request->has('end_date')) {
                $query->byDateRange($request->start_date, $request->end_date);
            }

            if ($request->has('search')) {
                $query->search($request->search);
            }

            // Sorting
            if ($request->has('sort_by')) {
                $sortDirection = $request->get('sort_direction', 'asc');
                $query->orderBy($request->sort_by, $sortDirection);
            } else {
                $query->orderBy('Pay_Voucher', 'desc');
            }

            // Pagination
            $perPage = $request->get('per_page', 15);
            $paymentVouchers = $query->paginate($perPage);

            // Transform the response to include tender work details from tender allocation
            $transformedData = $paymentVouchers->getCollection()->map(function ($voucher) {
                $data = $voucher->toArray();

                // Get tender work details from tender allocation if available
                if ($voucher->tenderAllocation &&
                    $voucher->tenderAllocation->tenderEntry &&
                    $voucher->tenderAllocation->tenderEntry->tenderWork) {

                    $tenderWork = $voucher->tenderAllocation->tenderEntry->tenderWork;

                    // Add tender work details to the response
                    $data['tender_work_from_allocation'] = [
                        'tend_work_id' => $tenderWork->tend_work_id,
                        'work_code' => $tenderWork->work_code,
                        'work_name' => $tenderWork->work_name,
                        'created_at' => $tenderWork->created_at ? $tenderWork->created_at->format('d-m-Y H:i:s') : null,
                        'updated_at' => $tenderWork->updated_at ? $tenderWork->updated_at->format('d-m-Y H:i:s') : null,
                        'deleted_at' => $tenderWork->deleted_at ? $tenderWork->deleted_at->format('d-m-Y H:i:s') : null,
                    ];
                } else {
                    $data['tender_work_from_allocation'] = null;
                }

                return $data;
            });

            // Replace the collection with transformed data
            $paymentVouchers->setCollection($transformedData);

            return response()->json([
                'success' => true,
                'data' => $paymentVouchers->items(),
                'meta' => [
                    'current_page' => $paymentVouchers->currentPage(),
                    'last_page' => $paymentVouchers->lastPage(),
                    'per_page' => $paymentVouchers->perPage(),
                    'total' => $paymentVouchers->total(),
                ]
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch payment vouchers',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), PaymentVoucher::createRules(), PaymentVoucher::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->all();

            // If Pay_Vou_No is not provided, it will be auto-generated in the model
            if (empty($data['Pay_Vou_No'])) {
                unset($data['Pay_Vou_No']);
            }

            // If tender work is provided, get vendor invoice parameters
            if (!empty($data['Tend_Work_Id']) && empty($data['tax_parameters'])) {
                $taxParameters = $this->getTaxParametersForTenderWork($data['Tend_Work_Id']);
                if ($taxParameters) {
                    $data['tax_parameters'] = $taxParameters;
                }
            }

            // Create payment voucher
            $paymentVoucher = PaymentVoucher::create($data);

            // Load relationships for response
            $paymentVoucher->load(['branch', 'ledger', 'subLedger', 'tenderAllocation', 'tenderWork']);

            return response()->json([
                'success' => true,
                'message' => 'Payment voucher created successfully',
                'data' => $paymentVoucher
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create payment voucher',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
   public function show($id)
{
    try {
        $paymentVoucher = PaymentVoucher::with([
            'branch',
            'ledger',
            'subLedger',
            'tenderAllocation' => function($q) {
                $q->with(['tenderEntry' => function($q2) {
                    $q2->with('tenderWork');
                }]);
            },
            'tenderWork'
        ])->find($id);

        if (!$paymentVoucher) {
            return response()->json([
                'success' => false,
                'message' => 'Payment voucher not found'
            ], 404);
        }

        // Transform the response to include tender work details from tender allocation
        $data = $paymentVoucher->toArray();

        // Get tender work details from tender allocation if available
        if ($paymentVoucher->tenderAllocation &&
            $paymentVoucher->tenderAllocation->tenderEntry &&
            $paymentVoucher->tenderAllocation->tenderEntry->tenderWork) {

            $tenderWork = $paymentVoucher->tenderAllocation->tenderEntry->tenderWork;

            // Add tender work details to the response
            $data['tender_work_from_allocation'] = [
                'tend_work_id' => $tenderWork->tend_work_id,
                'work_code' => $tenderWork->work_code,
                'work_name' => $tenderWork->work_name,
                'created_at' => $tenderWork->created_at ? $tenderWork->created_at->format('d-m-Y H:i:s') : null,
                'updated_at' => $tenderWork->updated_at ? $tenderWork->updated_at->format('d-m-Y H:i:s') : null,
                'deleted_at' => $tenderWork->deleted_at ? $tenderWork->deleted_at->format('d-m-Y H:i:s') : null,
            ];
        } else {
            $data['tender_work_from_allocation'] = null;
        }

        // NEW: Get ledger data based on subledger's Lg_ID
        if ($paymentVoucher->subLedger && $paymentVoucher->subLedger->Lg_ID) {
            $subLedgerLedger = \App\Models\LedgerMaster::find($paymentVoucher->subLedger->Lg_ID);
            if ($subLedgerLedger) {
                // Add this ledger data to the response
                $data['sub_ledger_ledger'] = $subLedgerLedger;
            }
        }

        // Get vendor invoice parameters and proposal_id
        $tendWorkId = null;
        $proposalId = null;
        $vendorParameters = null;

        // Get tend_work_id from either direct field or allocation
        if ($paymentVoucher->Tend_Work_Id) {
            $tendWorkId = $paymentVoucher->Tend_Work_Id;
        } elseif ($paymentVoucher->tenderAllocation &&
                  $paymentVoucher->tenderAllocation->tenderEntry &&
                  $paymentVoucher->tenderAllocation->tenderEntry->tend_work_id) {
            $tendWorkId = $paymentVoucher->tenderAllocation->tenderEntry->tend_work_id;
        }

        // Get vendor invoice parameters if tend_work_id is available
        if ($tendWorkId) {
            $vendorParams = VendorInvoiceParameter::with(['tenderWork'])
                ->where('tend_work_id', $tendWorkId)
                ->whereDate('effective_date', '<=', $paymentVoucher->bill_challan_date)
                ->whereNull('deleted_at')
                ->orderBy('effective_date', 'desc')
                ->first();

            if ($vendorParams) {
                $proposalId = $vendorParams->proposal_id;

                // Transform vendor parameters
                $taxParameters = [];
                if (is_array($vendorParams->tax_parameters)) {
                    foreach ($vendorParams->tax_parameters as $param) {
                        $taxParameters[] = [
                            'tax_type_id' => $param['tax_type_id'] ?? null,
                            'tax_code' => $param['tax_code'] ?? null,
                            'add_deduct' => $param['add_deduct'] ?? 0,
                            'add_deduct_text' => \App\Models\VendorInvoiceParameter::ADD_DEDUCT_TEXTS[$param['add_deduct'] ?? 0] ?? 'Unknown',
                            'Lg_Id' => $param['Lg_Id'] ?? null,
                            'percentage' => isset($param['percentage']) ? (float)$param['percentage'] : null,
                            'percentage_formatted' => isset($param['percentage']) ? number_format($param['percentage'], 2) . '%' : null,
                            'effective_percentage' => isset($param['percentage']) ?
                                (($param['add_deduct'] == \App\Models\VendorInvoiceParameter::DEDUCT ? -$param['percentage'] : $param['percentage'])) : 0,
                            'calculation_formula' => $param['calculation_formula'] ?? null,
                        ];
                    }
                }

                $vendorParameters = [
                    'proposal_id' => $vendorParams->proposal_id,
                    'tend_work_id' => $vendorParams->tend_work_id,
                    'tender_work' => $vendorParams->tenderWork ? [
                        'tend_work_id' => $vendorParams->tenderWork->tend_work_id,
                        'work_code' => $vendorParams->tenderWork->work_code,
                        'work_name' => $vendorParams->tenderWork->work_name
                    ] : null,
                    'effective_date' => $vendorParams->effective_date ? $vendorParams->effective_date->format('Y-m-d') : null,
                    'effective_date_formatted' => $vendorParams->effective_date ? $vendorParams->effective_date->format('d-m-Y') : null,
                    'tax_parameters' => $taxParameters,
                    'total_parameters' => $vendorParams->total_parameters,
                    'total_percentage' => $vendorParams->total_percentage,
                    'total_percentage_formatted' => number_format($vendorParams->total_percentage, 2) . '%',
                    'created_at' => $vendorParams->created_at ? $vendorParams->created_at->format('Y-m-d H:i:s') : null,
                    'updated_at' => $vendorParams->updated_at ? $vendorParams->updated_at->format('Y-m-d H:i:s') : null,
                    'deleted_at' => $vendorParams->deleted_at ? $vendorParams->deleted_at->format('Y-m-d H:i:s') : null,
                    'is_deleted' => !is_null($vendorParams->deleted_at),
                    'is_active' => is_null($vendorParams->deleted_at)
                ];
            }
        }

        // Add proposal_id to main data
        $data['proposal_id'] = $proposalId;

        // Add vendor invoice parameters to data
        if ($vendorParameters) {
            $data['vendor_invoice_parameters'] = $vendorParameters;
        }

        return response()->json([
            'success' => true,
            'data' => $data
        ], 200);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to fetch payment voucher',
            'error' => $e->getMessage()
        ], 500);
    }
}

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $paymentVoucher = PaymentVoucher::find($id);

            if (!$paymentVoucher) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment voucher not found'
                ], 404);
            }

            // Check if voucher can be edited
            if (!$paymentVoucher->canBeEdited()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment voucher cannot be edited in current status'
                ], 400);
            }

            $validator = Validator::make(
                $request->all(),
                PaymentVoucher::updateRules($id),
                PaymentVoucher::validationMessages()
            );

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->all();

            // Update payment voucher
            $paymentVoucher->update($data);
            $paymentVoucher->refresh();

            // Load relationships for response
            $paymentVoucher->load(['branch', 'ledger', 'subLedger', 'tenderAllocation', 'tenderWork']);

            return response()->json([
                'success' => true,
                'message' => 'Payment voucher updated successfully',
                'data' => $paymentVoucher
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update payment voucher',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $paymentVoucher = PaymentVoucher::find($id);

            if (!$paymentVoucher) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment voucher not found'
                ], 404);
            }

            // Check if can be deleted
            if (!$paymentVoucher->canBeDeleted()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment voucher cannot be deleted in current status'
                ], 400);
            }

            $paymentVoucher->delete();

            return response()->json([
                'success' => true,
                'message' => 'Payment voucher deleted successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete payment voucher',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Restore soft deleted payment voucher.
     */
    public function restore($id)
    {
        try {
            $paymentVoucher = PaymentVoucher::withTrashed()->find($id);

            if (!$paymentVoucher) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment voucher not found'
                ], 404);
            }

            if (!$paymentVoucher->trashed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment voucher is not deleted'
                ], 400);
            }

            $paymentVoucher->restore();
            $paymentVoucher->load(['branch', 'ledger', 'subLedger', 'tenderAllocation', 'tenderWork']);

            return response()->json([
                'success' => true,
                'message' => 'Payment voucher restored successfully',
                'data' => $paymentVoucher
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to restore payment voucher',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get payment voucher details with formatted data.
     */
    public function getDetails($id)
    {
        try {
            $paymentVoucher = PaymentVoucher::with([
                'branch',
                'ledger',
                'subLedger',
                'tenderAllocation',
                'tenderWork'
            ])->find($id);

            if (!$paymentVoucher) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment voucher not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $paymentVoucher->getPaymentDetails()
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch payment voucher details',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get summary statistics.
     */
    public function getSummary(Request $request)
    {
        try {
            $query = PaymentVoucher::query();

            if ($request->has('branch_id')) {
                $query->byBranch($request->branch_id);
            }

            if ($request->has('payment_status')) {
                $query->byPaymentStatus($request->payment_status);
            }

            if ($request->has('start_date') && $request->has('end_date')) {
                $query->byDateRange($request->start_date, $request->end_date);
            }

            $summary = [
                'total_vouchers' => $query->count(),
                'total_basic_amount' => $query->sum('basic_amount'),
                'total_net_amount' => $query->sum('net_amount'),
                'total_tax_amount' => $query->sum('total_tax_amount'),
                'average_net_amount' => $query->avg('net_amount'),
                'max_net_amount' => $query->max('net_amount'),
                'min_net_amount' => $query->min('net_amount'),
                'status_counts' => $query->groupBy('payment_status')
                    ->selectRaw('payment_status, count(*) as count')
                    ->pluck('count', 'payment_status')
            ];

            return response()->json([
                'success' => true,
                'data' => $summary
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch summary',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get tax parameters for a tender work
     */
    private function getTaxParametersForTenderWork($tenderWorkId)
    {
        try {
            // Get latest vendor invoice parameters for the tender work
            $parameters = VendorInvoiceParameter::with(['tenderWork'])
                ->where('tend_work_id', $tenderWorkId)
                ->whereNull('deleted_at')
                ->orderBy('effective_date', 'desc')
                ->first();

            if (!$parameters || !is_array($parameters->tax_parameters)) {
                return [];
            }

            $taxParameters = [];
            foreach ($parameters->tax_parameters as $param) {
                $taxParameters[] = [
                    'tax_type' => $param['tax_type_id'] ?? '',
                    'percentage' => $param['percentage'] ?? 0,
                    'add_deduct' => $param['add_deduct'] ?? 0,
                    'formula' => $param['calculation_formula'] ?? '',
                    'ledger_id' => $param['Lg_Id'] ?? null,
                    'tax_code' => $param['tax_code'] ?? ''
                ];
            }

            return $taxParameters;
        } catch (\Exception $e) {
            return [];
        }
    }

    /**
     * Approve a payment voucher - UPDATED: नए status check
     */
    public function approve($id)
    {
        try {
            $paymentVoucher = PaymentVoucher::find($id);

            if (!$paymentVoucher) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment voucher not found'
                ], 404);
            }

            // hod_app और acco_app भी जोड़े approval के लिए
            if (!in_array($paymentVoucher->payment_status, ['pending', 'quary', 'hod_app', 'acco_app'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment voucher cannot be approved in current status'
                ], 400);
            }

            $paymentVoucher->approve();
            $paymentVoucher->load(['branch', 'ledger', 'subLedger', 'tenderAllocation', 'tenderWork']);

            return response()->json([
                'success' => true,
                'message' => 'Payment voucher approved successfully',
                'data' => $paymentVoucher
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to approve payment voucher',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mark payment voucher as quary
     */
    public function markAsQuary($id)
    {
        try {
            $paymentVoucher = PaymentVoucher::find($id);

            if (!$paymentVoucher) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment voucher not found'
                ], 404);
            }

            $paymentVoucher->markAsQuary();
            $paymentVoucher->load(['branch', 'ledger', 'subLedger', 'tenderAllocation', 'tenderWork']);

            return response()->json([
                'success' => true,
                'message' => 'Payment voucher marked as quary successfully',
                'data' => $paymentVoucher
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to mark payment voucher as quary',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mark payment voucher as paid
     */
    public function markAsPaid($id)
    {
        try {
            $paymentVoucher = PaymentVoucher::find($id);

            if (!$paymentVoucher) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment voucher not found'
                ], 404);
            }

            if ($paymentVoucher->payment_status !== 'approved') {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment voucher must be approved before marking as paid'
                ], 400);
            }

            $paymentVoucher->markAsPaid();
            $paymentVoucher->load(['branch', 'ledger', 'subLedger', 'tenderAllocation', 'tenderWork']);

            return response()->json([
                'success' => true,
                'message' => 'Payment voucher marked as paid successfully',
                'data' => $paymentVoucher
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to mark payment voucher as paid',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get tender allocations by vendor with tender details including vendor invoice parameters
     */
    public function getTenderAllocationsByVendor(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'vendor_id' => 'required|integer|exists:subledg_master,SL_Id'
            ], [
                'vendor_id.required' => 'Vendor ID is required',
                'vendor_id.exists' => 'Vendor does not exist'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $vendorId = $request->vendor_id;

            // Get vendor details
            $vendor = SubLedgerMaster::find($vendorId);

            if (!$vendor) {
                return response()->json([
                    'success' => false,
                    'message' => 'Vendor not found'
                ], 404);
            }

            // Get all tender allocations for this vendor with proper relationships
            $tenderAllocations = TendorAllocation::with([
                'tenderEntry' => function ($query) {
                    $query->with([
                        'department',
                        'ledger',
                        'tenderWork',
                        'allDocuments'
                    ]);
                },
                'subLedger',
                'documents'
            ])
                ->where('SL_Id', $vendorId)
                ->get();

            // Format the response
            $response = [
                'vendor' => [
                    'id' => $vendor->SL_Id,
                    'code' => $vendor->SL_Code,
                    'name' => $vendor->SL_Name,
                    'total_tender_allocations' => $tenderAllocations->count()
                ],
                'tender_allocations' => $tenderAllocations->map(function ($allocation) {
                    $tenderEntry = $allocation->tenderEntry;

                    // Get tender work details using the correct field names
                    $tenderWorkDetails = null;
                    if ($tenderEntry && $tenderEntry->tenderWork) {
                        $tenderWork = $tenderEntry->tenderWork;
                        $tenderWorkDetails = [
                            'tender_work_id' => $tenderWork->tend_work_id,
                            'work_code' => $tenderWork->work_code,
                            'work_name' => $tenderWork->work_name,
                            'created_at' => $tenderWork->created_at ? $tenderWork->created_at->format('d-m-Y H:i:s') : null,
                            'updated_at' => $tenderWork->updated_at ? $tenderWork->updated_at->format('d-m-Y H:i:s') : null,
                            'deleted_at' => $tenderWork->deleted_at ? $tenderWork->deleted_at->format('d-m-Y H:i:s') : null,
                        ];
                    }

                    // Get vendor invoice parameters for this tender work
                    $vendorInvoiceParameters = [];
                    if ($tenderEntry && $tenderEntry->tend_work_id) {
                        try {
                            // Get all vendor invoice parameters for this tender work
                            $parameters = \App\Models\VendorInvoiceParameter::with(['tenderWork'])
                                ->where('tend_work_id', $tenderEntry->tend_work_id)
                                ->whereNull('deleted_at')
                                ->orderBy('effective_date', 'desc')
                                ->get();

                            $vendorInvoiceParameters = $parameters->map(function ($param) {
                                $transformedParams = [];

                                // Transform each tax parameter in the array
                                if (is_array($param->tax_parameters)) {
                                    foreach ($param->tax_parameters as $taxParam) {
                                        $transformedParams[] = \App\Models\VendorInvoiceParameter::transformParameterItem($taxParam);
                                    }
                                }

                                return [
                                    'proposal_id' => $param->proposal_id,
                                    'tend_work_id' => $param->tend_work_id,
                                    'effective_date' => $param->effective_date ? $param->effective_date->format('d-m-Y') : null,
                                    'total_parameters' => $param->total_parameters,
                                    'total_percentage' => $param->total_percentage,
                                    'formatted_total_percentage' => number_format($param->total_percentage, 2) . '%',
                                    'is_active' => $param->is_active,
                                    'created_at' => $param->created_at ? $param->created_at->format('d-m-Y H:i:s') : null,
                                    'updated_at' => $param->updated_at ? $param->updated_at->format('d-m-Y H:i:s') : null,
                                    'tax_parameters' => $transformedParams,
                                    'tender_work' => $param->tenderWork ? [
                                        'work_code' => $param->tenderWork->work_code,
                                        'work_name' => $param->tenderWork->work_name
                                    ] : null
                                ];
                            })->toArray();
                        } catch (\Exception $e) {
                            // Log error if needed
                        }
                    }

                    // Get tender entry documents
                    $tenderEntryDocuments = [];
                    if ($tenderEntry && $tenderEntry->allDocuments) {
                        $tenderEntryDocuments = $tenderEntry->allDocuments->map(function ($doc) {
                            return [
                                'document_id' => $doc->Doc_Id ?? $doc->Doc_Lib,
                                'document_name' => $doc->Document_Name,
                                'file_path' => $doc->File_Path,
                                'serial_no' => $doc->Serial_No
                            ];
                        })->filter(function ($doc) {
                            return $doc['document_name'] !== null;
                        })->values()->toArray();
                    }

                    // Get tender allocation documents
                    $allocationDocuments = [];
                    if ($allocation->documents && $allocation->documents->count() > 0) {
                        $allocationDocuments = $allocation->documents->map(function ($doc) {
                            return [
                                'document_id' => $doc->Doc_Id ?? $doc->Doc_Lib,
                                'document_name' => $doc->Document_Name,
                                'file_path' => $doc->File_Path,
                                'serial_no' => $doc->Serial_No
                            ];
                        })->filter(function ($doc) {
                            return $doc['document_name'] !== null;
                        })->values()->toArray();
                    }

                    return [
                        'tender_allocation_id' => $allocation->Tend_Alloc,
                        'gross_amount' => $allocation->Gross_Amount,
                        'formatted_gross_amount' => $allocation->Gross_Amount ? number_format($allocation->Gross_Amount, 2) : '0.00',
                        'multi_doc' => $allocation->Multi_Doc,
                        'created_at' => $allocation->created_at ? $allocation->created_at->format('d-m-Y H:i:s') : null,
                        'tender_entry' => $tenderEntry ? [
                            'tender_entry_id' => $tenderEntry->Tend_Ent_Id,
                            'branch_id' => $tenderEntry->Bra_id,
                            'branch_name' => $tenderEntry->department->desc ?? null,
                            'ledger_id' => $tenderEntry->Lg_Id,
                            'ledger_name' => $tenderEntry->ledger->Lg_Name ?? null,
                            'tender_work_id' => $tenderEntry->tend_work_id,
                            'tender_work_details' => $tenderWorkDetails,
                            'proposal_amount' => $tenderEntry->Proposal_Amount,
                            'formatted_proposal_amount' => $tenderEntry->Proposal_Amount ? number_format($tenderEntry->Proposal_Amount, 2) : null,
                            'date' => $tenderEntry->Date ? $tenderEntry->Date->format('d-m-Y') : null,
                            'description' => $tenderEntry->Description,
                            'doc_lib' => $tenderEntry->Doc_Lib,
                            'documents' => $tenderEntryDocuments,
                            'total_documents' => count($tenderEntryDocuments)
                        ] : null,
                        'vendor_invoice_parameters' => $vendorInvoiceParameters,
                        'total_vendor_parameters' => count($vendorInvoiceParameters),
                        'sub_ledger' => $allocation->subLedger ? [
                            'id' => $allocation->subLedger->SL_Id,
                            'code' => $allocation->subLedger->SL_Code,
                            'name' => $allocation->subLedger->SL_Name
                        ] : null,
                        'documents' => $allocationDocuments,
                        'total_documents' => count($allocationDocuments)
                    ];
                })
            ];

            return response()->json([
                'success' => true,
                'message' => 'Tender allocations fetched successfully',
                'data' => $response
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch tender allocations',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate a new voucher number
     */
    public function generateVoucherNumber(Request $request)
    {
        try {
            $voucherNumber = PaymentVoucher::generateVoucherNumber();

            return response()->json([
                'success' => true,
                'data' => [
                    'voucher_number' => $voucherNumber
                ]
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to generate voucher number',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get tax parameters by tender work ID
     */
    public function getTaxParametersByTenderWork($tenderWorkId)
    {
        try {
            $taxParameters = $this->getTaxParametersForTenderWork($tenderWorkId);

            return response()->json([
                'success' => true,
                'data' => [
                    'tender_work_id' => $tenderWorkId,
                    'tax_parameters' => $taxParameters,
                    'total_parameters' => count($taxParameters)
                ]
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve tax parameters',
                'error' => $e->getMessage()
            ], 500);
        }
    }


   // PaymentVoucherController.php में update करें

/**
 * Get tender entries by branch with vendor allocations - WITH LEDGER FILTER
 */
public function getTenderEntriesByBranchWithVendorAllocations(Request $request)
{
    try {
        $validator = Validator::make($request->all(), [
            'branch_id' => 'required|integer|exists:main_departments,id',
            'ledger_id' => 'nullable|integer|exists:ledger_master,Lg_Id' // NEW: ledger filter
        ], [
            'branch_id.required' => 'Branch ID is required',
            'branch_id.exists' => 'Branch does not exist',
            'ledger_id.exists' => 'Ledger does not exist' // NEW
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $branchId = $request->branch_id;
        $ledgerId = $request->ledger_id;

        // Get all tender entries for this branch
        $query = TenderEntryMaster::with([
            'department',
            'ledger',
            'tenderWork',
            'uploadedDocuments'
        ])
            ->where('Bra_id', $branchId);

        // NEW: Add ledger filter if provided
        if ($ledgerId) {
            $query->where('Lg_Id', $ledgerId);
        }

        $tenderEntries = $query->get();

        // Format the response with vendor allocations
        $formattedEntries = $tenderEntries->map(function ($entry) {
            // Get vendor allocations for this tender entry
            $vendorAllocations = TendorAllocation::with([
                'subLedger',
                'documents'
            ])
                ->where('Tend_Ent_Id', $entry->Tend_Ent_Id)
                ->get();

            // Get vendor invoice parameters for this tender work
            $vendorInvoiceParameters = [];
            if ($entry->tend_work_id) {
                $parameters = VendorInvoiceParameter::with(['tenderWork'])
                    ->where('tend_work_id', $entry->tend_work_id)
                    ->whereNull('deleted_at')
                    ->orderBy('effective_date', 'desc')
                    ->get();

                $vendorInvoiceParameters = $parameters->map(function ($param) {
                    $transformedParams = [];
                    if (is_array($param->tax_parameters)) {
                        foreach ($param->tax_parameters as $taxParam) {
                            $transformedParams[] = \App\Models\VendorInvoiceParameter::transformParameterItem($taxParam);
                        }
                    }

                    return [
                        'proposal_id' => $param->proposal_id,
                        'effective_date' => $param->effective_date ? $param->effective_date->format('d-m-Y') : null,
                        'total_parameters' => $param->total_parameters,
                        'total_percentage' => $param->total_percentage,
                        'formatted_total_percentage' => number_format($param->total_percentage, 2) . '%',
                        'tax_parameters' => $transformedParams
                    ];
                })->toArray();
            }

            return [
                'tender_entry_id' => $entry->Tend_Ent_Id,
                'branch_id' => $entry->Bra_id,
                'branch_name' => $entry->department->desc ?? null,
                'branch_code' => $entry->department->Grcod ?? null,
                'ledger_id' => $entry->Lg_Id,
                'ledger_code' => $entry->ledger->Lg_Code ?? null,
                'ledger_name' => $entry->ledger->Lg_Name ?? null,
                'tender_work_id' => $entry->tend_work_id,
                'tender_work_details' => $entry->tenderWork ? [
                    'work_code' => $entry->tenderWork->work_code,
                    'work_name' => $entry->tenderWork->work_name
                ] : null,
                'proposal_amount' => $entry->Proposal_Amount,
                'formatted_proposal_amount' => $entry->Proposal_Amount ? number_format($entry->Proposal_Amount, 2) : '0.00',
                'date' => $entry->Date ? $entry->Date->format('d-m-Y') : null,
                'description' => $entry->Description,
                'documents' => $entry->uploadedDocuments->map(function ($doc) {
                    return [
                        'document_id' => $doc->Doc_Id ?? $doc->Doc_Lib,
                        'document_name' => $doc->Document_Name,
                        'file_path' => $doc->File_Path,
                        'serial_no' => $doc->Serial_No
                    ];
                })->toArray(),
                'vendor_allocations' => $vendorAllocations->map(function ($allocation) {
                    return [
                        'allocation_id' => $allocation->Tend_Alloc,
                        'vendor_id' => $allocation->SL_Id,
                        'vendor_details' => $allocation->subLedger ? [
                            'vendor_code' => $allocation->subLedger->SL_Code,
                            'vendor_name' => $allocation->subLedger->SL_Name
                        ] : null,
                        'gross_amount' => $allocation->Gross_Amount,
                        'formatted_gross_amount' => $allocation->Gross_Amount ? number_format($allocation->Gross_Amount, 2) : '0.00',
                        'multi_doc' => $allocation->Multi_Doc,
                        'allocation_documents' => $allocation->documents->map(function ($doc) {
                            return [
                                'document_id' => $doc->Doc_Id ?? $doc->Doc_Lib,
                                'document_name' => $doc->Document_Name,
                                'file_path' => $doc->File_Path,
                                'serial_no' => $doc->Serial_No
                            ];
                        })->toArray(),
                        'created_at' => $allocation->created_at ? $allocation->created_at->format('d-m-Y H:i:s') : null
                    ];
                })->toArray(),
                'vendor_invoice_parameters' => $vendorInvoiceParameters,
                'has_vendor_allocations' => $vendorAllocations->count() > 0,
                'total_vendor_allocations' => $vendorAllocations->count(),
                'created_at' => $entry->created_at ? $entry->created_at->format('d-m-Y H:i:s') : null
            ];
        });

        return response()->json([
            'success' => true,
            'message' => 'Tender entries with vendor allocations fetched successfully',
            'data' => [
                'branch_id' => $branchId,
                'ledger_id' => $ledgerId,
                'total_tender_entries' => $tenderEntries->count(),
                'tender_entries' => $formattedEntries
            ]
        ], 200);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to fetch tender entries',
            'error' => $e->getMessage()
        ], 500);
    }
}




}
