<?php

namespace App\Http\Controllers;

use App\Models\PayrollMaster;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class PayrollMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = PayrollMaster::query();

        // Filter by ledger if provided
        if ($request->has('lg_id')) {
            $query->where('Lg_Id', $request->lg_id);
        }

        // Filter by payroll code if provided
        if ($request->has('payroll_code')) {
            $query->where('payroll_code', 'like', '%' . $request->payroll_code . '%');
        }

        // Filter by payroll name if provided
        if ($request->has('payroll_name')) {
            $query->where('payroll_name', 'like', '%' . $request->payroll_name . '%');
        }

        $payrolls = $query->with('ledger')->get();

        return response()->json([
            'success' => true,
            'data' => $payrolls
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate(PayrollMaster::createRules(), PayrollMaster::validationMessages());

        $payroll = PayrollMaster::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'Payroll created successfully',
            'data' => $payroll->load('ledger')
        ], Response::HTTP_CREATED);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $payroll = PayrollMaster::with('ledger')->find($id);

        if (!$payroll) {
            return response()->json([
                'success' => false,
                'message' => 'Payroll not found'
            ], Response::HTTP_NOT_FOUND);
        }

        return response()->json([
            'success' => true,
            'data' => $payroll
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $payroll = PayrollMaster::find($id);

        if (!$payroll) {
            return response()->json([
                'success' => false,
                'message' => 'Payroll not found'
            ], Response::HTTP_NOT_FOUND);
        }

        $validated = $request->validate(PayrollMaster::updateRules($id), PayrollMaster::validationMessages());

        $payroll->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Payroll updated successfully',
            'data' => $payroll->load('ledger')
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $payroll = PayrollMaster::find($id);

        if (!$payroll) {
            return response()->json([
                'success' => false,
                'message' => 'Payroll not found'
            ], Response::HTTP_NOT_FOUND);
        }

        $payroll->delete();

        return response()->json([
            'success' => true,
            'message' => 'Payroll deleted successfully'
        ]);
    }

    /**
     * Get payrolls by ledger ID
     */
    public function getByLedger($ledgerId)
    {
        $payrolls = PayrollMaster::where('Lg_Id', $ledgerId)
            ->with('ledger')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $payrolls
        ]);
    }

    /**
     * Get payrolls without ledger association
     */
    public function getWithoutLedger()
    {
        $payrolls = PayrollMaster::whereNull('Lg_Id')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $payrolls
        ]);
    }

    /**
     * Bulk update ledger for multiple payrolls
     */
    public function bulkUpdateLedger(Request $request)
    {
        $validated = $request->validate([
            'payroll_ids' => 'required|array',
            'payroll_ids.*' => 'exists:payroll_master,payroll_id',
            'Lg_Id' => 'nullable|integer|exists:ledger_master,Lg_Id'
        ]);

        PayrollMaster::whereIn('payroll_id', $validated['payroll_ids'])
            ->update(['Lg_Id' => $validated['Lg_Id']]);

        return response()->json([
            'success' => true,
            'message' => 'Ledger updated for selected payrolls'
        ]);
    }
}
