<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\PermissionRole;
use App\Models\RoleMaster;
use App\Models\Permission;
use App\Models\PermissionType;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class PermissionRoleController extends Controller
{
    /**
     * Display a listing of permission-role assignments
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $query = PermissionRole::with([
                'permission', 
                'role' => function($q) {
                    $q->select('Role_Id', 'Role_Name', 'Role_Prefix');
                }, 
                'permissionType'
            ]);

            // Filter by role_id if provided
            if ($request->has('role_id')) {
                $query->where('role_id', $request->role_id);
            }

            // Filter by permission_id if provided
            if ($request->has('permission_id')) {
                $query->where('permission_id', $request->permission_id);
            }

            $permissionRoles = $query->get();

            return response()->json([
                'success' => true,
                'data' => $permissionRoles,
                'message' => 'Permission roles retrieved successfully'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve permission roles',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created permission-role assignment
     */
    public function store(Request $request): JsonResponse
    {
        // Use manual validation instead of exists rules
        $validator = Validator::make($request->all(), [
            'permission_id' => 'required|integer',
            'role_id' => 'required|integer',
            'permission_type_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Manual validation for role_id
            $roleExists = DB::table('role_master')
                ->where('Role_Id', $request->role_id)
                ->exists();
            
            if (!$roleExists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => [
                        'role_id' => ['The selected role does not exist in role master']
                    ]
                ], 422);
            }

            // Manual validation for permission_id
            $permissionExists = DB::table('permissions')
                ->where('id', $request->permission_id)
                ->exists();
            
            if (!$permissionExists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => [
                        'permission_id' => ['The selected permission does not exist']
                    ]
                ], 422);
            }

            // Manual validation for permission_type_id
            $permissionTypeExists = DB::table('permission_types')
                ->where('id', $request->permission_type_id)
                ->exists();
            
            if (!$permissionTypeExists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => [
                        'permission_type_id' => ['The selected permission type does not exist']
                    ]
                ], 422);
            }

            // Check if assignment already exists
            $exists = PermissionRole::where('permission_id', $request->permission_id)
                ->where('role_id', $request->role_id)
                ->exists();
            
            if ($exists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Permission already assigned to this role'
                ], 409);
            }

            $permissionRole = PermissionRole::create($request->all());

            return response()->json([
                'success' => true,
                'data' => $permissionRole->load([
                    'permission',
                    'role' => function($q) {
                        $q->select('Role_Id', 'Role_Name', 'Role_Prefix');
                    },
                    'permissionType'
                ]),
                'message' => 'Permission assigned to role successfully'
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to assign permission to role',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified permission-role assignment
     */
    public function show($permissionId, $roleId): JsonResponse
    {
        try {
            $permissionRole = PermissionRole::with([
                'permission',
                'role' => function($q) {
                    $q->select('Role_Id', 'Role_Name', 'Role_Prefix');
                },
                'permissionType'
            ])
                ->where('permission_id', $permissionId)
                ->where('role_id', $roleId)
                ->first();

            if (!$permissionRole) {
                return response()->json([
                    'success' => false,
                    'message' => 'Permission-role assignment not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $permissionRole,
                'message' => 'Permission-role assignment retrieved successfully'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve permission-role assignment',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified permission-role assignment
     */
    public function update(Request $request, $permissionId, $roleId): JsonResponse
    {
        // Use manual validation
        $validator = Validator::make($request->all(), [
            'permission_type_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Manual validation for permission_type_id
            $permissionTypeExists = DB::table('permission_types')
                ->where('id', $request->permission_type_id)
                ->exists();
            
            if (!$permissionTypeExists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => [
                        'permission_type_id' => ['The selected permission type does not exist']
                    ]
                ], 422);
            }

            $permissionRole = PermissionRole::where('permission_id', $permissionId)
                ->where('role_id', $roleId)
                ->first();

            if (!$permissionRole) {
                return response()->json([
                    'success' => false,
                    'message' => 'Permission-role assignment not found'
                ], 404);
            }

            $permissionRole->update([
                'permission_type_id' => $request->permission_type_id
            ]);

            return response()->json([
                'success' => true,
                'data' => $permissionRole->load([
                    'permission',
                    'role' => function($q) {
                        $q->select('Role_Id', 'Role_Name', 'Role_Prefix');
                    },
                    'permissionType'
                ]),
                'message' => 'Permission-role assignment updated successfully'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update permission-role assignment',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified permission-role assignment
     */
    public function destroy($permissionId, $roleId): JsonResponse
    {
        try {
            $permissionRole = PermissionRole::where('permission_id', $permissionId)
                ->where('role_id', $roleId)
                ->first();

            if (!$permissionRole) {
                return response()->json([
                    'success' => false,
                    'message' => 'Permission-role assignment not found'
                ], 404);
            }

            $permissionRole->delete();

            return response()->json([
                'success' => true,
                'message' => 'Permission-role assignment removed successfully'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to remove permission-role assignment',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk assign permissions to a role
     */
    public function bulkAssign(Request $request): JsonResponse
    {
        // Use manual validation
        $validator = Validator::make($request->all(), [
            'role_id' => 'required|integer',
            'permissions' => 'required|array',
            'permissions.*.permission_id' => 'required|integer',
            'permissions.*.permission_type_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Manual validation for role_id
            $roleExists = DB::table('role_master')
                ->where('Role_Id', $request->role_id)
                ->exists();
            
            if (!$roleExists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => [
                        'role_id' => ['The selected role does not exist in role master']
                    ]
                ], 422);
            }

            DB::beginTransaction();

            $roleId = $request->role_id;
            $results = [
                'assigned' => 0,
                'skipped' => 0,
                'failed' => 0,
                'errors' => []
            ];

            foreach ($request->permissions as $index => $permissionData) {
                try {
                    // Manual validation for permission_id
                    $permissionExists = DB::table('permissions')
                        ->where('id', $permissionData['permission_id'])
                        ->exists();
                    
                    if (!$permissionExists) {
                        $results['failed']++;
                        $results['errors'][] = [
                            'index' => $index,
                            'permission_id' => $permissionData['permission_id'],
                            'error' => 'Permission does not exist'
                        ];
                        continue;
                    }

                    // Manual validation for permission_type_id
                    $permissionTypeExists = DB::table('permission_types')
                        ->where('id', $permissionData['permission_type_id'])
                        ->exists();
                    
                    if (!$permissionTypeExists) {
                        $results['failed']++;
                        $results['errors'][] = [
                            'index' => $index,
                            'permission_id' => $permissionData['permission_id'],
                            'error' => 'Permission type does not exist'
                        ];
                        continue;
                    }

                    // Check if already exists
                    $exists = PermissionRole::where('permission_id', $permissionData['permission_id'])
                        ->where('role_id', $roleId)
                        ->exists();

                    if ($exists) {
                        $results['skipped']++;
                        continue;
                    }

                    PermissionRole::create([
                        'permission_id' => $permissionData['permission_id'],
                        'role_id' => $roleId,
                        'permission_type_id' => $permissionData['permission_type_id']
                    ]);

                    $results['assigned']++;

                } catch (\Exception $e) {
                    $results['failed']++;
                    $results['errors'][] = [
                        'index' => $index,
                        'permission_id' => $permissionData['permission_id'],
                        'error' => $e->getMessage()
                    ];
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'data' => $results,
                'message' => 'Bulk assignment completed'
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Bulk assignment failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get permissions by role
     */
    public function permissionsByRole($roleId): JsonResponse
    {
        try {
            // Manual validation for role_id
            $roleExists = DB::table('role_master')
                ->where('Role_Id', $roleId)
                ->exists();
            
            if (!$roleExists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role not found'
                ], 404);
            }

            $permissions = PermissionRole::with(['permission', 'permissionType'])
                ->where('role_id', $roleId)
                ->get()
                ->map(function ($item) {
                    return [
                        'permission' => $item->permission,
                        'permission_type' => $item->permissionType,
                        'permission_type_id' => $item->permission_type_id
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $permissions,
                'message' => 'Permissions retrieved successfully for role'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve permissions for role',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get roles by permission
     */
    public function rolesByPermission($permissionId): JsonResponse
    {
        try {
            // Manual validation for permission_id
            $permissionExists = DB::table('permissions')
                ->where('id', $permissionId)
                ->exists();
            
            if (!$permissionExists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Permission not found'
                ], 404);
            }

            $roles = PermissionRole::with([
                'role' => function($q) {
                    $q->select('Role_Id', 'Role_Name', 'Role_Prefix');
                },
                'permissionType'
            ])
                ->where('permission_id', $permissionId)
                ->get()
                ->map(function ($item) {
                    return [
                        'role' => $item->role,
                        'permission_type' => $item->permissionType,
                        'permission_type_id' => $item->permission_type_id
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $roles,
                'message' => 'Roles retrieved successfully for permission'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve roles for permission',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Sync all permissions for a role (replace existing)
     */
    public function syncPermissions(Request $request, $roleId): JsonResponse
    {
        // Use manual validation
        $validator = Validator::make($request->all(), [
            'permissions' => 'required|array',
            'permissions.*.permission_id' => 'required|integer',
            'permissions.*.permission_type_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Manual validation for role_id
            $roleExists = DB::table('role_master')
                ->where('Role_Id', $roleId)
                ->exists();
            
            if (!$roleExists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role not found'
                ], 404);
            }

            DB::beginTransaction();
            
            // Delete existing permissions for this role
            PermissionRole::where('role_id', $roleId)->delete();
            
            $results = [
                'synced' => 0,
                'failed' => 0,
                'errors' => []
            ];
            
            // Add new permissions
            foreach ($request->permissions as $index => $permissionData) {
                try {
                    // Manual validation for permission_id
                    $permissionExists = DB::table('permissions')
                        ->where('id', $permissionData['permission_id'])
                        ->exists();
                    
                    if (!$permissionExists) {
                        $results['failed']++;
                        $results['errors'][] = [
                            'index' => $index,
                            'permission_id' => $permissionData['permission_id'],
                            'error' => 'Permission does not exist'
                        ];
                        continue;
                    }

                    // Manual validation for permission_type_id
                    $permissionTypeExists = DB::table('permission_types')
                        ->where('id', $permissionData['permission_type_id'])
                        ->exists();
                    
                    if (!$permissionTypeExists) {
                        $results['failed']++;
                        $results['errors'][] = [
                            'index' => $index,
                            'permission_id' => $permissionData['permission_id'],
                            'error' => 'Permission type does not exist'
                        ];
                        continue;
                    }
                    
                    PermissionRole::create([
                        'permission_id' => $permissionData['permission_id'],
                        'role_id' => $roleId,
                        'permission_type_id' => $permissionData['permission_type_id']
                    ]);
                    
                    $results['synced']++;
                    
                } catch (\Exception $e) {
                    $results['failed']++;
                    $results['errors'][] = [
                        'index' => $index,
                        'permission_id' => $permissionData['permission_id'],
                        'error' => $e->getMessage()
                    ];
                }
            }
            
            DB::commit();
            
            return response()->json([
                'success' => true,
                'data' => $results,
                'message' => 'Permissions synced successfully for role'
            ], 200);
            
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to sync permissions for role',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}