<?php

namespace App\Http\Controllers;

use App\Models\RelationshipMaster;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Throwable;

class RelationshipMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            // Get search query if any
            $query = $request->input('query');

            $relationships = RelationshipMaster::query();

            // Apply search filter if query exists
            if ($query) {
                $relationships->where(function ($q) use ($query) {
                    $q->where('relation_code', 'like', '%' . $query . '%')
                      ->orWhere('relation_name', 'like', '%' . $query . '%');
                });
            }

            // Order by latest
            $relationships->orderBy('relation_id', 'desc');

            $data = $relationships->get();

            return response()->json([
                'status' => 'success',
                'message' => 'Relationships fetched successfully',
                'data' => $data
            ]);
        } catch (Throwable $e) {
            Log::error('Relationship Fetch Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch relationships',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            // Validate request data
            $validated = $request->validate([
                'relation_code' => 'required|string|max:20|unique:relationship_master,relation_code',
                'relation_name' => 'required|string|max:100',
            ]);

            // Create relationship
            $relationship = RelationshipMaster::create($validated);

            return response()->json([
                'status' => 'success',
                'message' => 'Relationship created successfully',
                'data' => $relationship
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (Throwable $e) {
            Log::error('Relationship Create Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to create relationship',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try {
            // Find relationship by relation_id
            $relationship = RelationshipMaster::find($id);

            if (!$relationship) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Relationship not found'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Relationship fetched successfully',
                'data' => $relationship
            ]);
        } catch (Throwable $e) {
            Log::error('Relationship Show Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch relationship',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            // Find relationship by relation_id
            $relationship = RelationshipMaster::find($id);

            if (!$relationship) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Relationship not found'
                ], 404);
            }

            // Validate request data
            $validated = $request->validate([
                'relation_code' => 'sometimes|required|string|max:20|unique:relationship_master,relation_code,' . $id . ',relation_id',
                'relation_name' => 'sometimes|required|string|max:100',
            ]);

            // Update relationship
            $relationship->update($validated);

            return response()->json([
                'status' => 'success',
                'message' => 'Relationship updated successfully',
                'data' => $relationship
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (Throwable $e) {
            Log::error('Relationship Update Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to update relationship',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            // Find relationship by relation_id
            $relationship = RelationshipMaster::find($id);

            if (!$relationship) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Relationship not found'
                ], 404);
            }

            // Delete relationship
            $relationship->delete();

            return response()->json([
                'status' => 'success',
                'message' => 'Relationship deleted successfully',
                'data' => null
            ]);
        } catch (Throwable $e) {
            Log::error('Relationship Delete Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to delete relationship',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
