<?php

namespace App\Http\Controllers;

use App\Models\RoleMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class RoleMasterController extends Controller
{
    /**
     * Display a listing of roles.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            // Validation for query parameters
            $validator = Validator::make($request->all(), [
                'active_only' => 'nullable|boolean',
                'two_fac_auth' => 'nullable|string|in:Y,N',
                'search' => 'nullable|string|max:100',
                'with_user_count' => 'nullable|boolean',
                'sort_by' => 'nullable|string|in:Role_Id,Role_Name,Role_Prefix,created_at,updated_at',
                'sort_order' => 'nullable|string|in:asc,desc',
                'per_page' => 'nullable|integer|min:1|max:100'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $query = RoleMaster::query();

            // Filter by active status
            if ($request->has('active_only')) {
                $query->where('Is_Active', $request->boolean('active_only'));
            }

            // Filter by TwoFacAuth
            if ($request->has('two_fac_auth')) {
                $query->where('TwoFacAuth', $request->two_fac_auth);
            }

            // Search functionality
            if ($request->has('search')) {
                $query->search($request->search);
            }

            // Get roles with user count
            if ($request->has('with_user_count')) {
                $query->withUserCount();
            }

            // Sorting
            $sortBy = $request->get('sort_by', 'Role_Name');
            $sortOrder = $request->get('sort_order', 'asc');
            $query->orderBy($sortBy, $sortOrder);

            // Pagination
            $perPage = $request->get('per_page', 15);
            $roles = $query->paginate($perPage);

            // Format response
            $roles->getCollection()->transform(function ($role) {
                return [
                    'Role_Id' => $role->Role_Id,
                    'Role_Prefix' => $role->Role_Prefix,
                    'Role_Name' => $role->Role_Name,
                    'Full_Role' => $role->full_role,
                    'TwoFacAuth' => $role->TwoFacAuth,
                    'TwoFacAuth_Text' => $role->two_factor_auth_text,
                    'Is_Active' => $role->Is_Active,
                    'Status_Text' => $role->status_text,
                    'created_at' => $role->created_at,
                    'updated_at' => $role->updated_at
                ];
            });

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Roles fetched successfully',
                'data' => $roles
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch roles',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    public function getAllRolesList(Request $request)
    {
        try {
            // Validation
            $validator = Validator::make($request->all(), [
                'active_only' => 'nullable|boolean',
                'two_fac_auth' => 'nullable|string|in:Y,N',
                'search' => 'nullable|string|max:100',
                'with_user_count' => 'nullable|boolean',
                'sort_by' => 'nullable|string|in:Role_Id,Role_Name,Role_Prefix,created_at,updated_at',
                'sort_order' => 'nullable|string|in:asc,desc',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $query = RoleMaster::query();

            // Filter by active status
            if ($request->has('active_only')) {
                $query->where('Is_Active', $request->boolean('active_only'));
            }

            // Filter by Two Factor Auth
            if ($request->has('two_fac_auth')) {
                $query->where('TwoFacAuth', $request->two_fac_auth);
            }

            // Search
            if ($request->filled('search')) {
                $query->search($request->search);
            }

            // With user count
            if ($request->boolean('with_user_count')) {
                $query->withUserCount();
            }

            // Sorting
            $sortBy = $request->get('sort_by', 'Role_Name');
            $sortOrder = $request->get('sort_order', 'asc');
            $query->orderBy($sortBy, $sortOrder);

            // ✅ Fetch data (NO PAGINATION)
            $roles = $query->get();

            // Format response
            $roles = $roles->map(function ($role) {
                return [
                    'Role_Id' => $role->Role_Id,
                    'Role_Prefix' => $role->Role_Prefix,
                    'Role_Name' => $role->Role_Name,
                    'Full_Role' => $role->full_role,
                    'TwoFacAuth' => $role->TwoFacAuth,
                    'TwoFacAuth_Text' => $role->two_factor_auth_text,
                    'Is_Active' => $role->Is_Active,
                    'Status_Text' => $role->status_text,
                    'created_at' => $role->created_at,
                    'updated_at' => $role->updated_at,
                ];
            });

            return response()->json([
                'success' => true,
                'message' => 'GMC Gandhinagar - Roles fetched successfully',
                'data' => $roles
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch roles',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created role.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            // Validation rules
            $validator = Validator::make($request->all(), [
                'Role_Prefix' => [
                    'required',
                    'string',
                    'max:10',
                    'regex:/^[A-Z]+$/',
                    'unique:role_master,Role_Prefix'
                ],
                'Role_Name' => [
                    'required',
                    'string',
                    'max:50',
                    'unique:role_master,Role_Name'
                ],
                'TwoFacAuth' => [
                    'required',
                    'string',
                    'in:Y,N'
                ],
                'Is_Active' => [
                    'nullable',
                    'boolean'
                ]
            ], [
                'Role_Prefix.required' => 'Role prefix is required',
                'Role_Prefix.string' => 'Role prefix must be a string',
                'Role_Prefix.max' => 'Role prefix cannot exceed 10 characters',
                'Role_Prefix.regex' => 'Role prefix must contain only uppercase letters',
                'Role_Prefix.unique' => 'This role prefix already exists',

                'Role_Name.required' => 'Role name is required',
                'Role_Name.string' => 'Role name must be a string',
                'Role_Name.max' => 'Role name cannot exceed 50 characters',
                'Role_Name.unique' => 'This role name already exists',

                'TwoFacAuth.required' => 'Two Factor Authentication setting is required',
                'TwoFacAuth.string' => 'Two Factor Authentication must be a string',
                'TwoFacAuth.in' => 'Two Factor Authentication must be either Y or N',

                'Is_Active.boolean' => 'Active status must be true or false'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $role = RoleMaster::create([
                'Role_Prefix' => $request->Role_Prefix,
                'Role_Name' => $request->Role_Name,
                'TwoFacAuth' => $request->TwoFacAuth,
                'Is_Active' => $request->get('Is_Active', true)
            ]);

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Role created successfully',
                'data' => $role
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create role',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified role.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    /**
     * Display the specified role.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            // Validate ID parameter
            $validator = Validator::make(['id' => $id], [
                'id' => 'required|integer|min:1|exists:role_master,Role_Id'
            ], [
                'id.required' => 'Role ID is required',
                'id.integer' => 'Role ID must be an integer',
                'id.min' => 'Role ID must be a positive number',
                'id.exists' => 'Role not found'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            // FIX: Remove the column selection or use the correct column names
            $role = RoleMaster::with(['users'])->find($id);

            // OR if you need specific columns, find out what the actual column names are:
            // $role = RoleMaster::with(['users' => function($query) {
            //     // Check your gmc_users table for actual column names
            //     $query->select('user_id', 'first_name', 'last_name', 'email'); // or whatever columns exist
            // }])->find($id);

            if (!$role) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role not found'
                ], 404);
            }

            // Add user count information
            $roleData = $role->toArray();
            $roleData['user_count'] = $role->userCount();
            $roleData['default_user_count'] = $role->defaultUserCount();

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Role fetched successfully',
                'data' => $roleData
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch role',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified role.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            // Validate ID parameter
            $idValidator = Validator::make(['id' => $id], [
                'id' => 'required|integer|min:1|exists:role_master,Role_Id'
            ], [
                'id.required' => 'Role ID is required',
                'id.integer' => 'Role ID must be an integer',
                'id.min' => 'Role ID must be a positive number',
                'id.exists' => 'Role not found'
            ]);

            if ($idValidator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $idValidator->errors()
                ], 422);
            }

            $role = RoleMaster::find($id);

            if (!$role) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role not found'
                ], 404);
            }

            // Validation rules for request data
            $validator = Validator::make($request->all(), RoleMaster::updateRules($id), RoleMaster::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $role->update([
                'Role_Prefix' => $request->get('Role_Prefix', $role->Role_Prefix),
                'Role_Name' => $request->get('Role_Name', $role->Role_Name),
                'TwoFacAuth' => $request->get('TwoFacAuth', $role->TwoFacAuth),
                'Is_Active' => $request->get('Is_Active', $role->Is_Active)
            ]);

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Role updated successfully',
                'data' => $role
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update role',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified role.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            // Validate ID parameter
            $validator = Validator::make(['id' => $id], [
                'id' => 'required|integer|min:1|exists:role_master,Role_Id'
            ], [
                'id.required' => 'Role ID is required',
                'id.integer' => 'Role ID must be an integer',
                'id.min' => 'Role ID must be a positive number',
                'id.exists' => 'Role not found'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $role = RoleMaster::find($id);

            if (!$role) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role not found'
                ], 404);
            }

            // Check if role can be deleted
            if (!$role->canBeDeleted()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete role as it has users assigned'
                ], 400);
            }

            $role->delete();

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Role deleted successfully',
                'data' => null
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete role',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Deactivate a role.
     */
    public function deactivate($id)
    {
        try {
            // Validate ID parameter
            $validator = Validator::make(['id' => $id], [
                'id' => 'required|integer|min:1|exists:role_master,Role_Id'
            ], [
                'id.required' => 'Role ID is required',
                'id.integer' => 'Role ID must be an integer',
                'id.min' => 'Role ID must be a positive number',
                'id.exists' => 'Role not found'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $role = RoleMaster::find($id);

            if (!$role) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role not found'
                ], 404);
            }

            // Check if already deactivated
            if (!$role->Is_Active) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role is already deactivated'
                ], 400);
            }

            $role->deactivate();

            return response()->json([
                'success' => true,
                'message' => 'Role deactivated successfully',
                'data' => $role
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to deactivate role',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Activate a role.
     */
    public function activate($id)
    {
        try {
            // Validate ID parameter
            $validator = Validator::make(['id' => $id], [
                'id' => 'required|integer|min:1|exists:role_master,Role_Id'
            ], [
                'id.required' => 'Role ID is required',
                'id.integer' => 'Role ID must be an integer',
                'id.min' => 'Role ID must be a positive number',
                'id.exists' => 'Role not found'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $role = RoleMaster::find($id);

            if (!$role) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role not found'
                ], 404);
            }

            // Check if already active
            if ($role->Is_Active) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role is already active'
                ], 400);
            }

            $role->activate();

            return response()->json([
                'success' => true,
                'message' => 'Role activated successfully',
                'data' => $role
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to activate role',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Enable Two Factor Authentication for a role.
     */
    public function enableTwoFactorAuth($id)
    {
        try {
            // Validate ID parameter
            $validator = Validator::make(['id' => $id], [
                'id' => 'required|integer|min:1|exists:role_master,Role_Id'
            ], [
                'id.required' => 'Role ID is required',
                'id.integer' => 'Role ID must be an integer',
                'id.min' => 'Role ID must be a positive number',
                'id.exists' => 'Role not found'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $role = RoleMaster::find($id);

            if (!$role) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role not found'
                ], 404);
            }

            // Check if 2FA is already enabled
            if ($role->TwoFacAuth === 'Y') {
                return response()->json([
                    'success' => false,
                    'message' => 'Two Factor Authentication is already enabled'
                ], 400);
            }

            $role->enableTwoFactorAuth();

            return response()->json([
                'success' => true,
                'message' => 'Two Factor Authentication enabled successfully',
                'data' => $role
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to enable Two Factor Authentication',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Disable Two Factor Authentication for a role.
     */
    public function disableTwoFactorAuth($id)
    {
        try {
            // Validate ID parameter
            $validator = Validator::make(['id' => $id], [
                'id' => 'required|integer|min:1|exists:role_master,Role_Id'
            ], [
                'id.required' => 'Role ID is required',
                'id.integer' => 'Role ID must be an integer',
                'id.min' => 'Role ID must be a positive number',
                'id.exists' => 'Role not found'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $role = RoleMaster::find($id);

            if (!$role) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role not found'
                ], 404);
            }

            // Check if 2FA is already disabled
            if ($role->TwoFacAuth === 'N' || $role->TwoFacAuth === null) {
                return response()->json([
                    'success' => false,
                    'message' => 'Two Factor Authentication is already disabled'
                ], 400);
            }

            $role->disableTwoFactorAuth();

            return response()->json([
                'success' => true,
                'message' => 'Two Factor Authentication disabled successfully',
                'data' => $role
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to disable Two Factor Authentication',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get roles with user count.
     */
    public function getRolesWithUserCount(Request $request)
    {
        try {
            // Validation for query parameters
            $validator = Validator::make($request->all(), [
                'active_only' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $query = RoleMaster::withCount(['users as total_users', 'users as default_users' => function ($query) {
                $query->where('default_rid', 1);
            }]);

            if ($request->has('active_only')) {
                $query->where('Is_Active', $request->boolean('active_only'));
            }

            $roles = $query->get();

            return response()->json([
                'success' => true,
                'message' => 'Roles with user count fetched successfully',
                'data' => $roles
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch roles with user count',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk create roles.
     */
    public function bulkStore(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'roles' => 'required|array|min:1|max:50',
                'roles.*.Role_Prefix' => 'nullable|string|max:5',
                'roles.*.Role_Name' => 'required|string|max:25',
                'roles.*.TwoFacAuth' => 'nullable|string|in:Y,N',
                'roles.*.Is_Active' => 'nullable|boolean'
            ], RoleMaster::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $createdRoles = [];
            $errors = [];

            foreach ($request->roles as $index => $roleData) {
                try {
                    // Additional validation for each role
                    $roleValidator = Validator::make($roleData, [
                        'Role_Name' => 'required|string|max:25|unique:role_master,Role_Name',
                        'Role_Prefix' => 'nullable|string|max:5',
                        'TwoFacAuth' => 'nullable|string|in:Y,N',
                        'Is_Active' => 'nullable|boolean'
                    ], RoleMaster::validationMessages());

                    if ($roleValidator->fails()) {
                        $errors[] = [
                            'index' => $index,
                            'data' => $roleData,
                            'errors' => $roleValidator->errors()->toArray()
                        ];
                        continue;
                    }

                    $role = RoleMaster::create([
                        'Role_Prefix' => $roleData['Role_Prefix'] ?? null,
                        'Role_Name' => $roleData['Role_Name'],
                        'TwoFacAuth' => $roleData['TwoFacAuth'] ?? null,
                        'Is_Active' => $roleData['Is_Active'] ?? true
                    ]);

                    $createdRoles[] = $role;
                } catch (\Exception $e) {
                    $errors[] = [
                        'index' => $index,
                        'data' => $roleData,
                        'message' => $e->getMessage()
                    ];
                }
            }

            $response = [
                'success' => true,
                'message' => count($createdRoles) . ' role(s) created successfully',
                'data' => $createdRoles,
                'total_created' => count($createdRoles),
                'total_failed' => count($errors)
            ];

            if (count($errors) > 0) {
                $response['errors'] = $errors;
            }

            return response()->json($response, count($createdRoles) > 0 ? 201 : 400);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create bulk roles',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
