<?php

namespace App\Http\Controllers;

use App\Models\SpecialDutyMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Throwable;

class SpecialDutyMasterController extends Controller
{
    // 🔹 GET ALL
    public function index()
    {
        try {
            $data = SpecialDutyMaster::orderBy('duty_type_id', 'desc')->get();

            return response()->json([
                'success' => true,
                'data' => $data
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch records',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 STORE
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'duty_code' => 'nullable|string|max:20|unique:special_duty_master,duty_code',
                'duty_name' => 'required|string|max:100',
                'description' => 'nullable|string|max:255',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $duty = SpecialDutyMaster::create([
                'duty_code' => $request->duty_code,
                'duty_name' => $request->duty_name,
                'description' => $request->description,
                'is_active' => true
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Special Duty created successfully',
                'data' => $duty
            ], 201);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Create failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 SHOW
    public function show($id)
    {
        try {
            $duty = SpecialDutyMaster::find($id);

            if (!$duty) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $duty
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔹 UPDATE
    public function update(Request $request, $id)
    {
        try {
            $duty = SpecialDutyMaster::find($id);

            if (!$duty) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'duty_code' => 'sometimes|required|string|max:20|unique:special_duty_master,duty_code,' . $id . ',duty_type_id',
                'duty_name' => 'sometimes|required|string|max:100',
                'description' => 'sometimes|nullable|string|max:255',
                'is_active' => 'sometimes|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            // ✅ CORRECT: sometimes + validated + update
            $duty->update($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'Special Duty updated successfully',
                'data' => $duty
            ], 200);
        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Update failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    // 🔹 DELETE
    public function destroy($id)
    {
        try {
            $duty = SpecialDutyMaster::find($id);

            if (!$duty) {
                return response()->json([
                    'success' => false,
                    'message' => 'Record not found'
                ], 404);
            }

            $duty->delete();

            return response()->json([
                'success' => true,
                'message' => 'Special Duty deleted successfully'
            ], 200);
        } catch (Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Delete failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
