<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\SubCategory;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class SubCategoryController extends Controller
{
    /**
     * Display a listing of subcategories (without pagination)
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        try {
            $subCategories = SubCategory::with(['category', 'subSubCategories'])
                ->orderBy('sub_category_name')
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - SubCategories fetched successfully',
                'data' => $subCategories
            ], 200);

        } catch (\Exception $e) {
            Log::error('SubCategory Fetch Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch sub-categories. Please try again later.'
            ], 500);
        }
    }

    /**
     * Display paginated subcategories (separate pagination API)
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function paginate(Request $request)
    {
        try {
            // Validate pagination parameters
            $validator = Validator::make($request->all(), [
                'per_page' => 'nullable|integer|min:1|max:100',
                'page' => 'nullable|integer|min:1',
                'search' => 'nullable|string|max:100',
                'category_id' => 'nullable|integer|exists:categories,category_id',
                'is_active' => 'nullable|boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid pagination parameters',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Get pagination parameters with defaults
            $perPage = $request->get('per_page', 10);
            $page = $request->get('page', 1);
            $search = $request->get('search', '');
            $categoryId = $request->get('category_id', null);
            $isActive = $request->get('is_active', null);

            // Build query
            $query = SubCategory::with(['category', 'subSubCategories'])
                ->when($search, function ($query, $search) {
                    return $query->where(function ($q) use ($search) {
                        $q->where('sub_category_name', 'like', "%{$search}%")
                          ->orWhere('sub_category_code', 'like', "%{$search}%")
                          ->orWhere('description', 'like', "%{$search}%");
                    });
                })
                ->when($categoryId, function ($query, $categoryId) {
                    return $query->where('category_id', $categoryId);
                })
                ->when(!is_null($isActive), function ($query) use ($isActive) {
                    return $query->where('is_active', $isActive);
                })
                ->orderBy('sub_category_name');

            // Get paginated results
            $subCategories = $query->paginate($perPage, ['*'], 'page', $page);

            // Format pagination response
            $response = [
                'success' => true,
                'message' => 'GMC Gadhinagar - SubCategories paginated successfully',
                'data' => $subCategories->items(),
                'pagination' => [
                    'current_page' => $subCategories->currentPage(),
                    'first_page_url' => $subCategories->url(1),
                    'from' => $subCategories->firstItem(),
                    'last_page' => $subCategories->lastPage(),
                    'last_page_url' => $subCategories->url($subCategories->lastPage()),
                    'links' => $subCategories->linkCollection()->toArray(),
                    'next_page_url' => $subCategories->nextPageUrl(),
                    'path' => $subCategories->path(),
                    'per_page' => $subCategories->perPage(),
                    'prev_page_url' => $subCategories->previousPageUrl(),
                    'to' => $subCategories->lastItem(),
                    'total' => $subCategories->total(),
                ],
                'meta' => [
                    'total_records' => $subCategories->total(),
                    'current_page_records' => count($subCategories->items()),
                    'total_pages' => $subCategories->lastPage(),
                    'has_more_pages' => $subCategories->hasMorePages()
                ]
            ];

            return response()->json($response, 200);

        } catch (\Exception $e) {
            Log::error('SubCategory Paginate Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch paginated sub-categories. Please try again later.'
            ], 500);
        }
    }

    /**
     * Store a newly created subcategory.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'category_id' => 'required|integer|exists:categories,category_id',
                'sub_category_name' => 'required|string|max:100',
                'sub_category_code' => 'required|string|max:50|regex:/^[A-Z0-9]+$/',
                'description' => 'nullable|string|max:500',
                'is_active' => 'boolean'
            ], [
                'category_id.required' => 'Category selection is required',
                'category_id.exists' => 'Selected category does not exist',
                'sub_category_name.required' => 'Sub-category name is required',
                'sub_category_code.required' => 'Sub-category code is required',
                'sub_category_code.regex' => 'Sub-category code should contain only uppercase letters and numbers'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            // Check unique constraint
            $exists = SubCategory::where('category_id', $request->category_id)
                ->where('sub_category_code', strtoupper($request->sub_category_code))
                ->exists();

            if ($exists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => [
                        'sub_category_code' => ['This sub-category code already exists for the selected category']
                    ]
                ], 422);
            }

            $subCategory = SubCategory::create([
                'category_id' => $request->category_id,
                'sub_category_name' => $request->sub_category_name,
                'sub_category_code' => strtoupper($request->sub_category_code),
                'description' => $request->description,
                'is_active' => $request->boolean('is_active', true)
            ]);

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - SubCategory created successfully',
                'data' => $subCategory->load('category')
            ], 201);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            Log::error('SubCategory Create Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to create sub-category. Please check the data and try again.'
            ], 500);
        }
    }

    /**
     * Display the specified subcategory.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $subCategory = SubCategory::with(['category', 'subSubCategories'])->find($id);

            if (!$subCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sub-category not found. Please check the sub-category ID.'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - SubCategory details retrieved successfully',
                'data' => $subCategory
            ], 200);

        } catch (\Exception $e) {
            Log::error('SubCategory Show Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to retrieve sub-category details.'
            ], 500);
        }
    }

    /**
     * Update the specified subcategory.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $subCategory = SubCategory::find($id);

            if (!$subCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sub-category not found. Please check the sub-category ID.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'category_id' => 'required|integer|exists:categories,category_id',
                'sub_category_name' => 'required|string|max:100',
                'sub_category_code' => 'required|string|max:50|regex:/^[A-Z0-9]+$/',
                'description' => 'nullable|string|max:500',
                'is_active' => 'boolean'
            ], [
                'category_id.required' => 'Category selection is required',
                'category_id.exists' => 'Selected category does not exist',
                'sub_category_name.required' => 'Sub-category name is required',
                'sub_category_code.required' => 'Sub-category code is required',
                'sub_category_code.regex' => 'Sub-category code should contain only uppercase letters and numbers'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            // Check unique constraint (excluding current record)
            $exists = SubCategory::where('category_id', $request->category_id)
                ->where('sub_category_code', strtoupper($request->sub_category_code))
                ->where('sub_category_id', '!=', $id)
                ->exists();

            if ($exists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => [
                        'sub_category_code' => ['This sub-category code already exists for the selected category']
                    ]
                ], 422);
            }

            $subCategory->update([
                'category_id' => $request->category_id,
                'sub_category_name' => $request->sub_category_name,
                'sub_category_code' => strtoupper($request->sub_category_code),
                'description' => $request->description,
                'is_active' => $request->boolean('is_active', $subCategory->is_active)
            ]);

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - SubCategory updated successfully',
                'data' => $subCategory->refresh()->load('category')
            ], 200);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            Log::error('SubCategory Update Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to update sub-category. Please check the data and try again.'
            ], 500);
        }
    }

    /**
     * Remove the specified subcategory.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $subCategory = SubCategory::withCount(['subSubCategories'])->find($id);

            if (!$subCategory) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sub-category not found. Please check the sub-category ID.'
                ], 404);
            }

            if ($subCategory->sub_sub_categories_count > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete sub-category because it has associated sub-sub-categories. Please delete sub-sub-categories first.'
                ], 400);
            }

            $subCategory->delete();

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - SubCategory deleted successfully',
                'data' => null
            ], 200);

        } catch (\Illuminate\Database\QueryException $e) {
            Log::error('SubCategory Delete Query Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Cannot delete sub-category due to database constraints.'
            ], 409);

        } catch (\Exception $e) {
            Log::error('SubCategory Delete Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete sub-category. Please try again later.'
            ], 500);
        }
    }
}
