<?php

namespace App\Http\Controllers;

use App\Models\SubLedgerMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\UploadedFile;

class SubLedgerMasterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
      public function index(Request $request)
    {
        try {
            $query = SubLedgerMaster::with('ledger');

            // Filter by ledger if provided
            if ($request->has('ledger_id')) {
                $query->byLedger($request->ledger_id);
            }

            // Filter by status if provided
            if ($request->has('status')) {
                $query->where('SL_Status', $request->boolean('status'));
            }

            // Search functionality
            if ($request->has('search')) {
                $query->search($request->search);
            }

            // Filter by credit limit
            if ($request->has('min_credit_limit')) {
                $query->where('SL_Cr_Lim', '>=', $request->min_credit_limit);
            }

            if ($request->has('max_credit_limit')) {
                $query->where('SL_Cr_Lim', '<=', $request->max_credit_limit);
            }

            // Sorting
            $sortBy = $request->get('sort_by', 'SL_Code');
            $sortOrder = $request->get('sort_order', 'asc');
            $query->orderBy($sortBy, $sortOrder);

            // Pagination
            $perPage = $request->get('per_page', 15);
            $subLedgers = $query->paginate($perPage);

            // Load ledger relationship for all items
            $subLedgers->load('ledger');

            return response()->json([
                'success' => true,
                'data' => $subLedgers,
                'message' => 'Sub-ledgers retrieved successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve sub-ledgers',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function getAll(Request $request)
    {
        try {
            $query = SubLedgerMaster::with('ledger');

            // Filter by ledger if provided
            if ($request->has('ledger_id')) {
                $query->byLedger($request->ledger_id);
            }

            // Filter by status if provided
            if ($request->has('status')) {
                $query->where('SL_Status', $request->boolean('status'));
            }

            // Search functionality
            if ($request->has('search')) {
                $query->search($request->search);
            }

            // Filter by credit limit
            if ($request->has('min_credit_limit')) {
                $query->where('SL_Cr_Lim', '>=', $request->min_credit_limit);
            }

            if ($request->has('max_credit_limit')) {
                $query->where('SL_Cr_Lim', '<=', $request->max_credit_limit);
            }

            // Sorting
            $sortBy = $request->get('sort_by', 'SL_Code');
            $sortOrder = $request->get('sort_order', 'asc');
            $query->orderBy($sortBy, $sortOrder);

            // Get all data without pagination
            $subLedgers = $query->get();

            // Load ledger relationship for all items
            $subLedgers->load('ledger');

            return response()->json([
                'success' => true,
                'data' => $subLedgers,
                'count' => $subLedgers->count(),
                'message' => 'All sub-ledgers retrieved successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve sub-ledgers',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            // Convert boolean string values to actual booleans
            $data = $this->convertBooleanStrings($request->all());

            $validator = Validator::make($data, SubLedgerMaster::createRules(), SubLedgerMaster::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            // Check if parent ledger exists and is active
            $ledger = \App\Models\LedgerMaster::find($data['Lg_ID']);
            if (!$ledger) {
                return response()->json([
                    'success' => false,
                    'message' => 'Parent ledger not found'
                ], 404);
            }

            if (!$ledger->Lg_Status) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot create sub-ledger under inactive ledger'
                ], 400);
            }

            // Handle cancelled cheque file upload
            if ($request->hasFile('cancelled_cheque')) {
                $file = $request->file('cancelled_cheque');
                $path = $this->uploadCancelledCheque($file, $data['SL_Name']);

                if ($path) {
                    $data['Cancelled_Cheque_Path'] = $path;
                    $data['Cancelled_Cheque_Original_Name'] = $file->getClientOriginalName();
                }
            }

            $subLedger = SubLedgerMaster::create($data);

            return response()->json([
                'success' => true,
                'message' => 'Sub-ledger created successfully',
                'data' => $subLedger->load('ledger')
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create sub-ledger',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $subLedger = SubLedgerMaster::find($id);

            if (!$subLedger) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sub-ledger not found'
                ], 404);
            }

            // Convert boolean string values to actual booleans
            $data = $this->convertBooleanStrings($request->all());

            $validator = Validator::make($data, SubLedgerMaster::updateRules($id), SubLedgerMaster::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            // If changing ledger, check if new ledger exists and is active
            if (isset($data['Lg_ID']) && $data['Lg_ID'] != $subLedger->Lg_ID) {
                $newLedger = \App\Models\LedgerMaster::find($data['Lg_ID']);
                if (!$newLedger) {
                    return response()->json([
                        'success' => false,
                        'message' => 'New parent ledger not found'
                    ], 404);
                }

                if (!$newLedger->Lg_Status) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Cannot move sub-ledger to inactive ledger'
                    ], 400);
                }
            }

            // Handle cancelled cheque file upload
            if ($request->hasFile('cancelled_cheque')) {
                // Delete old file if exists
                if ($subLedger->Cancelled_Cheque_Path && Storage::exists($subLedger->Cancelled_Cheque_Path)) {
                    Storage::delete($subLedger->Cancelled_Cheque_Path);
                }

                $file = $request->file('cancelled_cheque');
                $path = $this->uploadCancelledCheque($file, $data['SL_Name'] ?? $subLedger->SL_Name);

                if ($path) {
                    $data['Cancelled_Cheque_Path'] = $path;
                    $data['Cancelled_Cheque_Original_Name'] = $file->getClientOriginalName();
                }
            }

            // Handle verification dates - if unverifying, set date to null
            if (isset($data['PAN_Verified']) && $data['PAN_Verified'] === false) {
                $data['PAN_Verified_Date'] = null;
            } elseif (isset($data['PAN_Verified']) && $data['PAN_Verified'] === true && !$subLedger->PAN_Verified) {
                $data['PAN_Verified_Date'] = now();
            }

            if (isset($data['GSTIN_Verified']) && $data['GSTIN_Verified'] === false) {
                $data['GSTIN_Verified_Date'] = null;
            } elseif (isset($data['GSTIN_Verified']) && $data['GSTIN_Verified'] === true && !$subLedger->GSTIN_Verified) {
                $data['GSTIN_Verified_Date'] = now();
            }

            if (isset($data['Bank_Details_Verified']) && $data['Bank_Details_Verified'] === false) {
                $data['Bank_Details_Verified_Date'] = null;
            } elseif (isset($data['Bank_Details_Verified']) && $data['Bank_Details_Verified'] === true && !$subLedger->Bank_Details_Verified) {
                $data['Bank_Details_Verified_Date'] = now();
            }

            $subLedger->update($data);

            return response()->json([
                'success' => true,
                'message' => 'Sub-ledger updated successfully',
                'data' => $subLedger->load('ledger')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update sub-ledger',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    private function convertBooleanStrings(array $data): array
{
    $booleanFields = [
        'SL_Status',
        'PAN_Verified',
        'GSTIN_Verified',
        'Bank_Details_Verified'
    ];

    foreach ($booleanFields as $field) {
        if (isset($data[$field])) {
            if (is_string($data[$field])) {
                $value = strtolower(trim($data[$field]));

                // Handle true values
                if (in_array($value, ['true', '1', 'yes', 'on'])) {
                    $data[$field] = true;
                }
                // Handle false values
                elseif (in_array($value, ['false', '0', 'no', 'off'])) {
                    $data[$field] = false;
                }
                // Handle null/empty as null (not false)
                elseif (in_array($value, ['null', ''])) {
                    $data[$field] = null;
                }
                // For any other string, leave as is or convert based on context
                else {
                    // Optional: you might want to default to false or leave as string
                    $data[$field] = false;
                }
            }
            elseif (is_numeric($data[$field])) {
                // Handle numeric values 1 or 0
                $data[$field] = (bool) $data[$field];
            }
            // Already boolean values will remain unchanged
        }
    }

    return $data;
}

    /**
     * Upload cancelled cheque file
     */
    private function uploadCancelledCheque(UploadedFile $file, $subLedgerName)
    {
        try {
            $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
            $extension = $file->getClientOriginalExtension();

            // Sanitize sub-ledger name for filename
            $safeName = preg_replace('/[^A-Za-z0-9_\-]/', '_', $subLedgerName);
            $safeName = substr($safeName, 0, 50);

            // Generate unique filename
            $filename = $safeName . '_cancelled_cheque_' . time() . '_' . uniqid() . '.' . $extension;

            // Store in storage/app/public/cancelled-cheques
            $path = $file->storeAs('cancelled-cheques', $filename, 'public');

            return $path;
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try {
            $subLedger = SubLedgerMaster::with('ledger')->find($id);

            if (!$subLedger) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sub-ledger not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $subLedger,
                'message' => 'Sub-ledger retrieved successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve sub-ledger',
                'error' => $e->getMessage()
            ], 500);
        }
    }



    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $subLedger = SubLedgerMaster::find($id);

            if (!$subLedger) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sub-ledger not found'
                ], 404);
            }

            $subLedger->delete();

            return response()->json([
                'success' => true,
                'message' => 'Sub-ledger deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete sub-ledger',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Deactivate a sub-ledger.
     */
    public function deactivate($id)
    {
        try {
            $subLedger = SubLedgerMaster::find($id);

            if (!$subLedger) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sub-ledger not found'
                ], 404);
            }

            $subLedger->deactivate();

            return response()->json([
                'success' => true,
                'message' => 'Sub-ledger deactivated successfully',
                'data' => $subLedger->load('ledger')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to deactivate sub-ledger',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Activate a sub-ledger.
     */
    public function activate($id)
    {
        try {
            $subLedger = SubLedgerMaster::find($id);

            if (!$subLedger) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sub-ledger not found'
                ], 404);
            }

            // Check if parent ledger is active
            $ledger = $subLedger->ledger;
            if ($ledger && !$ledger->Lg_Status) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot activate sub-ledger under inactive ledger'
                ], 400);
            }

            $subLedger->activate();

            return response()->json([
                'success' => true,
                'message' => 'Sub-ledger activated successfully',
                'data' => $subLedger->load('ledger')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to activate sub-ledger',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get sub-ledgers by ledger ID.
     */
    public function getByLedger($ledgerId)
    {
        try {
            $ledger = \App\Models\LedgerMaster::find($ledgerId);

            if (!$ledger) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ledger not found'
                ], 404);
            }

            $subLedgers = SubLedgerMaster::where('Lg_ID', $ledgerId)
                ->active()
                ->orderBy('SL_Code')
                ->get();

            return response()->json([
                'success' => true,
                'data' => [
                    'ledger' => [
                        'Lg_Id' => $ledger->Lg_Id,
                        'Lg_Code' => $ledger->Lg_Code,
                        'Lg_Name' => $ledger->Lg_Name
                    ],
                    'sub_ledgers' => $subLedgers
                ],
                'message' => 'Sub-ledgers retrieved successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve sub-ledgers',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk create sub-ledgers.
     */
    public function bulkStore(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'sub_ledgers' => 'required|array|min:1',
                'sub_ledgers.*.Lg_ID' => 'required|exists:Ledger_Master,Lg_Id',
                'sub_ledgers.*.SL_Code' => 'required|string|max:50',
                'sub_ledgers.*.SL_Name' => 'required|string|max:200',
                'sub_ledgers.*.SL_Address' => 'nullable|string',
                'sub_ledgers.*.SL_City' => 'nullable|string|max:100',
                'sub_ledgers.*.SL_Phone' => 'nullable|string|max:20',
                'sub_ledgers.*.SL_Email' => 'nullable|email|max:100',
                'sub_ledgers.*.SL_Cont_Pers' => 'nullable|string|max:100',
                'sub_ledgers.*.SL_Cr_Lim' => 'nullable|numeric|min:0',
                'sub_ledgers.*.SL_Draw_Pwr' => 'nullable|numeric|min:0',
                'sub_ledgers.*.SL_Status' => 'nullable|boolean'
            ], SubLedgerMaster::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $createdSubLedgers = [];
            $errors = [];

            foreach ($request->sub_ledgers as $index => $subLedgerData) {
                try {
                    // Check for duplicate code
                    $exists = SubLedgerMaster::where('SL_Code', $subLedgerData['SL_Code'])->exists();
                    if ($exists) {
                        $errors[] = [
                            'index' => $index,
                            'message' => 'Sub-ledger code ' . $subLedgerData['SL_Code'] . ' already exists'
                        ];
                        continue;
                    }

                    // Check if parent ledger is active
                    $ledger = \App\Models\LedgerMaster::find($subLedgerData['Lg_ID']);
                    if (!$ledger || !$ledger->Lg_Status) {
                        $errors[] = [
                            'index' => $index,
                            'message' => 'Parent ledger is not active or not found'
                        ];
                        continue;
                    }

                    $subLedger = SubLedgerMaster::create($subLedgerData);
                    $createdSubLedgers[] = $subLedger;
                } catch (\Exception $e) {
                    $errors[] = [
                        'index' => $index,
                        'message' => $e->getMessage()
                    ];
                }
            }

            return response()->json([
                'success' => true,
                'message' => count($createdSubLedgers) . ' sub-ledgers created successfully',
                'data' => $createdSubLedgers,
                'errors' => $errors,
                'total_created' => count($createdSubLedgers),
                'total_failed' => count($errors)
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create bulk sub-ledgers',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get sub-ledger statistics by ledger.
     */
    public function getStatistics(Request $request)
    {
        try {
            $ledgerId = $request->get('ledger_id');

            $query = SubLedgerMaster::selectRaw('
                COUNT(*) as total_count,
                SUM(CASE WHEN SL_Status = 1 THEN 1 ELSE 0 END) as active_count,
                SUM(CASE WHEN SL_Status = 0 THEN 1 ELSE 0 END) as inactive_count,
                SUM(SL_Cr_Lim) as total_credit_limit,
                SUM(SL_Draw_Pwr) as total_drawing_power,
                AVG(SL_Cr_Lim) as avg_credit_limit
            ');

            if ($ledgerId) {
                $query->where('Lg_ID', $ledgerId);
            }

            $statistics = $query->first();

            return response()->json([
                'success' => true,
                'data' => $statistics,
                'message' => 'Statistics retrieved successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve statistics',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
