<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\TenderEntryMaster;
use App\Models\DocumentLibrary;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class TenderEntryMasterController extends Controller
{


    public function index(Request $request)
    {
        try {
            $query = TenderEntryMaster::with([
                'department',
                'ledger',
                'tenderWork',
                'documentLibrary',
                'uploadedDocuments'
            ]);

            // Search functionality
            if ($request->has('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('Description', 'LIKE', "%{$search}%")
                        ->orWhere('Proposal_Amount', 'LIKE', "%{$search}%")
                        ->orWhereHas('department', function ($q2) use ($search) {
                            $q2->where('desc', 'LIKE', "%{$search}%");
                        })
                        ->orWhereHas('tenderWork', function ($q3) use ($search) {
                            $q3->where('work_name', 'LIKE', "%{$search}%");
                        })
                        ->orWhereHas('uploadedDocuments', function ($q4) use ($search) {
                            $q4->where('Description', 'LIKE', "%{$search}%");
                        });
                });
            }

            // Filter by department
            if ($request->has('bra_id')) {
                $query->where('Bra_id', $request->bra_id);
            }

            // Filter by date range
            if ($request->has('start_date') && $request->has('end_date')) {
                $query->whereBetween('Date', [$request->start_date, $request->end_date]);
            }

            // Order by
            $query->orderBy('Date', 'desc')->orderBy('Tend_Ent_Id', 'desc');

            $perPage = $request->get('per_page', 15);
            $tenders = $query->paginate($perPage);

            // Format the response to match the expected structure
            $formattedTenders = $tenders->through(function ($tender) {
                // Format the tender data
                $formattedTender = $this->formatTenderData($tender);

                // Prepare uploaded_documents
                $uploadedDocuments = [];
                if ($tender->uploadedDocuments && $tender->uploadedDocuments->count() > 0) {
                    foreach ($tender->uploadedDocuments as $document) {
                        $docData = $document->toArray();

                        if (is_string($docData['Doc_Upload'])) {
                            $docData['Doc_Upload'] = json_decode($docData['Doc_Upload'], true);
                        }

                        $uploadedDocuments[] = $docData;
                    }
                }

                // Return the tender object directly with all relationships
                // This matches your expected response structure
                $tenderData = $formattedTender->toArray();

                return [
                    'tender' => $tenderData,
                    'uploaded_documents' => $uploadedDocuments
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $formattedTenders,
                'message' => 'Tender entries retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving tender entries.',
                'error' => $e->getMessage()
            ], 500);
        }
    }


  /**
 * Get all tender data without pagination (exact same structure as index)
 */
public function getAllData(Request $request)
{
    try {
        $query = TenderEntryMaster::with([
            'department',
            'ledger',
            'tenderWork',
            'documentLibrary',
            'uploadedDocuments'
        ]);

        // Search functionality - exactly same as index
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('Description', 'LIKE', "%{$search}%")
                    ->orWhere('Proposal_Amount', 'LIKE', "%{$search}%")
                    ->orWhereHas('department', function ($q2) use ($search) {
                        $q2->where('desc', 'LIKE', "%{$search}%");
                    })
                    ->orWhereHas('tenderWork', function ($q3) use ($search) {
                        $q3->where('work_name', 'LIKE', "%{$search}%");
                    })
                    ->orWhereHas('uploadedDocuments', function ($q4) use ($search) {
                        $q4->where('Description', 'LIKE', "%{$search}%");
                    });
            });
        }

        // Filter by department - exactly same as index
        if ($request->has('bra_id')) {
            $query->where('Bra_id', $request->bra_id);
        }

        // Filter by date range - exactly same as index
        if ($request->has('start_date') && $request->has('end_date')) {
            $query->whereBetween('Date', [$request->start_date, $request->end_date]);
        }

        // Order by - exactly same as index
        $query->orderBy('Date', 'desc')->orderBy('Tend_Ent_Id', 'desc');

        // Get ALL records without pagination
        $tenders = $query->get();

        // Process all tenders with uploaded documents
        $allData = [];
        foreach ($tenders as $tender) {
            // Format the tender data
            $formattedTender = $this->formatTenderData($tender);

            // Prepare uploaded_documents
            $uploadedDocuments = [];
            if ($tender->uploadedDocuments && $tender->uploadedDocuments->count() > 0) {
                foreach ($tender->uploadedDocuments as $document) {
                    $docData = $document->toArray();

                    if (is_string($docData['Doc_Upload'])) {
                        $docData['Doc_Upload'] = json_decode($docData['Doc_Upload'], true);
                    }

                    $uploadedDocuments[] = $docData;
                }
            }

            // Get the tender data as array
            $tenderData = $formattedTender->toArray();

            // Merge tender data with uploaded documents
            $mergedData = array_merge($tenderData, [
                'uploaded_documents' => $uploadedDocuments
            ]);

            $allData[] = $mergedData;
        }

        // Return all data in a simple array format
        return response()->json([
            'success' => true,
            'data' => $allData,
            'count' => count($allData),
            'message' => 'All tender data retrieved successfully.'
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error retrieving tender data.',
            'error' => $e->getMessage()
        ], 500);
    }
}
    private function formatTenderDataForIndex($tender)
    {
        // Decode Doc_Upload for document_library if exists
        if ($tender->documentLibrary && is_string($tender->documentLibrary->Doc_Upload)) {
            $tender->documentLibrary->Doc_Upload = json_decode($tender->documentLibrary->Doc_Upload, true);
        }

        // Decode Doc_Upload for uploadedDocuments if exists
        if ($tender->uploadedDocuments) {
            $tender->uploadedDocuments->transform(function ($document) {
                if (is_string($document->Doc_Upload)) {
                    $document->Doc_Upload = json_decode($document->Doc_Upload, true);
                }
                return $document;
            });
        }

        // Also decode for allDocuments if loaded
        if ($tender->allDocuments) {
            $tender->allDocuments->transform(function ($document) {
                if (is_string($document->Doc_Upload)) {
                    $document->Doc_Upload = json_decode($document->Doc_Upload, true);
                }
                return $document;
            });
        }

        // Add a custom property to include all documents
        $tender->all_documents = $tender->allDocuments ?? collect();

        return $tender;
    }
    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        try {
            $tender = TenderEntryMaster::with([
                'department',
                'ledger',
                'tenderWork',
                'documentLibrary',
                'uploadedDocuments'
            ])->findOrFail($id);

            // Format the tender data
            $formattedTender = $this->formatTenderData($tender);

            // Prepare uploaded_documents
            $uploadedDocuments = [];
            if ($tender->uploadedDocuments && $tender->uploadedDocuments->count() > 0) {
                foreach ($tender->uploadedDocuments as $document) {
                    $docData = $document->toArray();

                    if (is_string($docData['Doc_Upload'])) {
                        $docData['Doc_Upload'] = json_decode($docData['Doc_Upload'], true);
                    }

                    $uploadedDocuments[] = $docData;
                }
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'tender' => $formattedTender,
                    'uploaded_documents' => $uploadedDocuments
                ],
                'message' => 'Tender entry retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Tender entry not found.'
            ], 404);
        }
    }



    /**
     * Format tender data with decoded Doc_Upload
     */
    private function formatTenderData($tender)
    {
        // Decode Doc_Upload for document_library if exists
        if ($tender->documentLibrary && is_string($tender->documentLibrary->Doc_Upload)) {
            $tender->documentLibrary->Doc_Upload = json_decode($tender->documentLibrary->Doc_Upload, true);
        }

        // Decode Doc_Upload for uploadedDocuments if exists
        if ($tender->uploadedDocuments && $tender->uploadedDocuments->count() > 0) {
            foreach ($tender->uploadedDocuments as $document) {
                if (is_string($document->Doc_Upload)) {
                    $document->Doc_Upload = json_decode($document->Doc_Upload, true);
                }
            }
        }

        return $tender;
    }


    /**
     * Store a newly created resource in storage.
     */
   public function store(Request $request)
    {
        DB::beginTransaction();
        try {

            $validator = Validator::make($request->all(), [
                'Bra_id'          => 'required|exists:main_departments,id',
                'Lg_Id'           => 'required|exists:ledger_master,Lg_Id',
                'tend_work_id'    => 'required|exists:new_tender_works,tend_work_id',
                'Proposal_Amount' => 'required|numeric|min:0',
                'Date'            => 'required|date',
                'Description'     => 'nullable|string',
                'Doc_Lib'         => 'nullable|exists:document_library,Doc_Lib',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'validation_error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->only([
                'Bra_id',
                'Lg_Id',
                'tend_work_id',
                'Proposal_Amount',
                'Date',
                'Description',
                'Doc_Lib'
            ]);

            // 🔥 STRICT LOGIC
            $data['available_amount'] = $request->Proposal_Amount;

            $entry = TenderEntryMaster::create($data);

            DB::commit();

            return response()->json([
                'status'  => 'success',
                'message' => 'Tender entry created successfully',
                'data'    => $entry
            ], 201);

        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'status'  => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
  public function update(Request $request, $id)
    {
        DB::beginTransaction();
        try {

            $entry = TenderEntryMaster::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'Bra_id'          => 'sometimes|exists:main_departments,id',
                'Lg_Id'           => 'sometimes|exists:ledger_master,Lg_Id',
                'tend_work_id'    => 'sometimes|exists:new_tender_works,tend_work_id',
                'Proposal_Amount' => 'sometimes|numeric|min:0',
                'Date'            => 'sometimes|date',
                'Description'     => 'nullable|string',
                'Doc_Lib'         => 'nullable|exists:document_library,Doc_Lib',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'validation_error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->only([
                'Bra_id',
                'Lg_Id',
                'tend_work_id',
                'Proposal_Amount',
                'Date',
                'Description',
                'Doc_Lib'
            ]);

            // 🔥 STRICT RULE
            if ($request->filled('Proposal_Amount')) {
                $data['available_amount'] = $request->Proposal_Amount;
            }

            $entry->update($data);

            DB::commit();

            return response()->json([
                'status'  => 'success',
                'message' => 'Tender entry updated successfully',
                'data'    => $entry
            ], 200);

        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'status'  => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $tender = TenderEntryMaster::findOrFail($id);
            $tenderId = $tender->Tend_Ent_Id;

            DB::beginTransaction();

            // Get documents to delete files
            $documents = DocumentLibrary::where('Menu_Id', $tender->Bra_id)
                ->where('Rec_Id', $tenderId)
                ->get();

            // Delete files from public storage
            foreach ($documents as $document) {
                if (is_string($document->Doc_Upload)) {
                    $uploadData = json_decode($document->Doc_Upload, true);
                    if (isset($uploadData['public_url'])) {
                        // Extract filename from URL
                        $urlParts = parse_url($uploadData['public_url']);
                        if (isset($urlParts['path'])) {
                            $filename = basename($urlParts['path']);
                            $filePath = public_path('storage/tender-documents/' . $filename);
                            if (file_exists($filePath)) {
                                unlink($filePath);
                            }
                        }
                    }
                }
            }

            // Delete database records
            DocumentLibrary::where('Menu_Id', $tender->Bra_id)
                ->where('Rec_Id', $tenderId)
                ->delete();

            $tender->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Tender entry and associated documents deleted successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error deleting tender entry.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get tender statistics
     */
    public function statistics()
    {
        try {
            $totalTenders = TenderEntryMaster::count();
            $totalAmount = TenderEntryMaster::sum('Proposal_Amount');
            $latestTender = TenderEntryMaster::with('department')
                ->orderBy('Date', 'desc')
                ->first();

            // Count total documents for tenders
            $totalDocuments = DocumentLibrary::where('Rec_Id', '!=', '')->count();

            return response()->json([
                'success' => true,
                'data' => [
                    'total_tenders' => $totalTenders,
                    'total_proposal_amount' => $totalAmount,
                    'latest_tender' => $latestTender ? $this->formatTenderData($latestTender) : null,
                    'total_documents' => $totalDocuments,
                    'average_amount' => $totalTenders > 0 ? $totalAmount / $totalTenders : 0
                ],
                'message' => 'Statistics retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving statistics.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get documents for a specific tender entry
     */
    public function getDocuments($id)
    {
        try {
            $tender = TenderEntryMaster::findOrFail($id);
            $tenderId = $tender->Tend_Ent_Id;

            $documents = DocumentLibrary::where('Menu_Id', $tender->Bra_id)
                ->where('Rec_Id', $tenderId)
                ->orderBy('Serial_No', 'asc')
                ->get()
                ->map(function ($document) {
                    $docData = $document->toArray();

                    if (is_string($docData['Doc_Upload'])) {
                        $docData['Doc_Upload'] = json_decode($docData['Doc_Upload'], true);
                    }

                    return $docData;
                });

            return response()->json([
                'success' => true,
                'data' => $documents,
                'message' => 'Tender documents retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving documents.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download a specific document
     */
    public function downloadDocument($docLibId)
    {
        try {
            $document = DocumentLibrary::findOrFail($docLibId);

            if (is_string($document->Doc_Upload)) {
                $uploadData = json_decode($document->Doc_Upload, true);

                if (isset($uploadData['public_url'], $uploadData['original_name'])) {
                    // Extract filename from URL
                    $urlParts = parse_url($uploadData['public_url']);
                    if (isset($urlParts['path'])) {
                        $filename = basename($urlParts['path']);
                        $filePath = public_path('storage/tender-documents/' . $filename);

                        if (file_exists($filePath)) {
                            return response()->download($filePath, $uploadData['original_name']);
                        }
                    }
                }
            }

            return response()->json([
                'success' => false,
                'message' => 'File not found.'
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error downloading document.',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
