<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\UserRoleMapping;
use App\Models\GmcUser;
use App\Models\RoleMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class UserRoleMappingController extends Controller
{
    /**
     * Display a listing of user role mappings.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $perPage = $request->get('per_page', 20);
            $page = $request->get('page', 1);
            $userId = $request->get('user_id', null);
            $roleId = $request->get('role_id', null);

            $query = UserRoleMapping::with(['user', 'role'])
                ->when($userId, function ($query, $userId) {
                    return $query->where('U_Id', $userId);
                })
                ->when($roleId, function ($query, $roleId) {
                    return $query->where('Role_Id', $roleId);
                })
                ->orderBy('URoleMap', 'desc');

            $mappings = $query->paginate($perPage, ['*'], 'page', $page);

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - User Role Mappings fetched successfully',
                'data' => $mappings->items(),
                'meta' => [
                    'current_page' => $mappings->currentPage(),
                    'from' => $mappings->firstItem(),
                    'to' => $mappings->lastItem(),
                    'per_page' => $mappings->perPage(),
                    'total' => $mappings->total(),
                    'last_page' => $mappings->lastPage()
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Role Mapping Fetch Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch role mappings.'
            ], 500);
        }
    }

    /**
     * Assign role to user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'U_Id' => 'required|exists:gmc_users,user_id',
                'Role_Id' => 'required|exists:role_master,Role_Id',
                'Default_Rid' => 'boolean'
            ], [
                'U_Id.required' => 'User ID is required',
                'U_Id.exists' => 'User does not exist',
                'Role_Id.required' => 'Role ID is required',
                'Role_Id.exists' => 'Role does not exist'
            ]);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }

            // Check if mapping already exists
            $existing = UserRoleMapping::where('U_Id', $request->U_Id)
                ->where('Role_Id', $request->Role_Id)
                ->exists();

            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'This role is already assigned to the user'
                ], 409);
            }

            // Get user to check
            $user = GmcUser::find($request->U_Id);

            // If setting as default, remove default from other roles
            $isDefault = $request->boolean('Default_Rid', false);
            if ($isDefault) {
                UserRoleMapping::where('U_Id', $request->U_Id)
                    ->update(['Default_Rid' => 0]);
            }

            // Create mapping
            $mapping = UserRoleMapping::create([
                'U_Id' => $request->U_Id,
                'Role_Id' => $request->Role_Id,
                'Default_Rid' => $isDefault ? 1 : 0
            ]);

            // Load relationships
            $mapping->load(['user', 'role']);

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Role assigned to user successfully',
                'data' => $mapping
            ], 201);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            Log::error('Role Assignment Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to assign role to user.'
            ], 500);
        }
    }

    /**
     * Display specific role mapping.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        try {
            $mapping = UserRoleMapping::with(['user', 'role'])->find($id);

            if (!$mapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role mapping not found.'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Role mapping details retrieved successfully',
                'data' => $mapping
            ], 200);

        } catch (\Exception $e) {
            Log::error('Role Mapping Show Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to retrieve role mapping details.'
            ], 500);
        }
    }

    /**
     * Update role mapping (mainly Default_Rid).
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $mapping = UserRoleMapping::find($id);

            if (!$mapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role mapping not found.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Default_Rid' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // If setting as default, remove default from other roles of same user
            if ($request->has('Default_Rid') && $request->boolean('Default_Rid')) {
                UserRoleMapping::where('U_Id', $mapping->U_Id)
                    ->where('URoleMap', '!=', $id)
                    ->update(['Default_Rid' => 0]);
            }

            $mapping->update([
                'Default_Rid' => $request->boolean('Default_Rid', $mapping->Default_Rid)
            ]);

            $mapping->load(['user', 'role']);

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Role mapping updated successfully',
                'data' => $mapping
            ], 200);

        } catch (\Exception $e) {
            Log::error('Role Mapping Update Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to update role mapping.'
            ], 500);
        }
    }

    /**
     * Remove role from user.
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy($id)
    {
        try {
            $mapping = UserRoleMapping::find($id);

            if (!$mapping) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role mapping not found.'
                ], 404);
            }

            // Check if this is the only role for user
            $userRoleCount = UserRoleMapping::where('U_Id', $mapping->U_Id)->count();

            if ($userRoleCount <= 1) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot remove the only role assigned to user.'
                ], 400);
            }

            // Check if removing default role
            $isDefault = $mapping->Default_Rid === 1;

            $mapping->delete();

            // If default role was removed, set another role as default
            if ($isDefault) {
                $newDefault = UserRoleMapping::where('U_Id', $mapping->U_Id)
                    ->first();

                if ($newDefault) {
                    $newDefault->update(['Default_Rid' => 1]);
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Role removed from user successfully',
                'data' => null
            ], 200);

        } catch (\Exception $e) {
            Log::error('Role Mapping Delete Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to remove role from user.'
            ], 500);
        }
    }

    /**
     * Get user roles with details.
     *
     * @param  int  $userId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getUserRoles($userId)
    {
        try {
            $user = GmcUser::find($userId);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found.'
                ], 404);
            }

            $roles = $user->assignedRoles();
            $defaultRole = $user->defaultRole();

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - User roles retrieved successfully',
                'data' => [
                    'user_id' => $user->user_id,
                    'username' => $user->username,
                    'default_role' => $defaultRole,
                    'assigned_roles' => $roles,
                    'total_roles' => $roles->count()
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Get User Roles Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to retrieve user roles.'
            ], 500);
        }
    }

    /**
     * Get role users.
     *
     * @param  int  $roleId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getRoleUsers($roleId)
    {
        try {
            $role = RoleMaster::find($roleId);

            if (!$role) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role not found.'
                ], 404);
            }

            $users = $role->users()
                ->with(['roleMappings' => function ($query) use ($roleId) {
                    $query->where('Role_Id', $roleId);
                }])
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Role users retrieved successfully',
                'data' => [
                    'role_id' => $role->Role_Id,
                    'role_name' => $role->Role_Name,
                    'users' => $users,
                    'total_users' => $users->count()
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Get Role Users Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to retrieve role users.'
            ], 500);
        }
    }

    /**
     * Assign multiple roles to user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $userId
     * @return \Illuminate\Http\JsonResponse
     */
    public function assignMultipleRoles(Request $request, $userId)
    {
        try {
            $user = GmcUser::find($userId);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'roles' => 'required|array|min:1',
                'roles.*.Role_Id' => 'required|exists:role_master,Role_Id',
                'roles.*.Default_Rid' => 'boolean',
                'set_default' => 'nullable|exists:role_master,Role_Id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $assigned = [];
            $failed = [];
            $defaultSet = false;

            foreach ($request->roles as $index => $roleData) {
                try {
                    // Check if role already assigned
                    if ($user->hasRole($roleData['Role_Id'])) {
                        $failed[] = [
                            'index' => $index,
                            'role_id' => $roleData['Role_Id'],
                            'reason' => 'Role already assigned to user'
                        ];
                        continue;
                    }

                    $isDefault = $roleData['Default_Rid'] ?? false;

                    // If setting as default and no default set yet
                    if ($isDefault && !$defaultSet) {
                        // Remove default from existing roles
                        UserRoleMapping::where('U_Id', $userId)
                            ->update(['Default_Rid' => 0]);
                        $defaultSet = true;
                    } elseif ($isDefault && $defaultSet) {
                        // Can't have multiple defaults
                        $roleData['Default_Rid'] = false;
                    }

                    // Assign role
                    $user->roles()->attach($roleData['Role_Id'], [
                        'Default_Rid' => $roleData['Default_Rid'] ? 1 : 0
                    ]);

                    $assigned[] = $roleData['Role_Id'];

                } catch (\Exception $e) {
                    $failed[] = [
                        'index' => $index,
                        'role_id' => $roleData['Role_Id'],
                        'reason' => $e->getMessage()
                    ];
                }
            }

            // Set specific role as default if requested
            if ($request->has('set_default')) {
                $user->setDefaultRole($request->set_default);
            }

            return response()->json([
                'success' => true,
                'message' => 'Roles assignment completed',
                'data' => [
                    'user_id' => $userId,
                    'username' => $user->username,
                    'assigned_count' => count($assigned),
                    'failed_count' => count($failed),
                    'assigned_roles' => $assigned,
                    'failed_assignments' => $failed
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Multiple Roles Assignment Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to assign multiple roles.'
            ], 500);
        }
    }

    /**
     * Set default role for user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $userId
     * @return \Illuminate\Http\JsonResponse
     */
    public function setDefaultRole(Request $request, $userId)
    {
        try {
            $user = GmcUser::find($userId);

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'Role_Id' => 'required|exists:role_master,Role_Id'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Check if user has this role
            if (!$user->hasRole($request->Role_Id)) {
                return response()->json([
                    'success' => false,
                    'message' => 'User does not have this role assigned.'
                ], 400);
            }

            // Set as default
            $user->setDefaultRole($request->Role_Id);

            $defaultRole = $user->defaultRole();

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Default role set successfully',
                'data' => [
                    'user_id' => $user->user_id,
                    'username' => $user->username,
                    'default_role' => $defaultRole
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Set Default Role Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to set default role.'
            ], 500);
        }
    }

    /**
     * Bulk update user roles.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function bulkUpdate(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'mappings' => 'required|array|min:1',
                'mappings.*.URoleMap' => 'required|exists:user_role_mapping,URoleMap',
                'mappings.*.Default_Rid' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 422);
            }

            $updated = 0;
            $failed = [];

            foreach ($request->mappings as $index => $mappingData) {
                try {
                    $mapping = UserRoleMapping::find($mappingData['URoleMap']);

                    if ($mappingData['Default_Rid'] ?? false) {
                        // If setting as default, remove default from other roles of same user
                        UserRoleMapping::where('U_Id', $mapping->U_Id)
                            ->where('URoleMap', '!=', $mappingData['URoleMap'])
                            ->update(['Default_Rid' => 0]);
                    }

                    $mapping->update([
                        'Default_Rid' => $mappingData['Default_Rid'] ? 1 : 0
                    ]);

                    $updated++;

                } catch (\Exception $e) {
                    $failed[] = [
                        'index' => $index,
                        'mapping_id' => $mappingData['URoleMap'],
                        'reason' => $e->getMessage()
                    ];
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Bulk update completed',
                'data' => [
                    'updated_count' => $updated,
                    'failed_count' => count($failed),
                    'failed_updates' => $failed
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Bulk Update Role Mappings Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to perform bulk update.'
            ], 500);
        }
    }

    /**
     * Get users without specific role.
     *
     * @param  int  $roleId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getUsersWithoutRole($roleId)
    {
        try {
            $role = RoleMaster::find($roleId);

            if (!$role) {
                return response()->json([
                    'success' => false,
                    'message' => 'Role not found.'
                ], 404);
            }

            // Get users who don't have this role
            $users = GmcUser::whereDoesntHave('roles', function ($query) use ($roleId) {
                    $query->where('Role_Id', $roleId);
                })
                ->where('is_active', true)
                ->whereNull('deleted_at')
                ->select('user_id', 'username', 'email', 'mobile')
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'GMC Gadhinagar - Users without role retrieved successfully',
                'data' => [
                    'role_id' => $role->Role_Id,
                    'role_name' => $role->Role_Name,
                    'users' => $users,
                    'total_users' => $users->count()
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Get Users Without Role Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Unable to retrieve users without role.'
            ], 500);
        }
    }
}
