<?php

namespace App\Http\Controllers;

use App\Models\VendorInvoiceParameter;
use App\Models\LedgerMaster;
use App\Models\FaTaxTypeMaster;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class VendorInvoiceParameterController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        try {
            $perPage = $request->get('per_page', 10);
            $search = $request->get('search', '');
            $withTrashed = $request->get('with_trashed', false);
            $sortBy = $request->get('sort_by', 'proposal_id');
            $sortOrder = $request->get('sort_order', 'desc');
            $tendWorkId = $request->get('tend_work_id');
            $date = $request->get('effective_date');

            $validSortColumns = ['proposal_id', 'effective_date', 'created_at', 'updated_at'];
            $sortBy = in_array($sortBy, $validSortColumns) ? $sortBy : 'proposal_id';
            $sortOrder = in_array(strtolower($sortOrder), ['asc', 'desc']) ? strtolower($sortOrder) : 'desc';

            // UPDATED: Load all related models with specific columns
            $query = VendorInvoiceParameter::with([
                'tenderWork:tend_work_id,work_code,work_name'
            ]);

            if ($withTrashed) {
                $query->withTrashed();
            }

            if ($search) {
                $query->where(function ($q) use ($search) {
                    $q->where('tend_work_id', 'like', "%{$search}%")
                        ->orWhereHas('tenderWork', function ($q1) use ($search) {
                            $q1->where('work_code', 'like', "%{$search}%")
                                ->orWhere('work_name', 'like', "%{$search}%");
                        });
                });
            }

            if ($tendWorkId) {
                $query->where('tend_work_id', $tendWorkId);
            }

            if ($date) {
                $query->whereDate('effective_date', $date);
            }

            $query->orderBy($sortBy, $sortOrder);

            $parameters = $query->paginate($perPage);

            // NEW: Load tax type and ledger data for each parameter
            $transformedData = $parameters->getCollection()->map(function ($item) {
                return $this->transformParameterWithDetails($item);
            });

            $parameters->setCollection($transformedData);

            return response()->json([
                'success' => true,
                'message' => 'Vendor Invoice Parameters retrieved successfully.',
                'data' => $parameters->items(),
                'meta' => [
                    'current_page' => $parameters->currentPage(),
                    'last_page' => $parameters->lastPage(),
                    'per_page' => $parameters->perPage(),
                    'total' => $parameters->total(),
                    'from' => $parameters->firstItem(),
                    'to' => $parameters->lastItem(),
                    'sort_by' => $sortBy,
                    'sort_order' => $sortOrder
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve Vendor Invoice Parameters.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make(
            $request->all(),
            VendorInvoiceParameter::createRules(),
            VendorInvoiceParameter::validationMessages()
        );

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed.',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Validate and format tax parameters
            $taxParameters = $this->validateAndFormatTaxParameters($request->tax_parameters);

            if (empty($taxParameters)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No valid tax parameters provided.'
                ], 422);
            }

            // Check for exact duplicate
            $existing = VendorInvoiceParameter::isExactDuplicate(
                $request->tend_work_id,
                $request->effective_date,
                $taxParameters
            );

            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'An identical tax parameter set already exists.',
                    'error_type' => 'duplicate'
                ], 409);
            }

            $parameter = VendorInvoiceParameter::create([
                'tend_work_id' => $request->tend_work_id,
                'effective_date' => $request->effective_date,
                'tax_parameters' => $taxParameters
            ]);

            // NEW: Load all related data
            $parameter->load(['tenderWork']);

            return response()->json([
                'success' => true,
                'message' => 'Tax parameters created successfully.',
                'data' => $this->transformParameterWithDetails($parameter)
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create tax parameters.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): JsonResponse
    {
        try {
            $parameter = VendorInvoiceParameter::with([
                'tenderWork:tend_work_id,work_code,work_name'
            ])->withTrashed()->find($id);

            if (!$parameter) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tax parameter not found.'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Tax parameter retrieved successfully.',
                'data' => $this->transformParameterWithDetails($parameter)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve tax parameter.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Transform parameter data with all related details
     */
    private function transformParameterWithDetails($parameter): array
    {
        $taxParameters = [];
        if (is_array($parameter->tax_parameters)) {
            foreach ($parameter->tax_parameters as $param) {
                $taxType = FaTaxTypeMaster::where('tax_type_id', $param['tax_type_id'])->first();
                $ledger = LedgerMaster::where('Lg_Id', $param['Lg_Id'])->first();

                $taxParameters[] = [
                    'tax_type_id' => $param['tax_type_id'] ?? null,
                    'tax_type_details' => $taxType ? [
                        'tax_type_id' => $taxType->tax_type_id,
                        'tax_code' => $taxType->tax_code,
                        'tax_name' => $taxType->tax_name,
                        'tax_percentage' => $taxType->tax_percentage,
                        'tax_percentage_formatted' => number_format($taxType->tax_percentage, 2) . '%'
                    ] : null,
                    'tax_code' => $param['tax_code'] ?? null,
                    'add_deduct' => $param['add_deduct'] ?? 0,
                    'add_deduct_text' => VendorInvoiceParameter::ADD_DEDUCT_TEXTS[$param['add_deduct'] ?? 0] ?? 'Unknown',
                    'Lg_Id' => $param['Lg_Id'] ?? null,
                    'ledger_details' => $ledger ? [
                        'Lg_Id' => $ledger->Lg_Id,
                        'Lg_Code' => $ledger->Lg_Code,
                        'Lg_Name' => $ledger->Lg_Name,
                        'Lg_Address' => $ledger->Lg_Address,
                        'Lg_City' => $ledger->Lg_City,
                        'Lg_Pin' => $ledger->Lg_Pin,
                        'Lg_Phone' => $ledger->Lg_Phone,
                        'Lg_Email' => $ledger->Lg_Email,
                        'Lg_Status' => $ledger->Lg_Status,
                        'status_text' => $ledger->getStatusTextAttribute(),
                        'full_address' => $ledger->getFullAddressAttribute()
                    ] : null,
                    'percentage' => isset($param['percentage']) ? (float)$param['percentage'] : null,
                    'percentage_formatted' => isset($param['percentage']) ? number_format($param['percentage'], 2) . '%' : null,
                    'effective_percentage' => isset($param['percentage']) ?
                        (($param['add_deduct'] == VendorInvoiceParameter::DEDUCT ? -$param['percentage'] : $param['percentage'])) : 0,
                    'calculation_formula' => $param['calculation_formula'] ?? null,
                ];
            }
        }

        return [
            'proposal_id' => $parameter->proposal_id,
            'tend_work_id' => $parameter->tend_work_id,
            'tender_work' => $parameter->tenderWork ? [
                'tend_work_id' => $parameter->tenderWork->tend_work_id,
                'work_code' => $parameter->tenderWork->work_code,
                'work_name' => $parameter->tenderWork->work_name
            ] : null,
            'effective_date' => $parameter->effective_date ? $parameter->effective_date->format('Y-m-d') : null,
            'effective_date_formatted' => $parameter->effective_date ? $parameter->effective_date->format('d-m-Y') : null,
            'tax_parameters' => $taxParameters,
            'total_parameters' => $parameter->total_parameters,
            'total_percentage' => $parameter->total_percentage,
            'total_percentage_formatted' => number_format($parameter->total_percentage, 2) . '%',
            'created_at' => $parameter->created_at ? $parameter->created_at->format('Y-m-d H:i:s') : null,
            'updated_at' => $parameter->updated_at ? $parameter->updated_at->format('Y-m-d H:i:s') : null,
            'deleted_at' => $parameter->deleted_at ? $parameter->deleted_at->format('Y-m-d H:i:s') : null,
            'is_deleted' => !is_null($parameter->deleted_at),
            'is_active' => is_null($parameter->deleted_at)
        ];
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        try {
            $parameter = VendorInvoiceParameter::find($id);

            if (!$parameter) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tax parameter not found.'
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'tend_work_id' => 'sometimes|required|exists:new_tender_works,tend_work_id',
                'effective_date' => 'sometimes|required|date|date_format:Y-m-d',
                'tax_parameters' => 'sometimes|required|array|min:1',
                'tax_parameters.*.tax_type_id' => 'required_with:tax_parameters|string|max:20|exists:fa_tax_type_master,tax_type_id',
                'tax_parameters.*.add_deduct' => 'required_with:tax_parameters|in:0,1',
                'tax_parameters.*.Lg_Id' => 'required_with:tax_parameters|exists:ledger_master,Lg_Id',
                'tax_parameters.*.percentage' => 'nullable|numeric|min:0|max:999999.99|regex:/^\d+(\.\d{1,2})?$/',
                'tax_parameters.*.tax_code' => 'nullable|string|max:20',
                'tax_parameters.*.calculation_formula' => 'nullable|string|max:255'
            ], VendorInvoiceParameter::validationMessages());

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed.',
                    'errors' => $validator->errors()
                ], 422);
            }

            $updateData = [];
            $fields = ['tend_work_id', 'effective_date', 'tax_parameters'];

            foreach ($fields as $field) {
                if ($request->has($field)) {
                    $updateData[$field] = $request->$field;
                }
            }

            // If updating tax_parameters, validate and format them
            if (isset($updateData['tax_parameters'])) {
                $taxParameters = $this->validateAndFormatTaxParameters($updateData['tax_parameters']);
                if (empty($taxParameters)) {
                    return response()->json([
                        'success' => false,
                        'message' => 'No valid tax parameters provided.'
                    ], 422);
                }
                $updateData['tax_parameters'] = $taxParameters;
            }

            // Check for duplicate if updating
            if (!empty($updateData)) {
                $tendWorkId = $updateData['tend_work_id'] ?? $parameter->tend_work_id;
                $effectiveDate = $updateData['effective_date'] ?? $parameter->effective_date;
                $taxParameters = $updateData['tax_parameters'] ?? $parameter->tax_parameters;

                $isDuplicate = VendorInvoiceParameter::isExactDuplicate(
                    $tendWorkId,
                    $effectiveDate,
                    $taxParameters,
                    $id
                );

                if ($isDuplicate) {
                    return response()->json([
                        'success' => false,
                        'message' => 'An identical tax parameter set already exists.',
                        'error_type' => 'duplicate'
                    ], 409);
                }
            }

            $parameter->update($updateData);
            $parameter->refresh();
            $parameter->load(['tenderWork']);

            return response()->json([
                'success' => true,
                'message' => 'Tax parameter updated successfully.',
                'data' => $this->transformParameter($parameter)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update tax parameter.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $parameter = VendorInvoiceParameter::find($id);

            if (!$parameter) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tax parameter not found.'
                ], 404);
            }

            $parameter->delete();

            return response()->json([
                'success' => true,
                'message' => 'Tax parameter deleted successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete tax parameter.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    public function getByTenderWorkAndDate($tendWorkId, $date): JsonResponse
    {
        try {
            $validator = Validator::make([
                'tend_work_id' => $tendWorkId,
                'effective_date' => $date
            ], [
                'tend_work_id' => 'required|exists:new_tender_works,tend_work_id',
                'effective_date' => 'required|date|date_format:Y-m-d'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed.',
                    'errors' => $validator->errors()
                ], 422);
            }

            // CORRECTED: Select specific columns
            $parameters = VendorInvoiceParameter::with([
                'tenderWork:tend_work_id,work_code,work_name'
            ])
                ->where('tend_work_id', $tendWorkId)
                ->whereDate('effective_date', $date)
                ->whereNull('deleted_at')
                ->orderBy('effective_date', 'desc')
                ->get();

            $transformedData = $parameters->map(function ($item) {
                return $this->transformParameter($item);
            });

            return response()->json([
                'success' => true,
                'message' => 'Tax parameters retrieved successfully.',
                'data' => $transformedData,
                'meta' => [
                    'tend_work_id' => $tendWorkId,
                    'effective_date' => $date,
                    'total_entries' => $parameters->count()
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve tax parameters.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Transform parameter data for API response.
     */
    private function transformParameter($parameter): array
    {
        $taxParameters = [];
        if (is_array($parameter->tax_parameters)) {
            foreach ($parameter->tax_parameters as $param) {
                $taxParameters[] = VendorInvoiceParameter::transformParameterItem($param);
            }
        }

        return [
            'proposal_id' => $parameter->proposal_id,
            'tend_work_id' => $parameter->tend_work_id,
            'tender_work' => $parameter->tenderWork ? [
                'tend_work_id' => $parameter->tenderWork->tend_work_id,
                'work_code' => $parameter->tenderWork->work_code,
                'work_name' => $parameter->tenderWork->work_name
            ] : null,
            'effective_date' => $parameter->effective_date ? $parameter->effective_date->format('Y-m-d') : null,
            'effective_date_formatted' => $parameter->effective_date ? $parameter->effective_date->format('d-m-Y') : null,
            'tax_parameters' => $taxParameters,
            'total_parameters' => $parameter->total_parameters,
            'total_percentage' => $parameter->total_percentage,
            'total_percentage_formatted' => number_format($parameter->total_percentage, 2) . '%',
            'created_at' => $parameter->created_at ? $parameter->created_at->format('Y-m-d H:i:s') : null,
            'updated_at' => $parameter->updated_at ? $parameter->updated_at->format('Y-m-d H:i:s') : null,
            'deleted_at' => $parameter->deleted_at ? $parameter->deleted_at->format('Y-m-d H:i:s') : null,
            'is_deleted' => !is_null($parameter->deleted_at),
            'is_active' => is_null($parameter->deleted_at)
        ];
    }

    /**
     * Validate and format tax parameters
     */
    private function validateAndFormatTaxParameters(array $taxParameters): array
    {
        $formatted = [];

        foreach ($taxParameters as $param) {
            // Validate required fields
            if (!isset($param['tax_type_id'], $param['add_deduct'], $param['Lg_Id'])) {
                continue;
            }

            // Format the parameter
            $formattedParam = [
                'tax_type_id' => (string)$param['tax_type_id'],
                'add_deduct' => (int)$param['add_deduct'],
                'Lg_Id' => (int)$param['Lg_Id'],
                'percentage' => isset($param['percentage']) ? (float)$param['percentage'] : null
            ];

            // Add optional fields if present
            if (isset($param['tax_code'])) {
                $formattedParam['tax_code'] = (string)$param['tax_code'];
            }

            if (isset($param['calculation_formula'])) {
                $formattedParam['calculation_formula'] = (string)$param['calculation_formula'];
            }

            $formatted[] = $formattedParam;
        }

        return $formatted;
    }

    /**
     * Restore soft deleted parameter
     */
    public function restore(string $id): JsonResponse
    {
        try {
            $parameter = VendorInvoiceParameter::withTrashed()->find($id);

            if (!$parameter) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tax parameter not found.'
                ], 404);
            }

            if (!$parameter->trashed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tax parameter is not deleted.'
                ], 400);
            }

            $parameter->restore();
            $parameter->load(['tenderWork']);

            return response()->json([
                'success' => true,
                'message' => 'Tax parameter restored successfully.',
                'data' => $this->transformParameter($parameter)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to restore tax parameter.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Permanently delete parameter
     */
    public function forceDelete(string $id): JsonResponse
    {
        try {
            $parameter = VendorInvoiceParameter::withTrashed()->find($id);

            if (!$parameter) {
                return response()->json([
                    'success' => false,
                    'message' => 'Tax parameter not found.'
                ], 404);
            }

            $parameter->forceDelete();

            return response()->json([
                'success' => true,
                'message' => 'Tax parameter permanently deleted successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to permanently delete tax parameter.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }
}
