<?php

namespace App\Http\Controllers;

use App\Models\WardMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class WardMasterController extends Controller
{
    public function index(Request $request)
    {
        $query = WardMaster::with(['zone', 'zone.organisation'])
                    ->where('is_active', true);

        // Filter by zone if provided
        if ($request->has('zone_id')) {
            $query->where('zone_id', $request->zone_id);
        }

        // Search functionality
        if ($request->has('search')) {
            $query->search($request->search);
        }

        // Pagination
        $perPage = $request->get('per_page', 15);
        $wards = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $wards,
            'message' => 'Wards retrieved successfully'
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'Ward_Code' => 'nullable|string|max:5',
            'Ward_Name' => 'required|string|max:25',
            'Zone_Id' => 'required|exists:zone_masters,Zone_Id',
        ], WardMaster::validationMessages());

        // Custom validation for unique ward code in zone
        $validator->after(function ($validator) use ($request) {
            if (!WardMaster::validateUniqueWardCode($request->Ward_Code, $request->Zone_Id)) {
                $validator->errors()->add('Ward_Code', 'This ward code already exists in the selected zone');
            }
        });

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $ward = WardMaster::create([
            'Ward_Code' => $request->Ward_Code,
            'Ward_Name' => $request->Ward_Name,
            'Zone_Id' => $request->Zone_Id,
            'is_active' => true
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Ward created successfully',
            'data' => $ward->load(['zone', 'zone.organisation'])
        ], 201);
    }

    public function show($id)
    {
        $ward = WardMaster::with(['zone', 'zone.organisation', 'cccs'])->find($id);

        if (!$ward) {
            return response()->json([
                'success' => false,
                'message' => 'Ward not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $ward,
            'message' => 'Ward retrieved successfully'
        ]);
    }

    public function update(Request $request, $id)
    {
        $ward = WardMaster::find($id);

        if (!$ward) {
            return response()->json([
                'success' => false,
                'message' => 'Ward not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'Ward_Code' => 'sometimes|required|string|max:5',
            'Ward_Name' => 'sometimes|required|string|max:25',
            'Zone_Id' => 'sometimes|required|exists:zone_masters,Zone_Id',
            'is_active' => 'sometimes|boolean'
        ], WardMaster::validationMessages());

        // Custom validation for unique ward code in zone
        $validator->after(function ($validator) use ($request, $id) {
            if ($request->has('Ward_Code') && $request->has('Zone_Id')) {
                if (!WardMaster::validateUniqueWardCode($request->Ward_Code, $request->Zone_Id, $id)) {
                    $validator->errors()->add('Ward_Code', 'This ward code already exists in the selected zone');
                }
            }
        });

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $ward->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Ward updated successfully',
            'data' => $ward->load(['zone', 'zone.organisation'])
        ]);
    }

    public function destroy($id)
    {
        $ward = WardMaster::find($id);

        if (!$ward) {
            return response()->json([
                'success' => false,
                'message' => 'Ward not found'
            ], 404);
        }

        if (!$ward->canBeDeleted()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete ward as it has dependent records'
            ], 400);
        }

        $ward->deactivate();

        return response()->json([
            'success' => true,
            'message' => 'Ward deactivated successfully'
        ]);
    }

    public function restore($id)
    {
        $ward = WardMaster::withTrashed()->find($id);

        if (!$ward) {
            return response()->json([
                'success' => false,
                'message' => 'Ward not found'
            ], 404);
        }

        $ward->activate();

        return response()->json([
            'success' => true,
            'message' => 'Ward activated successfully',
            'data' => $ward->load(['zone', 'zone.organisation'])
        ]);
    }

    // Get wards by zone
    public function getByZone($zoneId)
    {
        $wards = WardMaster::with('zone')
                    ->where('Zone_Id', $zoneId)
                    ->where('is_active', true)
                    ->get();

        return response()->json([
            'success' => true,
            'data' => $wards,
            'message' => 'Wards retrieved successfully'
        ]);
    }
}
