<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\WorkAllocation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class WorkAllocationController extends Controller
{
    /**
     * 🔥 SL-WISE GROUPED WORK ALLOCATION
     */
    public function index()
    {
        $data = WorkAllocation::with([
            'subLedger',
            'subLedger.ledger',
            'tenderEntry',
            'tenderEntry.department',
            'tenderEntry.tenderWork'
        ])
        ->get()
        ->groupBy('SL_Id')
        ->map(function ($items, $slId) {

            $subLedger = optional($items->first())->subLedger;

            return [
                'SL_Id'        => $slId,
                'sub_ledger'   => $subLedger,
                'total_amount' => $items->sum('work_amount'),

                // 🔹 Tender Entry wise breakup
                'tenders' => $items->groupBy('Tend_Ent_Id')->map(function ($tItems) {
                    $tender = optional($tItems->first())->tenderEntry;

                    return [
                        'Tend_Ent_Id' => $tender?->Tend_Ent_Id,
                        'tender'     => $tender,
                        'total_work' => $tItems->sum('work_amount'),
                        'allocations'=> $tItems->values()
                    ];
                })->values()
            ];
        })->values();

        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }

    /**
     * CREATE
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'SL_Id'        => 'required|exists:subledg_master,SL_Id',
            'Tend_Ent_Id'  => 'required|exists:tender_entry_master,Tend_Ent_Id',
            'work_amount' => 'nullable|numeric|min:0.01',
            'work_date'   => 'nullable|date'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = WorkAllocation::create($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Work allocation created successfully',
            'data' => $data
        ], 201);
    }

    /**
     * SHOW
     */
    public function show($id)
    {
        $data = WorkAllocation::with(['subLedger', 'tenderEntry'])->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }

    /**
     * UPDATE
     */
    public function update(Request $request, $id)
    {
        $allocation = WorkAllocation::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'work_amount' => 'nullable|numeric|min:0.01',
            'work_date'   => 'nullable|date'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $allocation->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Work allocation updated successfully'
        ]);
    }

    /**
     * DELETE
     */
    public function destroy($id)
    {
        WorkAllocation::findOrFail($id)->delete();

        return response()->json([
            'success' => true,
            'message' => 'Work allocation deleted successfully'
        ]);
    }
}
