<?php

namespace App\Http\Controllers;

use App\Models\ZoneMaster;
use App\Models\OrganisationSetting;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Validator;

class ZoneMasterController extends Controller
{
    /**
     * Display a listing of the resource with pagination.
     */
    public function index(Request $request)
    {
        // Start building query
        $query = ZoneMaster::query();

        // Eager load organization
        $query->with('organisation');

        // Apply filters
        if ($request->has('org_id')) {
            if ($request->org_id === 'null' || $request->org_id === '') {
                $query->whereNull('Org_Id');
            } else {
                $query->where('Org_Id', $request->org_id);
            }
        }

        if ($request->has('zone_code') && !empty($request->zone_code)) {
            $query->where('Zone_Code', 'like', '%' . $request->zone_code . '%');
        }

        if ($request->has('zone_name') && !empty($request->zone_name)) {
            $query->where('Zone_Name', 'like', '%' . $request->zone_name . '%');
        }

        // Filter global zones
        if ($request->has('global') && $request->global == 'true') {
            $query->whereNull('Org_Id');
        }

        // Search functionality
        if ($request->has('search') && !empty($request->search)) {
            $query->search($request->search);
        }

        // Apply sorting
        $sortField = $request->get('sort_by', 'Zone_Id');
        $sortDirection = $request->get('sort_dir', 'desc');

        $allowedSortFields = ['Zone_Id', 'Zone_Code', 'Zone_Name', 'Org_Id', 'created_at', 'updated_at'];
        if (!in_array($sortField, $allowedSortFields)) {
            $sortField = 'Zone_Id';
        }

        $sortDirection = strtolower($sortDirection) === 'asc' ? 'asc' : 'desc';
        $query->orderBy($sortField, $sortDirection);

        // Pagination
        $perPage = $request->get('per_page', 15);
        $perPage = min(max(1, $perPage), 100);

        $zones = $query->paginate($perPage);

        // Transform data
        $zones->getCollection()->transform(function ($zone) {
            $zone->full_zone = $zone->full_zone;
            $zone->organization_name = $zone->organization_name;
            $zone->is_global = $zone->isGlobal();
            return $zone;
        });

        return response()->json([
            'success' => true,
            'message' => 'Zones retrieved successfully',
            'data' => $zones
        ], Response::HTTP_OK);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Validate request data
        $validator = Validator::make($request->all(), [
            'Zone_Code' => 'nullable|string|max:5',
            'Zone_Name' => 'required|string|max:25',
            'Org_Id' => 'nullable|integer|exists:organisation_settings,Org_Id'
        ], [
            'Zone_Code.required' => 'Zone code is required',
            'Zone_Code.max' => 'Zone code cannot exceed 5 characters',
            'Zone_Name.required' => 'Zone name is required',
            'Zone_Name.max' => 'Zone name cannot exceed 25 characters',
            'Org_Id.exists' => 'The selected organization does not exist'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], Response::HTTP_BAD_REQUEST);
        }

        // Get validated data
        $validated = $validator->validated();

        // Check if Zone_Code is unique within organization
        $isUnique = ZoneMaster::validateUniqueZoneCode(
            $validated['Zone_Code'],
            $validated['Org_Id'] ?? null
        );

        if (!$isUnique) {
            $orgText = $validated['Org_Id'] ? ' for the selected organization' : ' as a global zone';
            return response()->json([
                'success' => false,
                'message' => "Zone code '{$validated['Zone_Code']}' already exists{$orgText}"
            ], Response::HTTP_BAD_REQUEST);
        }

        // Create zone
        $zone = ZoneMaster::create($validated);

        // Load organization relationship
        $zone->load('organisation');

        return response()->json([
            'success' => true,
            'message' => 'Zone created successfully',
            'data' => $zone
        ], Response::HTTP_CREATED);
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        if (!is_numeric($id) || $id <= 0) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid zone ID'
            ], Response::HTTP_BAD_REQUEST);
        }

        $zone = ZoneMaster::with('organisation')->find($id);

        if (!$zone) {
            return response()->json([
                'success' => false,
                'message' => 'Zone not found'
            ], Response::HTTP_NOT_FOUND);
        }

        // Add computed attributes
        $zone->full_zone = $zone->full_zone;
        $zone->organization_name = $zone->organization_name;
        $zone->is_global = $zone->isGlobal();

        return response()->json([
            'success' => true,
            'message' => 'Zone retrieved successfully',
            'data' => $zone
        ], Response::HTTP_OK);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        if (!is_numeric($id) || $id <= 0) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid zone ID'
            ], Response::HTTP_BAD_REQUEST);
        }

        $zone = ZoneMaster::find($id);

        if (!$zone) {
            return response()->json([
                'success' => false,
                'message' => 'Zone not found'
            ], Response::HTTP_NOT_FOUND);
        }

        // Validation rules
        $validator = Validator::make($request->all(), [
            'Zone_Code' => 'sometimes|required|string|max:5',
            'Zone_Name' => 'sometimes|required|string|max:25',
            'Org_Id' => 'nullable|integer|exists:organisation_settings,Org_Id'
        ], [
            'Zone_Code.max' => 'Zone code cannot exceed 5 characters',
            'Zone_Name.max' => 'Zone name cannot exceed 25 characters',
            'Org_Id.exists' => 'The selected organization does not exist'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], Response::HTTP_BAD_REQUEST);
        }

        // Get validated data
        $validated = $validator->validated();

        // Check for duplicate Zone_Code if it's being updated
        if (isset($validated['Zone_Code'])) {
            $orgId = isset($validated['Org_Id']) ? $validated['Org_Id'] : $zone->Org_Id;

            $isUnique = ZoneMaster::validateUniqueZoneCode(
                $validated['Zone_Code'],
                $orgId,
                $id
            );

            if (!$isUnique) {
                $orgText = $orgId ? ' for the selected organization' : ' as a global zone';
                return response()->json([
                    'success' => false,
                    'message' => "Zone code '{$validated['Zone_Code']}' already exists{$orgText}"
                ], Response::HTTP_BAD_REQUEST);
            }
        }

        // Update zone
        $zone->update($validated);

        // Refresh with organization
        $zone->refresh();
        $zone->load('organisation');

        return response()->json([
            'success' => true,
            'message' => 'Zone updated successfully',
            'data' => $zone
        ], Response::HTTP_OK);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        if (!is_numeric($id) || $id <= 0) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid zone ID'
            ], Response::HTTP_BAD_REQUEST);
        }

        $zone = ZoneMaster::find($id);

        if (!$zone) {
            return response()->json([
                'success' => false,
                'message' => 'Zone not found'
            ], Response::HTTP_NOT_FOUND);
        }

        // Check if zone can be deleted
        if (!$zone->canBeDeleted()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete zone because it has associated records'
            ], Response::HTTP_CONFLICT);
        }

        // Soft delete
        $zone->delete();

        return response()->json([
            'success' => true,
            'message' => 'Zone deleted successfully'
        ], Response::HTTP_OK);
    }

    /**
     * Get zones by organization.
     */
    public function getByOrganization(Request $request, $orgId)
    {
        if ($orgId === 'null' || $orgId === '') {
            // Get global zones
            $query = ZoneMaster::withoutOrganization();
        } else {
            if (!is_numeric($orgId) || $orgId <= 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid organization ID'
                ], Response::HTTP_BAD_REQUEST);
            }

            // Check if organization exists
            $organization = OrganisationSetting::find($orgId);
            if (!$organization) {
                return response()->json([
                    'success' => false,
                    'message' => 'Organization not found'
                ], Response::HTTP_NOT_FOUND);
            }

            $query = ZoneMaster::byOrganization($orgId);
        }

        // Eager load organization
        $query->with('organisation');

        // Apply search if provided
        if ($request->has('search') && !empty($request->search)) {
            $query->search($request->search);
        }

        // Sorting
        $sortField = $request->get('sort_by', 'Zone_Code');
        $sortDirection = $request->get('sort_dir', 'asc');
        $query->orderBy($sortField, $sortDirection);

        // Get all or paginate
        if ($request->has('paginate') && $request->paginate === 'false') {
            $zones = $query->get();
        } else {
            $perPage = $request->get('per_page', 15);
            $zones = $query->paginate($perPage);
        }

        // Transform data
        $zonesCollection = $zones instanceof \Illuminate\Pagination\LengthAwarePaginator
            ? $zones->getCollection()
            : $zones;

        $zonesCollection->transform(function ($zone) {
            $zone->full_zone = $zone->full_zone;
            $zone->organization_name = $zone->organization_name;
            $zone->is_global = $zone->isGlobal();
            return $zone;
        });

        return response()->json([
            'success' => true,
            'message' => 'Zones retrieved successfully',
            'data' => $zones
        ], Response::HTTP_OK);
    }

    /**
     * Get global zones (without organization).
     */
    public function getGlobalZones(Request $request)
    {
        $query = ZoneMaster::withoutOrganization();

        // Apply search if provided
        if ($request->has('search') && !empty($request->search)) {
            $query->search($request->search);
        }

        // Sorting
        $sortField = $request->get('sort_by', 'Zone_Code');
        $sortDirection = $request->get('sort_dir', 'asc');
        $query->orderBy($sortField, $sortDirection);

        // Pagination
        $perPage = $request->get('per_page', 15);
        $zones = $query->paginate($perPage);

        // Transform data
        $zones->getCollection()->transform(function ($zone) {
            $zone->full_zone = $zone->full_zone;
            $zone->organization_name = $zone->organization_name;
            $zone->is_global = true;
            return $zone;
        });

        return response()->json([
            'success' => true,
            'message' => 'Global zones retrieved successfully',
            'data' => $zones
        ], Response::HTTP_OK);
    }

    /**
     * Bulk create zones.
     */
    public function bulkStore(Request $request)
    {
        // Validate request data
        $validator = Validator::make($request->all(), [
            'zones' => 'required|array|min:1',
            'zones.*.Zone_Code' => 'required|string|max:5',
            'zones.*.Zone_Name' => 'required|string|max:25',
            'zones.*.Org_Id' => 'nullable|integer|exists:organisation_settings,Org_Id'
        ], [
            'zones.required' => 'Zones data is required',
            'zones.array' => 'Zones must be an array',
            'zones.min' => 'At least one zone is required',
            'zones.*.Zone_Code.required' => 'Zone code is required for all zones',
            'zones.*.Zone_Name.required' => 'Zone name is required for all zones',
            'zones.*.Org_Id.exists' => 'Organization does not exist'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], Response::HTTP_BAD_REQUEST);
        }

        $zonesData = $request->zones;
        $createdZones = [];
        $errors = [];

        foreach ($zonesData as $index => $zoneData) {
            // Check if Zone_Code is unique within organization
            $isUnique = ZoneMaster::validateUniqueZoneCode(
                $zoneData['Zone_Code'],
                $zoneData['Org_Id'] ?? null
            );

            if (!$isUnique) {
                $orgText = isset($zoneData['Org_Id']) ? ' for the selected organization' : ' as a global zone';
                $errors[] = "Row {$index}: Zone code '{$zoneData['Zone_Code']}' already exists{$orgText}";
                continue;
            }

            try {
                $zone = ZoneMaster::create($zoneData);
                $createdZones[] = $zone;
            } catch (\Exception $e) {
                $errors[] = "Row {$index}: " . $e->getMessage();
            }
        }

        if (!empty($errors)) {
            return response()->json([
                'success' => false,
                'message' => 'Some zones failed to create',
                'errors' => $errors,
                'created_count' => count($createdZones),
                'failed_count' => count($errors)
            ], Response::HTTP_BAD_REQUEST);
        }

        return response()->json([
            'success' => true,
            'message' => count($createdZones) . ' zone(s) created successfully',
            'data' => $createdZones
        ], Response::HTTP_CREATED);
    }
}
