<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class BudgetEntry extends Model
{
    use HasFactory;

    protected $table = 'budget_entries';
    protected $primaryKey = 'Budg_Id';
    public $timestamps = true;

    protected $fillable = [
        'Fin_Year',
        'Dept_Id',
        'Ledg_ID',
        'Amount',
        'Sug_Amount',
        'Beg_Doc',
        'Status',
        'Ref_Id',
        'Is_Latest', 
    ];

    protected $casts = [
        'Budg_Id' => 'integer',
        'Fin_Year' => 'integer',
        'Dept_Id' => 'integer',
        'Ledg_ID' => 'integer',
        'Amount' => 'decimal:2',
        'Sug_Amount' => 'decimal:2',
        'Beg_Doc' => 'array',
        'Status' => 'string',
        'Ref_Id' => 'integer',
        'Is_Latest' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // Status mapping - Updated with new statuses
    const STATUS_DRAFT = 'D';
    const STATUS_REVISED = 'R';
    const STATUS_APPROVE = 'P';
    const STATUS_FREEZE = 'F';
    const STATUS_ACTIVE = 'A';
    const STATUS_INACTIVE = 'I';
    const STATUS_RE_REVISED = 'RR';
    const STATUS_REJECT = 'RJ';
    const STATUS_JUSTIFICATION = 'J';

    // Store old attributes temporarily
    private $oldAttributes = [];

    // ============================================
    // RELATIONSHIPS
    // ============================================

    public function financialYear()
    {
        return $this->belongsTo(FinancialYearMaster::class, 'Fin_Year', 'financial_year_id');
    }

    public function department()
    {
        return $this->belongsTo(MainDepartment::class, 'Dept_Id', 'id');
    }

    public function ledger()
    {
        return $this->belongsTo(LedgerMaster::class, 'Ledg_ID', 'Lg_Id');
    }

    public function logEntries()
    {
        return $this->hasMany(BudgetLogBook::class, 'Budg_Id', 'Budg_Id')
                    ->orderBy('Date', 'desc')
                    ->orderBy('BL_Id', 'desc');
    }

    public function parentEntry()
    {
        return $this->belongsTo(BudgetEntry::class, 'Ref_Id', 'Budg_Id');
    }

    public function childEntries()
    {
        return $this->hasMany(BudgetEntry::class, 'Ref_Id', 'Budg_Id');
    }

    // ============================================
    // BUSINESS LOGIC METHODS
    // ============================================

    /**
     * Get full status text from code - Updated
     */
    public function getFullStatusText($statusCode)
    {
        $statusMap = [
            self::STATUS_DRAFT => 'Draft',
            self::STATUS_REVISED => 'Revised',
            self::STATUS_APPROVE => 'Approve',
            self::STATUS_FREEZE => 'Freeze',
            self::STATUS_ACTIVE => 'Active',
            self::STATUS_INACTIVE => 'Inactive',
            self::STATUS_RE_REVISED => 'Re-Revised',
            self::STATUS_REJECT => 'Reject',
            self::STATUS_JUSTIFICATION => 'Justification'
        ];

        return $statusMap[$statusCode] ?? $statusCode;
    }

    /**
     * Get version number (1-based)
     */
    public function getVersionNumberAttribute()
    {
        if ($this->Ref_Id == 0) {
            return 1;
        }

        // Count all ancestors to get version number
        $version = 2; // Starting from 2 because Ref_Id != 0
        $current = $this;

        while ($current->parentEntry) {
            $current = $current->parentEntry;
            $version++;
        }

        return $version;
    }

    /**
     * Get all ancestors (parent chain)
     */
    public function getAncestors()
    {
        $ancestors = [];
        $current = $this->parentEntry;

        while ($current) {
            $ancestors[] = $current;
            $current = $current->parentEntry;
        }

        return collect(array_reverse($ancestors));
    }

    /**
     * Get root entry (first version)
     */
    public function getRootEntryAttribute()
    {
        $current = $this;

        while ($current->parentEntry) {
            $current = $current->parentEntry;
        }

        return $current;
    }

    /**
     * Get version history including current
     */
    public function getVersionHistory()
    {
        $root = $this->getRootEntryAttribute();
        $versions = collect([$root]);

        $current = $root;
        while ($latestChild = $current->childEntries()->where('Is_Latest', false)->first()) {
            $versions->push($latestChild);
            $current = $latestChild;
        }

        // Add current if not already included
        if ($this->Budg_Id != $current->Budg_Id && $this->Budg_Id != $root->Budg_Id) {
            $versions->push($this);
        }

        return $versions;
    }

    // ============================================
    // ACCESSORS & MUTATORS
    // ============================================

    public function getFormattedAmountAttribute()
    {
        return '₹ ' . number_format($this->Amount, 2);
    }

    public function getFormattedSugAmountAttribute()
    {
        return $this->Sug_Amount ? '₹ ' . number_format($this->Sug_Amount, 2) : null;
    }

    public function getStatusTextAttribute()
    {
        return $this->getFullStatusText($this->Status);
    }

    public function setBegDocAttribute($value)
    {
        if (is_string($value)) {
            $value = json_decode($value, true);
        }

        $this->attributes['Beg_Doc'] = json_encode($value ?: []);
    }

    public function getBegDocAttribute($value)
    {
        if (is_string($value)) {
            return json_decode($value, true) ?: [];
        }

        return $value ?: [];
    }

    public function getHasDocumentsAttribute()
    {
        return !empty($this->Beg_Doc) && is_array($this->Beg_Doc) && count($this->Beg_Doc) > 0;
    }

    public function getDocumentCountAttribute()
    {
        return is_array($this->Beg_Doc) ? count($this->Beg_Doc) : 0;
    }

    public function getIsWorkflowStatusAttribute()
    {
        // All statuses except Active and Inactive are workflow statuses
        return !in_array($this->Status, ['A', 'I']);
    }

    public function getIsActiveAttribute()
    {
        return $this->Status == 'A';
    }

    public function getCanEditAttribute()
    {
        // Define which statuses allow editing
        $editableStatuses = ['D', 'R', 'RR', 'J'];
        return in_array($this->Status, $editableStatuses);
    }

    // ============================================
    // SCOPES
    // ============================================

    public function scopeByBudgetId($query, $budgId)
    {
        return $query->where('Budg_Id', $budgId);
    }

    public function scopeLatestVersions($query)
    {
        // Get only latest versions
        return $query->where('Is_Latest', true);
    }

    public function scopeVersionHistory($query, $rootId)
    {
        // Get all versions of a budget entry
        return $query->where(function($q) use ($rootId) {
            $q->where('Budg_Id', $rootId)
              ->orWhere('Ref_Id', $rootId)
              ->orWhere('Ref_Id', function($subquery) use ($rootId) {
                  $subquery->select('Budg_Id')
                          ->from('budget_entries')
                          ->where('Ref_Id', $rootId);
              });
        })->orderBy('created_at', 'asc');
    }

    // ============================================
    // BOOT METHOD - UPDATED
    // ============================================

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            // If Budg_Id is not set, generate new one
            if (!$model->Budg_Id) {
                $lastEntry = BudgetEntry::orderBy('Budg_Id', 'desc')->first();
                $model->Budg_Id = $lastEntry ? $lastEntry->Budg_Id + 1 : 1;
            }

            // Set default values for new entry
            if (!isset($model->Ref_Id)) {
                $model->Ref_Id = 0; // First version
            }
            if (!isset($model->Is_Latest)) {
                $model->Is_Latest = true; // By default, new entry is latest
            }
        });

        static::created(function ($model) {
            $userId = Auth::check() ? Auth::id() : null;

            BudgetLogBook::create([
                'Budg_Id' => $model->Budg_Id,
                'Status' => $model->Status, // Store code, not text
                'previous_status' => null,
                'Date' => now(),
                'user_id' => $userId,
                'budget_amount' => 0,
                'updated_amount' => $model->Amount,
                'suggested_amount' => $model->Sug_Amount,
                'remarks' => 'Budget entry ' . ($model->Ref_Id == 0 ? 'created' : 'version created') . ' with status: ' . $model->getFullStatusText($model->Status)
            ]);
        });

        static::updating(function ($model) {
            // Store old values in class property
            $model->oldAttributes = $model->getOriginal();
        });

        static::updated(function ($model) {
            // Check if we have old attributes
            if (!isset($model->oldAttributes) || empty($model->oldAttributes)) {
                return;
            }

            $userId = Auth::check() ? Auth::id() : null;
            $oldAttributes = $model->oldAttributes;

            // Check what changed
            $changes = [];
            $remarks = 'Budget entry updated: ';

            if (isset($oldAttributes['Amount']) && (float) $oldAttributes['Amount'] != (float) $model->Amount) {
                $changes[] = 'Amount changed from ' . $oldAttributes['Amount'] . ' to ' . $model->Amount;
                $remarks .= 'Amount updated from ' . $oldAttributes['Amount'] . ' to ' . $model->Amount . '. ';
            }

            if (isset($oldAttributes['Sug_Amount']) && $oldAttributes['Sug_Amount'] != $model->Sug_Amount) {
                $oldSug = $oldAttributes['Sug_Amount'] ?? 'null';
                $newSug = $model->Sug_Amount ?? 'null';
                $changes[] = 'Suggested amount changed from ' . $oldSug . ' to ' . $newSug;
                $remarks .= 'Suggested amount updated from ' . $oldSug . ' to ' . $newSug . '. ';
            }

            if (isset($oldAttributes['Status']) && $oldAttributes['Status'] != $model->Status) {
                $oldStatus = $model->getFullStatusText($oldAttributes['Status']);
                $newStatus = $model->getFullStatusText($model->Status);
                $changes[] = 'Status changed from ' . $oldStatus . ' to ' . $newStatus;
                $remarks .= 'Status changed from ' . $oldStatus . ' to ' . $newStatus . '. ';
            }

            // If no specific changes were detected, use generic message
            if (empty($changes)) {
                $remarks = 'Budget entry updated';
            }

            // Create log entry - Store STATUS CODE
            BudgetLogBook::create([
                'Budg_Id' => $model->Budg_Id,
                'Status' => $model->Status, // Store code
                'previous_status' => isset($oldAttributes['Status']) ? $oldAttributes['Status'] : null, // Store code
                'Date' => now(),
                'user_id' => $userId,
                'budget_amount' => $oldAttributes['Amount'] ?? 0,
                'updated_amount' => $model->Amount,
                'suggested_amount' => $model->Sug_Amount,
                'remarks' => trim($remarks)
            ]);
        });
    }
}
