<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class DepartmentManagement extends Model
{
    use HasFactory;

    protected $table = 'department_management';
    protected $primaryKey = 'Dept_Mng_Id';
    public $timestamps = true;

    protected $fillable = [
        'Dept_Id',
        'Wef_Date',
        'Dept_User_Id',
        'Dept_Post_ID',
        'reporting_user_id', // Changed from Reporting
        'designation_id',      // ✅ ADD
        'reporting_designation_id',

    ];

    protected $casts = [
        'Dept_Id' => 'integer',
        'Dept_User_Id' => 'integer',
        'Dept_Post_ID' => 'integer',
        'reporting_user_id' => 'integer', // Changed from Reporting
        'Wef_Date' => 'date',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'designation_id' => 'integer',   // ✅ ADD
        'reporting_designation_id' => 'integer',

    ];

    protected $dates = [
        'Wef_Date',
        'created_at',
        'updated_at',
    ];

    // ============================================
    // RELATIONSHIPS
    // ============================================

    /**
     * Relationship with MainDepartment
     */
    public function department()
    {
        return $this->belongsTo(MainDepartment::class, 'Dept_Id', 'id');
    }

    /**
     * Relationship with GmcUser (current user)
     */
    public function user()
    {
        return $this->belongsTo(GmcUser::class, 'Dept_User_Id', 'user_id');
    }

    /**
     * Relationship with EmployeeDesignationMaster
     */
    // public function designation()
    // {
    //     return $this->belongsTo(EmployeeDesignationMaster::class, 'Dept_Post_ID', 'designation_id');
    // }
    public function designation()
    {
        return $this->belongsTo(DesignationMenu::class, 'designation_id', 'id');
    }

    /**
     * Relationship with Reporting Manager (GmcUser)
     */
    public function reportingManager()
    {
        return $this->belongsTo(GmcUser::class, 'reporting_user_id', 'user_id');
    }
    public function reportingDesignation()
    {
        return $this->belongsTo(DesignationMenu::class, 'reporting_designation_id', 'id');
    }
    /**
     * Get all subordinates (users who report to this user)
     */
    public function subordinates()
    {
        return $this->hasMany(DepartmentManagement::class, 'reporting_user_id', 'Dept_User_Id');
    }

    /**
     * Get reporting manager's department management record
     */
    public function reportingManagerAssignment()
    {
        return $this->belongsTo(DepartmentManagement::class, 'reporting_user_id', 'Dept_User_Id');
    }

    // ============================================
    // SCOPES
    // ============================================

    /**
     * Scope for active department assignments
     */
    public function scopeActive($query)
    {
        return $query->whereHas('department', function ($q) {
            $q->where('isvisible', 1);
        });
    }

    /**
     * Scope for department wise filtering
     */
    public function scopeByDepartment($query, $deptId)
    {
        return $query->where('Dept_Id', $deptId);
    }

    /**
     * Scope for user wise filtering
     */
    public function scopeByUser($query, $userId)
    {
        return $query->where('Dept_User_Id', $userId);
    }

    /**
     * Scope for designation wise filtering
     */
    public function scopeByDesignation($query, $designationId)
    {
        return $query->where('Dept_Post_ID', $designationId);
    }

    /**
     * Scope for reporting manager wise filtering
     */
    public function scopeByReportingManager($query, $managerId)
    {
        return $query->where('reporting_user_id', $managerId);
    }

    /**
     * Scope for current assignments (based on Wef_Date)
     */
    public function scopeCurrent($query)
    {
        return $query->whereDate('Wef_Date', '<=', now())
            ->orderBy('Wef_Date', 'desc');
    }

    // ============================================
    // ACCESSORS & MUTATORS
    // ============================================

    /**
     * Get Wef_Date in specific format
     */
    public function getFormattedWefDateAttribute()
    {
        return $this->Wef_Date ? $this->Wef_Date->format('d-m-Y') : null;
    }

    /**
     * Set Wef_Date attribute
     */
    public function setWefDateAttribute($value)
    {
        $this->attributes['Wef_Date'] = \Carbon\Carbon::parse($value);
    }

    /**
     * Check if user has subordinates
     */
    public function getHasSubordinatesAttribute()
    {
        return $this->subordinates()->count() > 0;
    }

    /**
     * Get reporting manager name
     */
    public function getReportingManagerNameAttribute()
    {
        return $this->reportingManager ? $this->reportingManager->name : null;
    }

    /**
     * Get full reporting hierarchy path
     */
    public function getReportingHierarchyAttribute()
    {
        $hierarchy = [];
        $current = $this;

        while ($current && $current->reportingManagerAssignment) {
            $hierarchy[] = $current->reportingManagerAssignment;
            $current = $current->reportingManagerAssignment;
        }

        return collect($hierarchy);
    }
}
