<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class EmployeeMaster extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'employee_master';
    protected $primaryKey = 'emp_id';

    protected $fillable = [
        'emp_code',
        'first_name',
        'last_name',
        'dept_id',
        'designation_id',
        'grade_id',
        'emp_type_id',
        'doj',
        'is_active'
    ];

    protected $casts = [
        'doj' => 'date',
        'is_active' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime'
    ];

    protected $appends = ['full_name'];

    /**
     * Get the employee's full name.
     */
    public function getFullNameAttribute()
    {
        return trim($this->first_name . ' ' . $this->last_name);
    }

    /**
     * Relationship with FamilyInfoMaster (Family Members)
     */
    public function familyMembers()
    {
        return $this->hasMany(FamilyInfoMaster::class, 'emp_id', 'emp_id');
    }

    /**
     * Relationship with Department
     */
    public function department()
    {
        return $this->belongsTo(Department::class, 'dept_id', 'Dept_Id');
    }

    /**
     * Relationship with Designation - FIXED: Added proper foreign key mapping
     */
    public function designation()
    {
        return $this->belongsTo(DesignationMaster::class, 'designation_id', 'designation_id');
    }

    /**
     * Relationship with Grade
     */
    public function grade()
    {
        return $this->belongsTo(EmployeeGradeMaster::class, 'grade_id', 'grade_id');
    }

    /**
     * Relationship with Employee Type
     */
    public function employeeType()
    {
        return $this->belongsTo(EmployeeTypeMaster::class, 'emp_type_id', 'emp_type_id');
    }

    /**
     * Scope for active employees
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for inactive employees
     */
    public function scopeInactive($query)
    {
        return $query->where('is_active', false);
    }

    /**
     * Scope for searching employees
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('emp_code', 'like', "%{$search}%")
                ->orWhere('first_name', 'like', "%{$search}%")
                ->orWhere('last_name', 'like', "%{$search}%")
                ->orWhereRaw("CONCAT(first_name, ' ', last_name) LIKE ?", ["%{$search}%"]);
        });
    }

    /**
     * Scope for date range filtering
     */
    public function scopeDateRange($query, $startDate, $endDate = null)
    {
        if ($endDate) {
            return $query->whereBetween('doj', [$startDate, $endDate]);
        }

        return $query->where('doj', '>=', $startDate);
    }
}
