<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;

class GmcUser extends Authenticatable
{
    use HasFactory, SoftDeletes, Notifiable;

    protected $table = 'gmc_users';
    protected $primaryKey = 'user_id';
    public $timestamps = true;

    protected $fillable = [
        'username',
        'email',
        'mobile',
        'password_hash',
        'Role_Id',
        'department_id',
        'is_active',
        'is_locked',
        'locked_reason',
        'locked_until',
        'failed_login_attempts',
        'last_login',
        'last_password_change',
        'mfa_enabled',
        'mfa_secret',
        'session_timeout',
        'password_expiry_days',
        'created_by',
        'updated_by'
    ];

    protected $hidden = [
        'password_hash',
        'mfa_secret',
        'deleted_at',
        'deleted_by',
        'deletion_reason'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'is_locked' => 'boolean',
        'locked_until' => 'datetime',
        'failed_login_attempts' => 'integer',
        'last_login' => 'datetime',
        'last_password_change' => 'datetime',
        'mfa_enabled' => 'boolean',
        'session_timeout' => 'integer',
        'password_expiry_days' => 'integer',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime'
    ];

    protected $appends = [
        'full_name',
        'account_status',
        'days_since_last_login',
        'password_expiry_days_left'
    ];

    /**
     * Mutator for password
     */
    public function setPasswordHashAttribute($value)
    {
        $this->attributes['password_hash'] = Hash::make($value);
    }

    /**
     * Accessor for password
     */
    public function getAuthPassword()
    {
        return $this->password_hash;
    }

    /**
     * Accessor for full name (if you add first_name, last_name fields)
     */
    public function getFullNameAttribute()
    {
        // If you add name fields later
        // return trim($this->first_name . ' ' . $this->last_name);
        return $this->username; // Placeholder
    }

    /**
     * Accessor for account status
     */
    public function getAccountStatusAttribute()
    {
        if (!$this->is_active) {
            return 'Inactive';
        }

        if ($this->is_locked) {
            return 'Locked';
        }

        if ($this->trashed()) {
            return 'Deleted';
        }

        return 'Active';
    }

    /**
     * Accessor for days since last login
     */
    public function getDaysSinceLastLoginAttribute()
    {
        if (!$this->last_login) {
            return 'Never';
        }

        return $this->last_login->diffInDays(Carbon::now());
    }

    /**
     * Accessor for password expiry days left
     */
    public function getPasswordExpiryDaysLeftAttribute()
    {
        if (!$this->last_password_change) {
            return $this->password_expiry_days;
        }

        $expiryDate = $this->last_password_change->addDays($this->password_expiry_days);
        $daysLeft = Carbon::now()->diffInDays($expiryDate, false);

        return max(0, $daysLeft);
    }

    /**
     * Relationship with RoleMaster (Single role from gmc_users table)
     */
    public function singleRole()
    {
        return $this->belongsTo(RoleMaster::class, 'Role_Id', 'Role_Id');
    }

    /**
     * Relationship with RoleMappings (Multiple roles)
     */
    public function roleMappings()
    {
        return $this->hasMany(UserRoleMapping::class, 'U_Id', 'user_id');
    }

    /**
     * Relationship with Roles through mapping (Multiple roles)
     */
    public function roles()
    {
        return $this->belongsToMany(
            RoleMaster::class,
            'user_role_mapping',
            'U_Id',
            'Role_Id',
            'user_id',
            'Role_Id'
        )->withPivot('Default_Rid', 'URoleMap', 'created_at', 'updated_at');
    }

    /**
     * Relationship with UserSessions
     */
    public function sessions()
    {
        return $this->hasMany(GmcUserSession::class, 'user_id', 'user_id');
    }

    /**
     * Relationship with AuditLogs
     */
    public function auditLogs()
    {
        return $this->hasMany(GmcUserAuditLog::class, 'user_id', 'user_id');
    }

    /**
     * Relationship with Created Users
     */
    public function createdUsers()
    {
        return $this->hasMany(GmcUser::class, 'created_by', 'user_id');
    }

    /**
     * Relationship with Updated Users
     */
    public function updatedUsers()
    {
        return $this->hasMany(GmcUser::class, 'updated_by', 'user_id');
    }

    /**
     * Relationship with Deleted Users
     */
    public function deletedUsers()
    {
        return $this->hasMany(GmcUser::class, 'deleted_by', 'user_id');
    }

    /**
     * Get default role from mapping
     */
   public function defaultRole()
{
    return $this->belongsToMany(
        RoleMaster::class,
        'user_role_mapping',
        'U_Id',
        'Role_Id',
        'user_id',
        'Role_Id'
    )->wherePivot('Default_Rid', 1)
     ->withPivot('Default_Rid', 'URoleMap', 'created_at', 'updated_at')
     ->limit(1);
}

    /**
     * Get all assigned roles
     */
    public function assignedRoles()
    {
        return $this->roles()
            ->orderByPivot('Default_Rid', 'desc')
            ->orderByPivot('created_at', 'asc')
            ->get();
    }

    /**
     * Get non-default roles
     */
    public function nonDefaultRoles()
    {
        return $this->roles()
            ->wherePivot('Default_Rid', 0)
            ->get();
    }

    /**
     * Check if user has role
     */
    public function hasRole($roleId)
    {
        return $this->roles()
            ->where('Role_Id', $roleId)
            ->exists();
    }

    /**
     * Check if user has any of given roles
     */
    public function hasAnyRole(array $roleIds)
    {
        return $this->roles()
            ->whereIn('Role_Id', $roleIds)
            ->exists();
    }

    /**
     * Check if user has all given roles
     */
    public function hasAllRoles(array $roleIds)
    {
        $userRoleIds = $this->roles()->pluck('Role_Id')->toArray();
        return count(array_intersect($roleIds, $userRoleIds)) === count($roleIds);
    }

    /**
     * Assign role to user
     */
    public function assignRole($roleId, $isDefault = false)
    {
        // Check if role already assigned
        if ($this->hasRole($roleId)) {
            return false;
        }

        // If setting as default, remove default from other roles
        if ($isDefault) {
            UserRoleMapping::where('U_Id', $this->user_id)
                ->update(['Default_Rid' => 0]);
        }

        // Assign new role
        return $this->roles()->attach($roleId, [
            'Default_Rid' => $isDefault ? 1 : 0
        ]);
    }

    /**
     * Remove role from user
     */
    public function removeRole($roleId)
    {
        $mapping = $this->roleMappings()
            ->where('Role_Id', $roleId)
            ->first();

        if (!$mapping) {
            return false;
        }

        $wasDefault = $mapping->Default_Rid === 1;
        $result = $mapping->delete();

        // If default role was removed, set another role as default
        if ($wasDefault && $result) {
            $newDefault = $this->roleMappings()->first();
            if ($newDefault) {
                $newDefault->update(['Default_Rid' => 1]);
            }
        }

        return $result;
    }

    /**
     * Set default role
     */
    public function setDefaultRole($roleId)
    {
        // Check if user has this role
        if (!$this->hasRole($roleId)) {
            return false;
        }

        // Remove default from all roles
        UserRoleMapping::where('U_Id', $this->user_id)
            ->update(['Default_Rid' => 0]);

        // Set new default
        return UserRoleMapping::where('U_Id', $this->user_id)
            ->where('Role_Id', $roleId)
            ->update(['Default_Rid' => 1]);
    }

    /**
     * Check if account is locked
     */
    public function isAccountLocked()
    {
        if (!$this->is_locked) {
            return false;
        }

        if ($this->locked_until && $this->locked_until->isPast()) {
            $this->unlockAccount();
            return false;
        }

        return true;
    }

    /**
     * Unlock account
     */
    public function unlockAccount()
    {
        $this->update([
            'is_locked' => false,
            'locked_reason' => null,
            'locked_until' => null,
            'failed_login_attempts' => 0
        ]);
    }

    /**
     * Increment failed login attempts
     */
    public function incrementFailedAttempts()
    {
        $this->increment('failed_login_attempts');

        if ($this->failed_login_attempts >= 5) {
            $this->lockAccount('Too many failed login attempts');
        }
    }

    /**
     * Lock account
     */
    public function lockAccount($reason = null)
    {
        $this->update([
            'is_locked' => true,
            'locked_reason' => $reason,
            'locked_until' => now()->addMinutes(30),
            'failed_login_attempts' => 5
        ]);

        // Invalidate all active sessions
        $this->sessions()->where('is_active', true)->update(['is_active' => false]);
    }

    /**
     * Reset failed attempts on successful login
     */
    public function resetFailedAttempts()
    {
        $this->update(['failed_login_attempts' => 0]);
    }

    /**
     * Check if password is expired
     */
    public function isPasswordExpired()
    {
        if (!$this->last_password_change) {
            return true;
        }

        return $this->last_password_change->addDays($this->password_expiry_days)->isPast();
    }

    /**
     * Get active sessions
     */
    public function activeSessions()
    {
        return $this->sessions()
            ->where('is_active', true)
            ->where('expires_at', '>', now())
            ->get();
    }

    /**
     * Invalidate all sessions
     */
    public function invalidateAllSessions()
    {
        return $this->sessions()
            ->where('is_active', true)
            ->update(['is_active' => false]);
    }

    /**
     * Get recent audit logs
     */
    public function recentAuditLogs($limit = 20)
    {
        return $this->auditLogs()
            ->orderBy('performed_at', 'desc')
            ->limit($limit)
            ->get();
    }

    /**
     * Scope for active users
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true)
                     ->whereNull('deleted_at');
    }

    /**
     * Scope for inactive users
     */
    public function scopeInactive($query)
    {
        return $query->where('is_active', false)
                     ->whereNull('deleted_at');
    }

    /**
     * Scope for locked users
     */
    public function scopeLocked($query)
    {
        return $query->where('is_locked', true)
                     ->where('locked_until', '>', now())
                     ->whereNull('deleted_at');
    }

    /**
     * Scope for not locked users
     */
    public function scopeNotLocked($query)
    {
        return $query->where(function ($q) {
            $q->where('is_locked', false)
              ->orWhere('locked_until', '<', now());
        })->whereNull('deleted_at');
    }

    /**
     * Scope for users with expired password
     */
    public function scopePasswordExpired($query)
    {
        return $query->whereNotNull('last_password_change')
                     ->whereRaw('last_password_change + INTERVAL password_expiry_days DAY < NOW()')
                     ->whereNull('deleted_at');
    }

    /**
     * Scope for users with MFA enabled
     */
    public function scopeMfaEnabled($query)
    {
        return $query->where('mfa_enabled', true)
                     ->whereNull('deleted_at');
    }

    /**
     * Scope for users with MFA disabled
     */
    public function scopeMfaDisabled($query)
    {
        return $query->where('mfa_enabled', false)
                     ->whereNull('deleted_at');
    }

    /**
     * Scope for users created by specific user
     */
    public function scopeCreatedBy($query, $userId)
    {
        return $query->where('created_by', $userId);
    }

    /**
     * Scope for users with specific role
     */
    public function scopeWithRole($query, $roleId)
    {
        return $query->whereHas('roles', function ($q) use ($roleId) {
            $q->where('Role_Id', $roleId);
        });
    }

    /**
     * Scope for users with default role
     */
    public function scopeWithDefaultRole($query, $roleId)
    {
        return $query->whereHas('roles', function ($q) use ($roleId) {
            $q->where('Role_Id', $roleId)
              ->where('default_rid', 1);
        });
    }

    /**
     * Search users by username, email, or mobile
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('username', 'like', "%{$search}%")
              ->orWhere('email', 'like', "%{$search}%")
              ->orWhere('mobile', 'like', "%{$search}%");
        });
    }

    /**
     * Get users with role count
     */
    public function scopeWithRoleCount($query)
    {
        return $query->withCount('roles');
    }

    /**
     * Get users with session count
     */
    public function scopeWithSessionCount($query)
    {
        return $query->withCount('sessions');
    }

    /**
     * Get users with active session count
     */
    public function scopeWithActiveSessionCount($query)
    {
        return $query->withCount(['sessions as active_sessions_count' => function ($query) {
            $query->where('is_active', true)
                  ->where('expires_at', '>', now());
        }]);
    }

    /**
     * Bulk assign roles to user
     */
    public function bulkAssignRoles(array $roleData)
    {
        $results = [
            'assigned' => [],
            'failed' => [],
            'skipped' => []
        ];

        foreach ($roleData as $index => $data) {
            try {
                $roleId = $data['Role_Id'];
                $isDefault = $data['Default_Rid'] ?? false;

                // Check if already has role
                if ($this->hasRole($roleId)) {
                    $results['skipped'][] = [
                        'role_id' => $roleId,
                        'reason' => 'Already assigned'
                    ];
                    continue;
                }

                // Assign role
                $this->assignRole($roleId, $isDefault);
                $results['assigned'][] = $roleId;

            } catch (\Exception $e) {
                $results['failed'][] = [
                    'index' => $index,
                    'role_id' => $roleId,
                    'error' => $e->getMessage()
                ];
            }
        }

        return $results;
    }

    /**
     * Bulk remove roles from user
     */
    public function bulkRemoveRoles(array $roleIds)
    {
        $results = [
            'removed' => [],
            'failed' => []
        ];

        foreach ($roleIds as $roleId) {
            try {
                $success = $this->removeRole($roleId);

                if ($success) {
                    $results['removed'][] = $roleId;
                } else {
                    $results['failed'][] = [
                        'role_id' => $roleId,
                        'error' => 'Role not assigned to user'
                    ];
                }

            } catch (\Exception $e) {
                $results['failed'][] = [
                    'role_id' => $roleId,
                    'error' => $e->getMessage()
                ];
            }
        }

        return $results;
    }

    /**
     * Get user permissions (through roles)
     */
    public function permissions()
    {
        // If you have a permission system
        // return $this->roles->flatMap->permissions->unique();
        return collect(); // Placeholder
    }

    /**
     * Check if user has permission
     */
    public function hasPermission($permission)
    {
        // Implement based on your permission system
        return $this->roles->contains(function ($role) use ($permission) {
            return $role->hasPermission($permission);
        });
    }

    /**
     * Get user's highest role (by hierarchy)
     */
    public function highestRole()
    {
        return $this->roles->sortBy('hierarchy_level')->first();
    }

    /**
     * Check if user can manage another user
     */
    public function canManageUser(GmcUser $otherUser)
    {
        $myHighestRole = $this->highestRole();
        $theirHighestRole = $otherUser->highestRole();

        if (!$myHighestRole || !$theirHighestRole) {
            return false;
        }

        return $myHighestRole->canManage($theirHighestRole);
    }

    /**
     * Get all manageable users
     */
    public function manageableUsers()
    {
        $myHighestRole = $this->highestRole();

        if (!$myHighestRole) {
            return collect();
        }

        return GmcUser::where('user_id', '!=', $this->user_id)
            ->get()
            ->filter(function ($user) {
                return $this->canManageUser($user);
            });
    }

    /**
     * Log user activity
     */
    public function logActivity($actionType, $actionDetails, $ipAddress = null, $userAgent = null)
    {
        return GmcUserAuditLog::logAction(
            $this->user_id,
            $actionType,
            $actionDetails,
            $ipAddress,
            $userAgent
        );
    }

    /**
     * Create new session
     */
    public function createSession($token, $ipAddress = null, $userAgent = null, $rememberMe = false)
    {
        $timeout = $rememberMe ? 60 * 24 * 30 : $this->session_timeout;

        return GmcUserSession::create([
            'user_id' => $this->user_id,
            'session_token' => $token,
            'ip_address' => $ipAddress,
            'user_agent' => $userAgent,
            'device_info' => $userAgent,
            'login_time' => now(),
            'last_activity' => now(),
            'expires_at' => now()->addMinutes($timeout),
            'is_active' => true
        ]);
    }

    /**
     * Soft delete user with reason
     */
    public function softDelete($deletedBy, $reason)
    {
        $this->update([
            'deleted_at' => now(),
            'deleted_by' => $deletedBy,
            'deletion_reason' => $reason,
            'is_active' => false,
            'updated_by' => $deletedBy
        ]);

        // Invalidate all sessions
        $this->invalidateAllSessions();

        return $this;
    }

    /**
     * Restore soft deleted user
     */
    public function restoreUser($restoredBy)
    {
        $this->update([
            'deleted_at' => null,
            'deleted_by' => null,
            'deletion_reason' => null,
            'is_active' => true,
            'updated_by' => $restoredBy
        ]);

        return $this;
    }


public function department()
{
    return $this->belongsTo(
        \App\Models\MainDepartment::class,
        'department_id',
        'id'
    );
}

}
