<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class GrantDivisionMapping extends Model
{
    use HasFactory;

    protected $table = 'grant_division_mapping';
    protected $primaryKey = 'gt_dv_id';

    protected $fillable = [
        'grant_ty_id',
        'dept_ids',  // JSON array for multiple departments
        'status'
    ];

    protected $casts = [
        'status' => 'integer',
        'dept_ids' => 'array',  // Cast JSON to array
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    // Relationship with FasGrantTypeMaster
    public function grantType()
    {
        return $this->belongsTo(FasGrantTypeMaster::class, 'grant_ty_id', 'grant_type_id');
    }

    // Get all departments from dept_ids array (now referencing main_departments)
    public function departments()
    {
        if (!empty($this->dept_ids)) {
            return MainDepartment::whereIn('id', $this->dept_ids)->get();
        }

        return collect();
    }

    // Get single department (first one) for backward compatibility
    public function getDepartmentAttribute()
    {
        if (!empty($this->dept_ids)) {
            return MainDepartment::find($this->dept_ids[0]);
        }

        return null;
    }

    // Helper methods
    public function addDepartment($departmentId)
    {
        $deptIds = $this->dept_ids ?? [];

        if (!in_array($departmentId, $deptIds)) {
            $deptIds[] = $departmentId;
            $this->dept_ids = $deptIds;
            return $this->save();
        }

        return false;
    }

    public function removeDepartment($departmentId)
    {
        $deptIds = $this->dept_ids ?? [];
        $index = array_search($departmentId, $deptIds);

        if ($index !== false) {
            unset($deptIds[$index]);
            $this->dept_ids = array_values($deptIds); // Re-index array
            return $this->save();
        }

        return false;
    }

    // Check if department exists in mapping
    public function hasDepartment($departmentId)
    {
        $deptIds = $this->dept_ids ?? [];
        return in_array($departmentId, $deptIds);
    }

    // Get department IDs as array
    public function getDepartmentIdsAttribute()
    {
        return $this->dept_ids ?? [];
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    public function scopeInactive($query)
    {
        return $query->where('status', 0);
    }

    public function scopeByGrantType($query, $grantTypeId)
    {
        return $query->where('grant_ty_id', $grantTypeId);
    }

    public function scopeByDepartment($query, $departmentId)
    {
        // Search in dept_ids JSON array
        return $query->whereJsonContains('dept_ids', $departmentId);
    }

    // Scope for multiple departments
    public function scopeByMultipleDepartments($query, array $departmentIds)
    {
        return $query->where(function($q) use ($departmentIds) {
            foreach ($departmentIds as $deptId) {
                $q->orWhereJsonContains('dept_ids', $deptId);
            }
        });
    }
}
