<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class GrantEntry extends Model
{
    use HasFactory;

    protected $table = 'grant_entries';
    protected $primaryKey = 'Grant_Id';
    public $timestamps = true;

    protected $fillable = [
        'Fin_Year',
        'Dept_Id',
        'Ledg_ID',
        'Amount',
        'Sug_Amount',
        'Beg_Doc',
        'Status',
        'Ref_Id',
        'Is_Latest',
    ];

    protected $casts = [
        'Grant_Id' => 'integer',
        'Fin_Year' => 'integer',
        'Dept_Id' => 'integer',
        'Ledg_ID' => 'integer',
        'Amount' => 'decimal:2',
        'Sug_Amount' => 'decimal:2',
        'Beg_Doc' => 'array',
        'Status' => 'string',
        'Ref_Id' => 'integer',
        'Is_Latest' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // Status mapping
    const STATUS_DRAFT = 'D';
    const STATUS_REVISED = 'R';
    const STATUS_APPROVE = 'P';
    const STATUS_FREEZE = 'F';
    const STATUS_ACTIVE = 'A';
    const STATUS_INACTIVE = 'I';
    const STATUS_RE_REVISED = 'RR';
    const STATUS_REJECT = 'RJ';
    const STATUS_JUSTIFICATION = 'J';

    // ============================================
    // RELATIONSHIPS (REMOVED logEntries)
    // ============================================

    public function financialYear()
    {
        return $this->belongsTo(FinancialYearMaster::class, 'Fin_Year', 'financial_year_id');
    }

    public function department()
    {
        return $this->belongsTo(MainDepartment::class, 'Dept_Id', 'id');
    }

    public function ledger()
    {
        return $this->belongsTo(LedgerMaster::class, 'Ledg_ID', 'Lg_Id');
    }

    // NO logEntries relationship
    // public function logEntries()
    // {
    //     return $this->hasMany(GrantLogBook::class, 'Grant_Id', 'Grant_Id')
    //                 ->orderBy('Date', 'desc')
    //                 ->orderBy('GL_Id', 'desc');
    // }

    public function parentEntry()
    {
        return $this->belongsTo(GrantEntry::class, 'Ref_Id', 'Grant_Id');
    }

    public function childEntries()
    {
        return $this->hasMany(GrantEntry::class, 'Ref_Id', 'Grant_Id');
    }

    // ============================================
    // BUSINESS LOGIC METHODS
    // ============================================

    /**
     * Get full status text from code
     */
    public function getFullStatusText($statusCode)
    {
        $statusMap = [
            self::STATUS_DRAFT => 'Draft',
            self::STATUS_REVISED => 'Revised',
            self::STATUS_APPROVE => 'Approve',
            self::STATUS_FREEZE => 'Freeze',
            self::STATUS_ACTIVE => 'Active',
            self::STATUS_INACTIVE => 'Inactive',
            self::STATUS_RE_REVISED => 'Re-Revised',
            self::STATUS_REJECT => 'Reject',
            self::STATUS_JUSTIFICATION => 'Justification'
        ];

        return $statusMap[$statusCode] ?? $statusCode;
    }

    /**
     * Get version number (1-based)
     */
    public function getVersionNumberAttribute()
    {
        if ($this->Ref_Id == 0) {
            return 1;
        }

        // Count all ancestors to get version number
        $version = 2; // Starting from 2 because Ref_Id != 0
        $current = $this;

        while ($current->parentEntry) {
            $current = $current->parentEntry;
            $version++;
        }

        return $version;
    }

    /**
     * Get all ancestors (parent chain)
     */
    public function getAncestors()
    {
        $ancestors = [];
        $current = $this->parentEntry;

        while ($current) {
            $ancestors[] = $current;
            $current = $current->parentEntry;
        }

        return collect(array_reverse($ancestors));
    }

    /**
     * Get root entry (first version)
     */
    public function getRootEntryAttribute()
    {
        $current = $this;

        while ($current->parentEntry) {
            $current = $current->parentEntry;
        }

        return $current;
    }

    /**
     * Get version history including current
     */
    public function getVersionHistory()
    {
        $root = $this->getRootEntryAttribute();
        $versions = collect([$root]);

        $current = $root;
        while ($latestChild = $current->childEntries()->where('Is_Latest', false)->first()) {
            $versions->push($latestChild);
            $current = $latestChild;
        }

        // Add current if not already included
        if ($this->Grant_Id != $current->Grant_Id && $this->Grant_Id != $root->Grant_Id) {
            $versions->push($this);
        }

        return $versions;
    }

    // ============================================
    // ACCESSORS & MUTATORS
    // ============================================

    public function getFormattedAmountAttribute()
    {
        return '₹ ' . number_format($this->Amount, 2);
    }

    public function getFormattedSugAmountAttribute()
    {
        return $this->Sug_Amount ? '₹ ' . number_format($this->Sug_Amount, 2) : null;
    }

    public function getStatusTextAttribute()
    {
        return $this->getFullStatusText($this->Status);
    }

    public function setBegDocAttribute($value)
    {
        if (is_string($value)) {
            $value = json_decode($value, true);
        }

        $this->attributes['Beg_Doc'] = json_encode($value ?: []);
    }

    public function getBegDocAttribute($value)
    {
        if (is_string($value)) {
            return json_decode($value, true) ?: [];
        }

        return $value ?: [];
    }

    public function getHasDocumentsAttribute()
    {
        return !empty($this->Beg_Doc) && is_array($this->Beg_Doc) && count($this->Beg_Doc) > 0;
    }

    public function getDocumentCountAttribute()
    {
        return is_array($this->Beg_Doc) ? count($this->Beg_Doc) : 0;
    }

    public function getIsWorkflowStatusAttribute()
    {
        // All statuses except Active and Inactive are workflow statuses
        return !in_array($this->Status, ['A', 'I']);
    }

    public function getIsActiveAttribute()
    {
        return $this->Status == 'A';
    }

    public function getCanEditAttribute()
    {
        // Define which statuses allow editing
        $editableStatuses = ['D', 'R', 'RR', 'J'];
        return in_array($this->Status, $editableStatuses);
    }

    // ============================================
    // SCOPES
    // ============================================

    public function scopeByGrantId($query, $grantId)
    {
        return $query->where('Grant_Id', $grantId);
    }

    public function scopeLatestVersions($query)
    {
        // Get only latest versions
        return $query->where('Is_Latest', true);
    }

    public function scopeVersionHistory($query, $rootId)
    {
        // Get all versions of a grant entry
        return $query->where(function($q) use ($rootId) {
            $q->where('Grant_Id', $rootId)
              ->orWhere('Ref_Id', $rootId)
              ->orWhere('Ref_Id', function($subquery) use ($rootId) {
                  $subquery->select('Grant_Id')
                          ->from('grant_entries')
                          ->where('Ref_Id', $rootId);
              });
        })->orderBy('created_at', 'asc');
    }

    // ============================================
    // BOOT METHOD - SIMPLIFIED (NO LOGGING)
    // ============================================

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            // If Grant_Id is not set, generate new one
            if (!$model->Grant_Id) {
                $lastEntry = GrantEntry::orderBy('Grant_Id', 'desc')->first();
                $model->Grant_Id = $lastEntry ? $lastEntry->Grant_Id + 1 : 1;
            }

            // Set default values for new entry
            if (!isset($model->Ref_Id)) {
                $model->Ref_Id = 0; // First version
            }
            if (!isset($model->Is_Latest)) {
                $model->Is_Latest = true; // By default, new entry is latest
            }

            // Set default status if not provided
            if (!isset($model->Status)) {
                $model->Status = 'D'; // Default to Draft
            }
        });

      
    }
}
