<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class LedgerMaster extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'ledger_master';
    protected $primaryKey = 'Lg_Id';

    protected $fillable = [
        'Lg_Code',
        'Lg_Gr',
        'Gr_Id',
        'Lg_Name',
        'Lg_Address',
        'Lg_City',
        'Lg_Pin',
        'Lg_Phone',
        'Lg_Email',
        'Lg_Sub',
        'Lg_Buk',
        'Description',
        'Lg_Cr_Lim',
        'Lg_Drw_Pwr',
        'Lg_Status'
    ];

    protected $casts = [
        'Lg_Cr_Lim' => 'decimal:2',
        'Lg_Drw_Pwr' => 'decimal:2',
        'Lg_Status' => 'boolean'
    ];

    // Validation rules for create
    public static function createRules()
    {
        return [
            'Lg_Code' => 'nullable|string|max:50|unique:ledger_master,Lg_Code',
            'Lg_Gr' => 'nullable|string|max:100',
            'Gr_Id' => 'nullable|integer|exists:ledg_grps,Gr_Id',
            'Lg_Name' => 'required|string|max:200',
            'Lg_Address' => 'nullable|string',
            'Lg_City' => 'nullable|string|max:100',
            'Lg_Pin' => 'nullable|string|max:20',
            'Lg_Phone' => 'nullable|string|max:20',
            'Lg_Email' => 'nullable|email|max:100',
            'Lg_Sub' => 'nullable|string|max:100',
            'Lg_Buk' => 'nullable|string|max:100',
            'Description' => 'nullable|string|max:255',
            'Lg_Cr_Lim' => 'nullable|numeric|min:0|max:9999999999999.99',
            'Lg_Drw_Pwr' => 'nullable|numeric|min:0|max:9999999999999.99',
            'Lg_Status' => 'nullable|boolean'
        ];
    }

   // Validation rules for update
public static function updateRules($id)
{
    return [
        'Lg_Code' => 'nullable|string|max:50|unique:ledger_master,Lg_Code,' . $id . ',Lg_Id',
        'Lg_Gr' => 'nullable|string|max:100',
        'Gr_Id' => 'nullable|integer|exists:ledg_grps,Gr_Id',
        'Lg_Name' => 'sometimes|required|string|max:200',
        'Lg_Address' => 'nullable|string',
        'Lg_City' => 'nullable|string|max:100',
        'Lg_Pin' => 'nullable|string|max:20',
        'Lg_Phone' => 'nullable|string|max:20',
        'Lg_Email' => 'nullable|email|max:100',
        'Lg_Sub' => 'nullable|string|max:100',
        'Lg_Buk' => 'nullable|string|max:100',
        'Description' => 'nullable|string|max:255',
        'Lg_Cr_Lim' => 'nullable|numeric|min:0|max:9999999999999.99',
        'Lg_Drw_Pwr' => 'nullable|numeric|min:0|max:9999999999999.99',
        'Lg_Status' => 'nullable|boolean'
    ];
}
    // Custom validation messages
    public static function validationMessages()
    {
        return [
            'Lg_Code.required' => 'Ledger code is required',
            'Lg_Code.unique' => 'This ledger code already exists',
            'Lg_Code.max' => 'Ledger code cannot exceed 50 characters',
            'Gr_Id.exists' => 'Selected group does not exist',
            'Lg_Name.required' => 'Ledger name is required',
            'Lg_Name.max' => 'Ledger name cannot exceed 200 characters',
            'Lg_Email.email' => 'Please enter a valid email address',
            'Description.max' => 'Description cannot exceed 255 characters',
            'Lg_Cr_Lim.numeric' => 'Credit limit must be a number',
            'Lg_Drw_Pwr.numeric' => 'Drawing power must be a number'
        ];
    }

    // Relationships
    public function subLedgers()
    {
        return $this->hasMany(SubLedgerMaster::class, 'Lg_ID', 'Lg_Id');
    }

    public function ledgerGroup()
    {
        return $this->belongsTo(LedgGrp::class, 'Gr_Id', 'Gr_Id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('Lg_Status', true);
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('Lg_Code', 'like', "%{$search}%")
              ->orWhere('Lg_Name', 'like', "%{$search}%")
              ->orWhere('Lg_Gr', 'like', "%{$search}%")
              ->orWhere('Lg_City', 'like', "%{$search}%")
              ->orWhere('Lg_Phone', 'like', "%{$search}%")
              ->orWhere('Lg_Email', 'like', "%{$search}%")
              ->orWhere('Description', 'like', "%{$search}%");
        });
    }

    public function scopeWithCreditLimit($query)
    {
        return $query->where('Lg_Cr_Lim', '>', 0);
    }

    public function scopeWithDrawingPower($query)
    {
        return $query->where('Lg_Drw_Pwr', '>', 0);
    }

    // Attribute accessors
    public function getFullAddressAttribute()
    {
        $addressParts = [];
        if ($this->Lg_Address) $addressParts[] = $this->Lg_Address;
        if ($this->Lg_City) $addressParts[] = $this->Lg_City;
        if ($this->Lg_Pin) $addressParts[] = $this->Lg_Pin;

        return implode(', ', $addressParts);
    }

    public function getFormattedCreditLimitAttribute()
    {
        return number_format($this->Lg_Cr_Lim, 2);
    }

    public function getFormattedDrawingPowerAttribute()
    {
        return number_format($this->Lg_Drw_Pwr, 2);
    }

    public function getStatusTextAttribute()
    {
        return $this->Lg_Status ? 'Active' : 'Inactive';
    }

    // Business logic methods
    public function canBeDeleted()
    {
        // Check if ledger has sub-ledgers
        return $this->subLedgers()->count() === 0;
    }

    public function deactivate()
    {
        $this->update(['Lg_Status' => false]);
        return $this;
    }

    public function activate()
    {
        $this->update(['Lg_Status' => true]);
        return $this;
    }

    public function getAvailableCredit()
    {
        return max(0, $this->Lg_Cr_Lim - $this->Lg_Drw_Pwr);
    }
}
